
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;

/**
 * A model for the CANopen device to read carousel sensors.
 * This sensors are read by PDO. (in response to a command "sync" sent to the CANbus)
 * 
 * @author virieux
 */
public class CanOpenTTC580  extends CanOpenDevice {
    
    /**
     * pdoVal is coded on 64bits and represents from right to left :
     * - 3 bits for socket indentification
     * - 3 bits unused
     * - 2 bits unused
     * - 4 bits for State of filter at standby for clampXminus
     * - 4 bits for State of filter at standby for clampXplus
     * - 12 bits for ClampXminus lockSensor value
     * - 12 bits for ClampXminus filterPresenceSensor value
     * - 12 bits for ClampXplus lockSensor value
     * - 12 bits for ClampXplus filterPresenceSensor value
     * 
     */
    long pdoVal = 0L;

    public CanOpenTTC580(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }


    /**
     * 
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return PDO1 value")
    public long getPdoVal() {
        return pdoVal;
    }

    /**
     * 
     * @param pdoVal 
     */
    public void setPdoVal(long pdoVal) {
        this.pdoVal = pdoVal;
    }
    
    
    /**
     * send a "sync" command to CANbus, process response and update pdoVal.
     * a reply to a sync command should be :
     * PDO1=xxx, PDO2=xxxx, PDO3=xxxxx etc...
     */
    @Command(type=Command.CommandType.QUERY, description="send a sync command to CANbus, "
            + "process response and update pdoVal.")
    public void updatePDO() {
        pdoVal = extractFirstPDO(tcpProxy.sendCanOpen("sync"));
    }
    
    /**
     * check replyToSync syntaxe and extract the value of PDO1
     * @param replyToSync
     * @return PDO1 value
     */
    protected long extractFirstPDO(String replyToSync) {
        String[] words = replyToSync.split(",");
        if (words[1].startsWith("PDO1") && (words[0].split("=")).length > 0) {
            String pdoValStr = (words[1].split("="))[1];
           return Long.valueOf(pdoValStr,16);
        } else {
            throw new IllegalArgumentException(name + " " + replyToSync + ": bad response to sync command. ");
        }
    }
    
    /**
     * return socketId from PDO value
     * socket identification are the first 3 bits (reading from left to right).
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return socketId from PDO value")
    public short getSocketId() {
        /* 61 = 64 - nbBits (nbBits =3)*/
        return (short) ((pdoVal >>> 61));
    }
    
    /**
     * return lock sensor for clampXminus
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return lock sensor for clampXminus")
    public int getLockXm() {
        return (int) (pdoVal >>> 36) & 0xFFF;
    }
    
    /**
     * return filter presence sensor for clampXminus
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return filter presence sensor for clampXminus")
    public int getFilterPresenceXm() {
        return (int) (pdoVal >>> 24) & 0xFFF;
    }
    
    /**
     * return lock sensor for clampXplus
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return lock sensor for clampXplus")
    public int getLockXp() {
        return (int) (pdoVal >>> 12) & 0xFFF;
    }    
    
    /**
     * return filter presence sensor value for clampXplus.
     * this value is coded on the last 12 bits (reading from left to right).
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return filter presence sensor for clampXplus")
    public int getFilterPresenceXp() {
        return (int) pdoVal & 0xFFF;
    }
}
