package org.lsst.ccs.subsystems.fcs.drivers;

import java.math.BigInteger;
import java.util.ArrayList;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import static org.lsst.ccs.bus.states.AlertState.WARNING;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HardwareController;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.CWrapperNotConnected;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.BAD_SERIAL_NB;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.EMCY;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_MISSING;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.READ_PDO_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_TOO_SHORT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.UNSYNC_BOOT;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;

/**
 * This Module starts a tcpip server, waits for the connection of a client whose
 * name is the value of the field clientName. This client is supposed to be a
 * CANopen C-wrapper. When the client starts it scans the CANopen network and
 * send to the server some information on the CANopen bootedNodes living on the
 * CAN bus. The main job of the class CanOpenProxy is to check that the CANopen
 * devices booted on the CAN bus are the CANopen devices which are listed in the
 * configuration system. 
 *
 * @author virieux
 */
public class CanOpenProxy extends FcsTcpProxy implements HardwareController,
        ClearAlertHandler, BridgeToHardware {

    @LookupField(strategy = Strategy.TOP)
    private Subsystem s;

    /*This is the key word that has to be sent by the client when it connects to the tcp proxy. */
    @ConfigurationParameter(isFinal = true)
    private String clientName;

    /**
     * A timeout for the booting process : during initialization, we wait for
     * the boot messages coming from CANbus field. When this amount of time is
     * over, we don't wait any more. This has to be tuned during the test bench.
     * UNIT = milliseconds
     *
     */
    @ConfigurationParameter(range = "1000..10000",
            description = "A timeout for the hardware booting process")
    private long hardwareBootTimeout = 1000;

    /**
     * This map of PieceOfHardware which key is CANopen nodeID of the hardware
     * represents the list of pieces of hardware (CANopen devices) this
     * CanOpenProxy manages.
     */
    protected final Map<Integer, PieceOfHardware> hardwareMapByNodeID = new HashMap<>();

    @LookupField(strategy = Strategy.CHILDREN)
    protected final Map<String, PieceOfHardware> hardwareMapByName = new HashMap<>();

    private boolean hardwareIDError;
    protected boolean hardwareBootProcessEnded = false;
    protected int bootedDeviceNB;

    private PDOStorage pdoStorage;

    private ScheduledFuture<?> checkDevicesHandle;
    private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(1);
    //Used because we have to wait until all the pieces of hardware are booted.
    protected final Condition bootingCompleted = lock.newCondition();

    /**
     * Creates a CanOpenProxy with a tcpip port number to start the tcp server
     * on, a client name, and a hardware booting process timeout.
     *
     * @param portNumber
     * @param fieldBusTimeout
     * @param clientName
     * @param hardwareBootTimeout
     */
    public CanOpenProxy(
            int portNumber,
            int fieldBusTimeout,
            String clientName,
            long hardwareBootTimeout) {
        super(portNumber, fieldBusTimeout);
        this.clientName = clientName;
        this.hardwareBootTimeout = hardwareBootTimeout;
    }

    /**
     * @return the clientName
     */
    public String getMyClientName() {
        return clientName;
    }

    public PDOStorage getPdoStorage() {
        return pdoStorage;
    }

    /**
     * ***********************************************************************************************
     * ******************** END OF SETTERS AND GETTERS
     * **********************************************
     * ***********************************************************************************************
     * /**
     *
     * /**
     * This methods stops the CWrapper client, and the TCP server.
     */
    @Override
    public void disconnectHardware() {
        //TODO we have to test if all the hardware is stopped before doing stopServer
        this.stopServer();
        initialize();
    }

    @Override
    public void connectHardware() {
        this.startServer();
        this.startThreadReader();
    }

    @Override
    public void init() {
        super.init();
        initialize();

        for (PieceOfHardware poh : hardwareMapByName.values()) {
            hardwareMapByNodeID.put(poh.getNodeID(), poh);
        }

        FCSLOG.info(name + ": init MODULE CanOpenProxy.");
        FCSLOG.info(name + ":NUMBER OF CAN OPEN DEVICES EXPECTED =" + hardwareMapByName.size());
        FCSLOG.info(this.toString());

        //we have to read the errors table for the maxon motor and can open devices.
        CanOpenErrorsTable.loadCanOpenErrorTables();
    }

    /**
     * This method initializes the fields of the tcpProxy. It's used when we
     * start the Module and when we disconnect hardware.
     */
    public void initialize() {
        this.bootedDeviceNB = 0;
        hardwareBootProcessEnded = false;
        this.hardwareIDError = false;
        hardwareMapByName.values().stream().forEach((poh) -> {
            poh.setBooted(false);
        });
        this.pdoStorage = new PDOStorage();
    }

    /**
     * Check if all pieces of hardware in the configuration are booted with the corect serial number.
     * Raise alerts if not.
     * Waits until all hardware is booted or timeout is reached. 
     * 
     */
    @Command(type=Command.CommandType.ACTION, description="Check if all pieces of hardware "
            + "are booted with the corect serial number.")
    @Override
    public void bootProcess() {
        FCSLOG.info(name + ": BEGIN OF BOOT PROCESS");
        initialize();
        
        s.updateAgentState(FilterState.CAN_DEVICES_BOOTING,
                FilterReadinessState.NOT_READY);
        lock.lock();
        try {
            readDevicesInfo(System.currentTimeMillis(), this.hardwareBootTimeout);
            waitForEndOfBooting();
            
        } catch (Exception ex) {
            this.raiseAlarm(HARDWARE_ERROR, " ERROR during booting process", ex);
            
        } finally {
            FCSLOG.debug(getName() + ": finally in bootProcess");
            hardwareBootProcessEnded = true;
            bootingCompleted.signalAll();
            lock.unlock();
        }
        if (!this.hardwareIDError) {
            FCSLOG.info(name + " ALL HARDWARE is booted");
            s.updateAgentState(FcsEnumerations.FilterState.HOMING_TO_BE_DONE,
                    FcsEnumerations.FilterReadinessState.READY); 
        } else {
            FCSLOG.info(name + " SOME HARDWARE is MISSING");
        }
        FCSLOG.info(name + " BOOTED HARDWARE=" + listBootedNodes());        
        FCSLOG.info(name + ": END OF BOOT PROCESS");
    }

    /**
     * check that all pieces of hardware is booted.
     */
    @Override
    public void postStart() {

        FCSLOG.debug(name + ": BEGIN postStart");
        bootProcess();
        FCSLOG.debug(name + ": END postStart");
        //for the GUI
        this.publishData();
    }

    /**
     * This method is called in the checkHardware method during the
     * initialization process. It check devices at fixed rate with a scheduler
     * until all devices are booted or booting timeout is exceeded.
     *
     * @param beginTime
     * @param timeout
     */
    private void readDevicesInfo(
            final long beginTime, final long timeout) {

        final Runnable checkDevices = () -> {
            // Retrieve some information as serial number, vendor, for the CANopen devices
            // with command (info,node_id)
            retrieveHardwareInfo();
            this.publishData();
            long duration = System.currentTimeMillis() - beginTime;
            FCSLOG.fine("duration="+duration);
            FCSLOG.fine("bootedDeviceNB="+bootedDeviceNB);                
            if (duration > timeout || this.bootedDeviceNB == hardwareMapByName.size()) {
                hardwareBootProcessEnded = true;
                cancelReadDevicesInfo();
            }

        };
        checkDevicesHandle = scheduler.scheduleAtFixedRate(checkDevices,
                500, 500, TimeUnit.MILLISECONDS);
    }

    /**
     * This stops the reading of the sensors.
     *
     */
    private void cancelReadDevicesInfo() {
        lock.lock();
        try {
            FCSLOG.debug(name + " => stop waiting for devices info.");
            bootingCompleted.signalAll();
        } finally {
            lock.unlock();
        }
        checkDevicesHandle.cancel(true);
        FCSLOG.debug(name + " => readDevicesInfo canceled");
    }

    /**
     * This method waits until the booting process is completed. This methods is
     * called by checkHardware and has already acquired the lock.
     */
    private void waitForEndOfBooting() {
        while (!hardwareBootProcessEnded) {
            try {
                FCSLOG.info(name + " waiting until all pieces of hardware are booted.");
                bootingCompleted.await();
            } catch (InterruptedException ex) {
                FCSLOG.info(name + ": InterruptedException received=" + ex);
                break;
            }
        }
        FCSLOG.info(name + " STOP WAITING FOR END OF BOOTING PROCESS");
    }

    @Override
    public void checkStarted() {
        FCSLOG.info(name + " BEGIN checkStarted");
        this.bootProcess();
        FCSLOG.info(getName() + " END checkStarted");
    }

    /**
     * This shutdowns the scheduler. This is executed during CLOSING phase when
     * the CanOpenProxy is stopped. cf checkStopped
     */
    @Override
    public void shutdownNow() {
        super.shutdownNow();
        scheduler.shutdown();
    }

    /**
     * For a piece of hardware that this tcpProxy manages, this methods returns
     * the name of the device when the CANopen node id is given as argument.
     *
     * @param aNodeID
     * @return
     */
    public String getNodeName(int aNodeID) {
        if (this.hardwareMapByNodeID.get(aNodeID) == null) {
            return "UnknownDevice" + aNodeID;
        } else {
            return this.hardwareMapByNodeID.get(aNodeID).getName();
        }
    }



    /**
     * List the can open bootedNodes which are in the bootedNodes table.
     *
     * @return the list of can open bootedNodes and the information stored in
     * this.bootedNodes.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "Print the list of CANopen nodes which are booted on "
            + "the CAN bus. (values are in HEXA)")
    public String listBootedNodes() {
        if (this.bootedDeviceNB == 0) {
            return name + ": no booted CANopen devices.";
        } else {
            StringBuilder sb = new StringBuilder();
            for (PieceOfHardware poh: this.hardwareMapByName.values()) {
                if (poh.isBooted()) {
                    sb.append("\n==>");
                    sb.append(poh);
                }
            }
            return sb.toString();
        }
    }

    /**
     * For end users, in engineering mode, this method can be used to send Can
     * Open commands to the Wrapper. Test if the command is valid before sending
     * it to the CAN bus.
     *
     * @param command A Can Open command that the Wrapper should understand.
     * @return the response from the Wrapper
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Send a CanOpen command to the Can Bus.")
    public String sendCanOpenCommand(String command) {
        if (!this.tcpServerStarted) {
            throw new FcsHardwareException(name + ": is not started, can't send CanOpen commands.");
        }
        FcsUtils.checkCommand(command);
        return sendCanOpen(command);
    }

    /**
     * Check that a nodeID entered by an end user at the console is a valid one:
     * - nodeID should be in the hardwareMapByNodeID.
     *
     * @param nodeID
     */
    public void checkNodeID(int nodeID) {
        if (!hardwareMapByNodeID.containsKey(nodeID)) {
            throw new IllegalArgumentException(nodeID
                    + " is not in the hardware list for tcpProxy " + name);
        }
    }

    /**
     * Send a command on the CANbus. This method is for internal usage. 
     * For end user usage see sendCanOpenCommand.
     *
     * @param command
     * @return
     * @throws FcsHardwareException
     */
    protected String sendCanOpen(String command) {
        try {
            FCSLOG.debug(name + ":Sending CANopen command : " + command);

            return (String) call(getMyClientName(), command);

        } catch (CWrapperNotConnected ex) {
            throw new FcsHardwareException("CWrapper not connected- check ethernet connections.", ex);
        }
    }

    /**
     * Command to be used by the end user at the console. (for internal usage,
     * see writeSDO(String,String,String,String,String); Write a SDO message and
     * send it to the can open stack. The parameters are given in the following
     * format:
     *
     * @param nodeID FORMAT=hexa
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=decimal between 0 and 4.
     * @param value FORMAT=decimal
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Send a CanOpen writeSDO command to the Can Bus. "
            + " \nValues of the argument are to be given in hexadecimal format for: index and subindex"
            + "\n and in decimal format for size and value."
            + "\n size represents the number of bytes on which the value is encoded. See device documentation.")
    public String writeSDO(int nodeID, int index, int subindex, int size, int value) {
        checkNodeID(nodeID);
        if (size < 0 || size > 4) {
            throw new IllegalArgumentException("size must be > 0 and < 4");
        }
        String request = FcsUtils.buildWsdoCommand(nodeID, index, subindex, size, value);
        String sdoResponseLine = sendCanOpen(request);
        /*response of a writeSDO command is : wsdo,nodeID,errorCode*/
        String[] words = sdoResponseLine.split(",");

        int errorCode = Integer.parseInt(words[2], 16);
        String msg;
        if (0 == errorCode) {
            return "OK";
        } else {
            msg = name + ":wrong writeSDO command : " + sdoResponseLine;
            processSDORequestError(request, sdoResponseLine, nodeID, errorCode, ALARM);
            throw new SDORequestException(msg);
        }
    }

    /**
     * Read a SDO with the given index and subindex RETURNS the value read in
     * hexa (String) if no error occured or throws an Exception if an error
     * occured.
     *
     * @param nodeID
     * @param index
     * @param subindex
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    public int readSDO(int nodeID, int index, int subindex) {
        String request = FcsUtils.buildRsdoCommand(nodeID, index, subindex);

        String sdoLine = sendCanOpen(FcsUtils.buildRsdoCommand(nodeID, index, subindex));
        return Integer.parseInt(processResponseToReadSDO(sdoLine, nodeID, request), 16);
    }

    public int readSDOLong(int nodeID, int index, int subindex) {
        String request = FcsUtils.buildRsdoCommand(nodeID, index, subindex);
        String sdoLine = sendCanOpen(FcsUtils.buildRsdoCommand(nodeID, index, subindex));
        return new BigInteger(processResponseToReadSDO(sdoLine, nodeID, request), 16).intValue();
    }
    

    /**
     * Check a response to a readSDO command. Return the data requested if the
     * response is a valid response otherwise throw an Exception of type
     * SDORequestException or ShortResponseToSDORequestException.
     *
     * @param response
     * @param nodeID
     * @param request
     * @return
     * @throws ShortResponseToSDORequestException
     * @throws SDORequestException
     */
    public String processResponseToReadSDO(String response, int nodeID, String request) {
        String[] words = response.split(",");
        int responseLength = words.length;
        String msg;
        String deviceName;
        /* response of a readSDO command should be : rsdo,nodeID,errorCode,data */
        int errorCode = Integer.parseInt(words[2], 16);

        if (errorCode == 0 && (responseLength > 3)) {
            return words[3];

        } else if (errorCode == 0 && responseLength == 3) {
            msg = name + ":readSDO request received a too short response=" + response + " request was" + request;
            deviceName = getNodeName(nodeID);
            this.raiseWarning(deviceName + ":" + SDO_TOO_SHORT, SDO_TOO_SHORT.getLongDescription(), msg);
            throw new ShortResponseToSDORequestException(msg);

        } else if (-1 == errorCode) {
            msg = name + ":wrong readSDO command : " + request + " response:" + response;
            deviceName = getNodeName(nodeID);
            this.raiseAlarm(SDO_ERROR, msg, deviceName);
            throw new SDORequestException(msg);

        } else {
            msg = name + ":wrong readSDO command : " + request + " response:" + response;
            processSDORequestError(request, response, nodeID, errorCode, ALARM);
            throw new SDORequestException(msg);
        }
    }

    /**
     * Command to be used by the end user at the console in engineering mode.
     *
     * Send a CANOpen readSDO command to the Can Bus and returns the response in
     * hexadecimal format. Check CANOpen nodeID.
     *
     * @param nodeID the node id
     * @param index the index
     * @param subindex the sub index
     * @return an hex string representation.
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "Send a CANOpen readSDO command to the Can Bus.")
    public String readSDOCommand(int nodeID, int index, int subindex) {
        checkNodeID(nodeID);
        return Integer.toHexString(readSDO(nodeID, index, subindex));
    }

    /**
     * When we receive a response with an error to a SDO request, this methods
     * retreives the errorName in the CANOpen tables, create an error message
     * and raise an Alert.
     *
     * @param request
     * @param response
     * @param errorCode
     * @param nodeID
     * @param alertState
     */
    public void processSDORequestError(String request, String response,
            int nodeID, int errorCode, AlertState alertState) {
        String errorName = CanOpenErrorsTable.getCommErrorNameByCode(errorCode);
        String deviceName = getNodeName(nodeID);
        String msg = name + ": SDO request was going wrong for device: " + deviceName;
        msg = msg + ",request=" + request + ",received response=" + response;
        msg = msg + ",errorCode=" + errorCode + ",errorName=" + errorName;
        if (alertState == WARNING) {
            this.raiseWarning(SDO_ERROR, msg, deviceName);
        } else {
            this.raiseAlarm(SDO_ERROR, msg, deviceName);
        }
    }

    /**
     * This method sends a sync command to the can open stack and returns the
     * reply. example of sync reply we can receive :
     * sync,23_1=25a7,23_2=113,23_3=10,23_4=109,23_5=108,23_6=104,23_7=101,23_8=112
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Override
    public PDOStorage readPDOs() {
        try {
            String replyToSync = sendCanOpen("sync");
            FCSLOG.finest(name + ": replyToSync=" + replyToSync);
            this.pdoStorage.updatePDOs(replyToSync);
            return this.pdoStorage;

        } catch (FcsHardwareException ex) {
            String msg = name + ": error in response to a sync command.";
            this.raiseAlarm(READ_PDO_ERROR, msg, name, ex);
            throw new FcsHardwareException(msg, ex);
        }
    }

    /**
     * Identification of the hardware : we want to retrieve the information
     * stored in the hardware of the booted bootedNodes and update the array of
     * bootedNodes with this information. So for each booted node, we send a
     * command : - info,node_ID to the CWrapper and then we update the
     * bootedNodes list with the information returned in the command result. The
     * serial number is a main information to retrieve.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Identification of the hardware : we want to retrieve the information stored "
                    + "in the hardware of the CANopen devices")
    public void retrieveHardwareInfo() {
        FCSLOG.info(getName() + ":Identification of the hardware");
        hardwareMapByName.values().stream().filter((poh) -> (!poh.isBooted())).forEach((poh) -> {
            updateDeviceInfo(poh);
        });
        publishData();
    }

    /**
     * This method sends a message info,nodeID to the CANbus and update the
     * CANopen device information from data received from the device.
     *
     * @param poh
     * @throws HardwareException
     */
    private void updateDeviceInfo(PieceOfHardware poh) {
        int nodeID = poh.getNodeID();
        String commandInfo = "info," + Integer.toHexString(nodeID);
        try {
            FCSLOG.debug(getName() + ":Sending to can open command :" + commandInfo);
            String result = sendCanOpen(commandInfo);
            processInfoMessage(poh, result);
            
        } catch (CanOpenCallTimeoutException ex) {
            //TODO raiseAlarm only at the end of process ?
            if (!"tmpSensorsDevice".equals(poh.getName())) {
                this.hardwareIDError = true;
                this.raiseAlarm(HARDWARE_MISSING, " no response to command : " + commandInfo, 
                        poh.getName(), ex);
            } else {
                this.raiseWarning(HARDWARE_MISSING, " no response to command : " + commandInfo, 
                        poh.getName(), ex);
            }
            
        } catch (FcsHardwareException ex) {
            this.hardwareIDError = true;
            this.raiseAlarm(HARDWARE_ERROR, " Error in boot process for device." , poh.getName(), ex);
        }
    }

    /**
     * This method returns true if : all the hardware items are booted and
     * identified and the hardware have the node ID expected within the
     * configuration and the hardware is initialized.
     *
     * @return a boolean
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all CANopen devices are booted and identified."
            + "Identified means that the serial numbers match the CANopen nodeID which reside in"
            + "configuration.")
    @Override
    public boolean allDevicesBooted() {
        boolean bootedDevicesNumberOK = this.bootedDeviceNB == hardwareMapByNodeID.size();
        FCSLOG.info(name + ":hardwareBootProcessEnded=" + hardwareBootProcessEnded);
        FCSLOG.info(name + ":canOpenNodeNumbersOK=" + bootedDevicesNumberOK);
        FCSLOG.info(name + ":hardwareIDError=" + this.hardwareIDError);
        return this.hardwareBootProcessEnded && bootedDevicesNumberOK && !this.hardwareIDError;
    }

    @Override
    void processBootMessage(int nodeID) {
        if (hardwareMapByNodeID.containsKey(nodeID)) {
            PieceOfHardware poh = hardwareMapByNodeID.get(nodeID);
            if (poh.isBooted()) {
                /*the node is already booted */
                checkResetNode(nodeID, poh.getName());
            }
        } else {
            this.raiseAlarm(UNSYNC_BOOT, nodeID + ":UNKNOWN device. This device is not in the harware list.");
        }
    }

    private void checkResetNode(int nodeID, String deviceName) {
        if (hardwareBootProcessEnded) {
            /*the boot process is ended: something weird has just happened*/
            String cause = getName() + " received an unsynchronous boot message for node ID = " + nodeID
                    + " Has the device been reset or powered on ?";
            this.raiseWarning(UNSYNC_BOOT, deviceName, cause);
        }
    }

    /**
     * Process an emergency message received from a CANopen device on the status
     * bus : - ALARM or WARNING Alert is raised, - a message is logged, -
     * Observers are notified (CANopen devices are CanOpenProxy Observers).
     *
     * @param message
     */
    @Override
    public void processEmcyMessage(String message) {
        String[] words = message.split(",");
        /*NodeID without the leading zeros*/
        int nodeID = Integer.parseInt(words[1], 16);
        int deviceErrorCode = Integer.parseInt(words[2], 16);
        String deviceErrorName = CanOpenErrorsTable.getDeviceErrorNameByCode(deviceErrorCode);

        int errReg = Integer.parseInt(words[3], 16);
        String errorRegisterName = CanOpenErrorsTable.getErrorRegisterNameByCode(errReg);

        String deviceName = this.getNodeName(nodeID);
        EmergencyMessage emcyMsg = new EmergencyMessage(nodeID, deviceName, deviceErrorCode,
                deviceErrorName, errReg, errorRegisterName);
        // Pass emergency message to affected device
        PieceOfHardware poh = hardwareMapByNodeID.get(nodeID);
        if (poh != null) {
            // The use of the scheduler is required in order to leave the reader thread as soon as possible.
            s.getScheduler().schedule(() -> {
                poh.onEmergencyMessage(emcyMsg);
            }, 0, TimeUnit.SECONDS);
        }
        FCSLOG.warning(name + " received EMERGENCY message=" + message + " for nodeID=" + Integer.toHexString(nodeID));
        if (errReg == 0 || 0x20 == nodeID || 0x22 == nodeID) {
            //because tempsSensorDevice is always in error when it starts.
            //should be resolved in another way => setting canbus transmission speed for example.
            //proximitySensorsDevice sends also an emergency message deviceErrorCode: 0x3110
            this.raiseWarning(EMCY, emcyMsg.toString(), deviceName);
        } else {
            this.raiseAlarm(EMCY, emcyMsg.toString(), deviceName);
        }
    }

    /**
     * Process the response to a command info,node_ID.
     *
     * @param message
     */
    void processInfoMessage(PieceOfHardware poh, String message) {
        FCSLOG.debug(getName() + ":Received on socket command = " + message);
        String[] words = message.split(",");
        int nodeID = Integer.parseInt(words[1], 16);

        FCSLOG.debug(getName() + ":checking serial number for nodeID " + Integer.toHexString(nodeID));
        String type = words[2];
        String vendor = words[3];
        String productCode = words[4];
        String revision = words[5];
        String serialNB = words[6];
        FCSLOG.info(name + "/" + poh.getName() + " informations read on device:" + " type=" + type + " vendor=" + vendor
                + " product code=" + productCode + " revision=" + revision + " serialNB" + serialNB);
        if (poh.getSerialNB().equals(serialNB)) {
            poh.setBooted(true);
            this.bootedDeviceNB++;
        } else {
            this.hardwareIDError = true;
            this.raiseAlarm(BAD_SERIAL_NB, " serial number read on device=" + serialNB
                + " ==>serial number expected=" + poh.getSerialNB() + " ==>device configuration=" + poh, poh.getName());
        }
    }








    /**
     * Return a String with the list of hardware expected in this subsystem. 
     * For debug purpose.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware expected in this subsystem.")
    @Override
    public String printHardwareList() {
        return hardwareMapByName.toString();
    }

    /**
     * Return a printed list of hardware with the initialization state. For
     * debug purpose.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware with the initialization state.")
    @Override
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(name + " CANopen devices : {");
        for (PieceOfHardware poh : this.hardwareMapByNodeID.values()) {
            sb.append(poh.printState());
            sb.append(";\n");
        }
        sb.append('}');
        return sb.toString();
    }

    /**
     * For the GUI : Return an Array containing hardware names handled by this
     * component.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return an Array containing hardware names handled by this component.")
    @Override
    public List<String> listHardwareNames() {
        List<String> res = new ArrayList<>();
        for (PieceOfHardware pieceOfHardware : this.hardwareMapByName.values()) {
            res.add(pieceOfHardware.getName());
        }
        return res;
    }

    /**
     * Overridden method from ClearAlertHandler interface to define conditions
     * when Alerts can be cleared.
     *
     * @param alert
     * @return
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert) {
        switch (alert.getAlertId()) {
            case "FCS001":
                if (this.clientContext.socket.isConnected()
                        && !this.clientContext.socket.isClosed()) {
                    return ClearAlertCode.CLEAR_ALERT;
                } else {
                    return ClearAlertCode.DONT_CLEAR_ALERT;
                }

            case "FCS002":
                return ClearAlertCode.CLEAR_ALERT;

            default:
                return ClearAlertCode.UNKWNOWN_ALERT;
        }
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Publish booting information for all CANopen devices.")
    @Override
    public void publishData() {
        FCSLOG.info(getName() + " is publishing:" + this.bootedDeviceNB);
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(name, this.bootedDeviceNB));
        for (PieceOfHardware device : hardwareMapByName.values()) {
            device.publishData();
        }
    }



    /**
     * For tests. Return a printed format of PDOStorage.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return a printed format of PDOStorage.")
    public String printPDOStorage() {
        return this.pdoStorage.toString();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append("/clientName=");
        sb.append(this.clientName);
        sb.append("/hardwareBootTimeout=");
        sb.append(this.hardwareBootTimeout);
        return sb.toString();
    }

    // -- BridgeToHardware interface methods -----------------------------------
    @Override
    public boolean isCWrapperConnected() {
        return isReady(getMyClientName());
    }

    @Override
    public boolean isRealHardware() {
        return true;
    }

}
