
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_TIMEOUT;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * A class to represent the CANopen pluto gateway which is a device where we can read sensors.
 * Its raises an alert :
 * - FCS004 in readNewHexaValues
 * 
 * @author virieux
 */
public class CanOpenPlutoGateway extends PlutoGatewayModule {
    
   /**
    * A Logger for CommandDispenser.
    */
    public static final Logger LOCAL_LOG = Logger.getLogger("org.lsst.ccs.subsystems.fcs.CanOpenPlutoGateway");
    


    /**
     * build a new CanOpenPlutoGateway
     * @param nodeID
     * @param serialNB
     */
    public CanOpenPlutoGateway(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }



    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Initialize the Gateway.")
    public void initializeAndCheckHardware()  {
        checkBooted();
        try {
            FCSLOG.info(name + " BEGIN initializeHardware");
            this.configurePlutoGateway();
            this.initialized = true;
            FCSLOG.info(name + " is INITIALIZED.");
            //publishData for the LoaderHardwareGUI
            this.publishData();

        } catch (SDORequestException | FailedCommandException ex) {
            throw new FcsHardwareException(name,ex);
        }
    }

    /**
     * Configures the gateway.
     * Configuration depends on specific needs for each subsystems.
     * This methods switches to the right configuration in reading the name of the gateway.
     * 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Configures this gateway for subsystem needs. "
                + "If gateway's name contains loader it executes configureLoaderPlutoGateway"
                + "else it executes configureAutochangerPlutoGateway.")
    public void configurePlutoGateway() {
        if (name.contains("loader")) {
            configureLoaderPlutoGateway();
        } else if (getName().startsWith("acSensors")){
            configureAutochangerPlutoGateway();
        }
    }

    /**
     * Configures this gateway to receive 3 data blocs from Pluto.
     * Used for autochanger pluto gateway.
     * Assuming Pluto has 3 fonctions Gateway_to_User_C First one =>
     * Fonction_ID=01 Second one => Fonction_ID=02 Third one => Fonction_ID=03
     *
     * @throws SDORequestException
     */
    private void configureAutochangerPlutoGateway()  {
        
        FCSLOG.debug(name+ " configureAutochangerGateway");

        //Clear config 
        this.tcpProxy.writeSDO(getNodeID(), 0x2010, 1, 0x1, 0x0);

        //Set Additionnal data TPDO1 => define Area_0 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2011, 1, 0x2, 0x17);

        //Set Additionnal data TPDO1 => define Area_1 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2011, 2, 0x2, 0x4);

        //Set Additionnal data TPDO2 => define Aera_2  (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2012, 0x1, 0x2, 0x117);
        
        //Set Additionnal data TPDO2 => define Aera_3  (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2012, 0x2, 0x02, 0x143);
    }
    
    /**
     * Configures plutoGateway for loader needs.
     */
    private void configureLoaderPlutoGateway() {
        FCSLOG.debug(name+ " configureLoaderPlutoGateway");
        
        //Clear additional data configuration.
        this.tcpProxy.writeSDO(getNodeID(), 0x2010, 1, 1, 0x1);

        //Set Additionnal data TPDO1 => define Area_0 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2011, 1, 2, 0x1);

        //Set Additionnal data TPDO1 => define Area_1 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2011, 2, 2, 0x2);

        //Set Additionnal data TPDO2 => define Aera_2  (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2012, 1, 2, 0x3);
        
        //Activate additional data for Area_0, Area_1 and Area_2 (Enable Data To Pluto Areas 0-3)
        this.tcpProxy.writeSDO(getNodeID(), 0x2002, 1, 1, 0x7);
    }

    /**
     * Write a transmission type in the CPU of the pluto gateway.
     * @param transmissionType
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    public void writeTransmissionType(int transmissionType)  {
        String ack = this.tcpProxy.writeSDO(getNodeID(), 0x2005, 1, 1, transmissionType);
        String goodAck = "OK";
        if (!ack.equals(goodAck)) {
            throw new FailedCommandException(name + " :bad returned ack for wsdo, index 1801");
        }
    }

    /* 
     * @return values in hexa read on the device
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return values in hexa read on the device.")
    @Override
    public int[] readNewHexaValues()  {

        checkBooted();
        try {            
            /*Index 6000*/
            hexaValues[0] = tcpProxy.readSDO(getNodeID(), 0x6000, 1);
//            LOCAL_LOG.finest(String.format("%s : first byte read in hexaValue= %s", name, hexaValues[0]));
            hexaValues[1] = tcpProxy.readSDO(getNodeID(), 0x6000, 2);
//            LOCAL_LOG.finest(String.format("%s : second byte read in hexaValue= %s", name, hexaValues[1]));            
            hexaValues[2] = tcpProxy.readSDO(getNodeID(), 0x6000, 3);
//            LOCAL_LOG.finest(String.format("%s : third byte read in hexaValue= %s", name, hexaValues[2]));
            hexaValues[3] = tcpProxy.readSDO(getNodeID(), 0x6000, 4);
//            LOCAL_LOG.finest(String.format("%s : 4th byte read in hexaValue= %s", name, hexaValues[3]));
            
            /*Index 6001*/
            hexaValues[4] = tcpProxy.readSDO(getNodeID(), 0x6001, 1);
//            LOCAL_LOG.finest(String.format("%s : 5th byte read in hexaValue= %s", name, hexaValues[4]));
            hexaValues[5] = tcpProxy.readSDO(getNodeID(), 0x6001, 2);
//            LOCAL_LOG.finest(String.format("%s : 6th byte read in hexaValue= %s", name, hexaValues[5]));            
            hexaValues[6] = tcpProxy.readSDO(getNodeID(), 0x6001, 3);
//            LOCAL_LOG.finest(String.format("%s : 7th byte read in hexaValue= %s", name, hexaValues[6]));
            hexaValues[7] = tcpProxy.readSDO(getNodeID(), 0x6001, 4);
//            LOCAL_LOG.finest(String.format("%s : 8th byte read in hexaValue= %s", name, hexaValues[7])); 
            
            /*Index 6002*/
            hexaValues[8] = tcpProxy.readSDO(getNodeID(), 0x6002, 1);
//            LOCAL_LOG.finest(String.format("%s : 9th byte read in hexaValue= %s", name, hexaValues[8]));
            hexaValues[9] = tcpProxy.readSDO(getNodeID(), 0x6002, 2);
//            LOCAL_LOG.finest(String.format("%s : 10th byte read in hexaValue= %s", name, hexaValues[9]));            
            hexaValues[10] = tcpProxy.readSDO(getNodeID(), 0x6002, 3);
//            LOCAL_LOG.finest(String.format("%s : 11th byte read in hexaValue= %s", name, hexaValues[10]));
            hexaValues[11] = tcpProxy.readSDO(getNodeID(), 0x6002, 4);
//            LOCAL_LOG.finest(String.format("%s : 12th byte read in hexaValue= %s", name, hexaValues[11]));  
            
            /*Index 6003*/
            hexaValues[12] = tcpProxy.readSDO(getNodeID(), 0x6003, 1);
//            LOCAL_LOG.finest(String.format("%s : 13th byte read in hexaValue= %s", name, hexaValues[12]));
            hexaValues[13] = tcpProxy.readSDO(getNodeID(), 0x6003, 2);
//            LOCAL_LOG.finest(String.format("%s : 14th byte read in hexaValue= %s", name, hexaValues[13]));            
            hexaValues[14] = tcpProxy.readSDO(getNodeID(), 0x6003, 3);
//            LOCAL_LOG.finest(String.format("%s : 15th byte read in hexaValue= %s", name, hexaValues[14]));
            hexaValues[15] = tcpProxy.readSDO(getNodeID(), 0x6003, 4);
//            LOCAL_LOG.finest(String.format("%s : 16th byte read in hexaValue= %s", name, hexaValues[15]));             
            
            
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING SENSOR:" + ex);

        } catch (SDORequestException ex) {
            String message = name + "=> ERROR IN READING SENSOR:";
            this.raiseAlarm("FCS004:"+name, message, ex);
            
        } catch (CanOpenCallTimeoutException ex) {
            String msg = name + ":" + CAN_BUS_TIMEOUT.getLongDescription()
                    + " to command: readNewHexaValues - POWER FAILURE ? ";
            this.raiseWarning(name + ":" + CAN_BUS_TIMEOUT, CAN_BUS_TIMEOUT.getLongDescription(),msg + ex);
        }
        return hexaValues;
    }
    
    /**
     * Read new values for sensors connected to this gateway from a PDO request. 
     * @return an Array of 16 bytes in hexadecimal format
     */
    @Override
    public int[] readHexaValuesFromPDO() {
        int[] values = new int[16];
        PDOStorage pdos = tcpProxy.readPDOs();
        for (int i = 0; i < 16; i++) {
            values[i] = pdos.getVarValue(nodeID, i);
        }
        return values;
    }


}
