package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.HashMap;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerWithBrake;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This class is the driver for the EPOS controller which controls the
 * Autochanger ONLINE clamps. A holding brake can be associated to a Controller.
 * This brake is activated when the controller is powered off. 
 * For Autochanger Online clamp, a brake is associated with the controller of the
 * clamp.  *
 * Configuration of this controller, and the way we release or activate brake is
 * specific to the ONLINE clamps and different from the way it's done for Linear
 * Rails trucks. It's the raison why this class has been created.
 *
 * ATTENTION : activateBrake for the OnlineClamp consists in forcing to ONE bit
 * 15 of parameter DigitalOutputFonctionnalityState. For Linear Rails Truck,
 * activateBrake consists in forcing to ZERO bit 15 of the same parameter.
 *
 *
 * @author virieux
 */
public class CanOpenEPOSOnlineClamp extends CanOpenEPOS implements EPOSControllerWithBrake {

    /**
     * build a new CanOpenEPOS70
     *
     * @param nodeID
     * @param serialNB
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming
     */
    public CanOpenEPOSOnlineClamp(int nodeID, String serialNB,
            HashMap<String, Integer> paramsForCurrent,
            HashMap<String, Integer> paramsForProfilePosition,
            HashMap<String, Integer> paramsForHoming) {
        super(nodeID, serialNB, paramsForCurrent, paramsForProfilePosition, paramsForHoming);
    }

    /**
     * Activate brake to prevent clamp motion.
     *
     *
     * activateBrake for the ONlineClamps consists in forcing to ONE bit 15 of
     * parameter DigitalOutputFonctionnalityState.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Activate brake to prevent clamp motion.")
    @Override
    public void activateBrake() {
        int hexaValue = readParameter(Parameter.DigitalOutputFonctionnalityState);
        writeParameter(Parameter.DigitalOutputFonctionnalityState,
                FcsUtils.force2zero(hexaValue, 15));
        FCSLOG.info(getName() + ": brake activated.");
    }

    /**
     * Release brake to be able to move clamp. A holding brake can be associated
     * to a Controller. Brake is associated with the controllers of the ONLINE
     * clamp.
     *
     * releaseBrake for an ONlineClamp controller consists in forcing to ZERO
     * bit 15 of parameter DigitalOutputFonctionnalityState.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Release brake to be able to move clamp.")
    public void releaseBrake() {
        //If the controller is not enabled and we release brakes it could lead to a 
        //lost of a filter.
        if (!isEnabled()) {
            throw new RejectedCommandException(name + " can't releaseBrake if not enabled.");
        }
        doReleaseBrake();
    }

    @Override
    public void activateBrakeAndDisable() {
        activateBrake();
        disable();
    }

    @Override
    public void enableAndReleaseBrake() {
        enable();
        doReleaseBrake();
    }

    @Override
    public void doReleaseBrake() {
        int hexaValue = readParameter(Parameter.DigitalOutputFonctionnalityState);
        writeParameter(Parameter.DigitalOutputFonctionnalityState,
                FcsUtils.force2one(hexaValue, 15));
        FCSLOG.info(getName() + ": brake released.");
    }

    /**
     * return true if brake if activated.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return true if brake if activated.")
    @Override
    public boolean isBrakeActivated() {
        int val = readParameter(Parameter.DigitalOutputFonctionnalityState);
        int[] digitalOutputInBinary = FcsUtils.toReverseBinary(val);
        return digitalOutputInBinary[15] == 1;
    }
}
