package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.HashMap;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForLinearRail;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This class is the driver of the EPOS controller EPOS24 which controls motion
 * of Autochanger Trucks on the linear rails.
 *
 * @author virieux
 */
public class CanOpenEPOSLinearRailTruck extends CanOpenEPOS implements EPOSControllerForLinearRail {

    @ConfigurationParameter(description = "minimal position of the encoder ribbon.")
    private final int encoderRibbonMinValue;

    private boolean homingDone;

    private boolean configDigitalInputOK = true;
    private boolean configDigitalOutputOK = true;
    private StringBuilder configDigitalInputSb;
    private StringBuilder configDigitalOutputSb;

    /**
     * Build a new CanOpenEPOS24
     *
     * @param nodeID
     * @param serialNB
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming
     * @param encoderRibbonMinValue
     */
    public CanOpenEPOSLinearRailTruck(int nodeID, String serialNB,
            HashMap<String, Integer> paramsForCurrent,
            HashMap<String, Integer> paramsForProfilePosition,
            HashMap<String, Integer> paramsForHoming,
            int encoderRibbonMinValue) {
        super(nodeID, serialNB, paramsForCurrent, paramsForProfilePosition, paramsForHoming);
        this.encoderRibbonMinValue = encoderRibbonMinValue;
    }

    /**
     * return true if homing is done: the controller knows its absolute
     * position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, description = "return true if homing is done: "
            + "the controller knows its absolute position.")
    @Override
    public boolean isHomingDone() {
        return homingDone;
    }

    private void checkInMode(EposMode mode) {
        if (readMode() != mode) {
            throw new FcsHardwareException(getName() + " is not in mode " + mode);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return true if homing has to be done.")
    @Override
    public void homing() {
        checkInitialized();
        homingDone = false;
        int ssiPosition = readSSIPosition();
        int pos = ssiPosition - encoderRibbonMinValue;
        FCSLOG.debug(getName() + " ssiPosition=" + ssiPosition + ", position=" + pos);

        if (Math.abs(pos - readPosition()) < 10) {
            FCSLOG.info(getName() + " no homing to be done.");
            homingDone = true;
        } else {
            doHoming(pos);
            publishData();
        }
    }
    
    private void doHoming(int pos) {
        FCSLOG.debug(getName() + " ==> BEGIN homing");
        EposMode myMode = readMode();
        enable();
        defineAbsolutePosition(pos);
        checkHomingDone(pos);
        homingDone = true;
        changeMode(myMode);
        publishData();
        activateBrakeAndDisable();        
        FCSLOG.debug(getName() + " ==> END homing");
     }
    
    private void checkHomingDone(int pos) {
        long timeout = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        while (!isTargetReached() && duration <= timeout) {
            FCSLOG.info(getName() + " homing target not yet reached. duration=" + duration);
            duration = System.currentTimeMillis() - timeStart;
        }
        if (isTargetReached() && pos == readPosition()) {
            writeControlWord(0xF);

        } else {
            publishData();
            activateBrakeAndDisable();
            throw new FcsHardwareException(getName() + " couldn't do homing :"
                    + " target is not reached.");
        }
    }

    /**
     * Activates brake to prevent trucks motion. A holding brake can be
     * associated to a Controller. This brake is activated when the controller
     * is powered on. A brake is associated with the controller of the
     * Autochanger Linear Rail truck.      
     *
     * activateBrake for the ONlineClamps consists in forcing to ZERO bit 15 of
     * parameter DigitalOutputFonctionnalityState.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Activate brake to prevent truck motion.")
    @Override
    public void activateBrake() {
        int val = readParameterInHexa(Parameter.DigitalOutputFonctionnalityState);
        writeParameter(Parameter.DigitalOutputFonctionnalityState,
                FcsUtils.force2zero(val, 15));
        FCSLOG.info(getName() + ": brake activated.");
    }

    /**
     * Release holding brake to be able to move truck. A holding brake can be
     * associated to a Controller. In CPPM test bench a Brake is associated with
     * the controllers of the trucks. Release the brake
     *
     * doReleaseBrake for an Autochanger Linear Truck controller consists in
     * forcing to ONE bit 15 of parameter DigitalOutputFonctionnalityState.      *
     * Can't be a command because if this controller is a slave we have to check
     * if master controller is enabled. See AutoChangerTwoTrucksModule.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Override
    public void doReleaseBrake() {
        //If the controller is not enabled and we release brakes it could lead to a 
        //lost of a filter.
        if (!isEnabled()) {
            String msg = getName() + " was not enable before releaseBrake.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }
        int val = readParameterInHexa(Parameter.DigitalOutputFonctionnalityState);
        writeParameter(Parameter.DigitalOutputFonctionnalityState,
                FcsUtils.force2one(val, 15));
        FCSLOG.info(getName() + ": brake released.");
    }

    /**
     * return true if brake if activated.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return true if brake if activated.")
    @Override
    public boolean isBrakeActivated() {
        int val = readParameterInHexa(Parameter.DigitalOutputFonctionnalityState);
        int[] digitalOutputInBinary = FcsUtils.toReverseBinary(val);
        return digitalOutputInBinary[15] == 0;
    }

    /**
     * activateBrakeAndDisable
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "activate brake to prevent motion and disable controller.")
    @Override
    public void activateBrakeAndDisable() {
        activateBrake();
        /* mandatory : we must let time to the controller to activate the brake */
        /* otherwise if we disable before brake are holding, a filter could fall. */
        try {
            Thread.sleep(500);
        } catch (InterruptedException ex) {
            FCSLOG.error(ex);
        }
        disable();
    }

    /**
     * enableAndReleaseBrake Can't be a command because if this controller is a
     * slave we have to check if master controller is enabled. See
     * AutoChangerTwoTrucksModule.
     */
    @Override
    public void enableAndReleaseBrake() {
        enable();
        doReleaseBrake();
    }

    @Override
    public void enableAndWriteAbsolutePosition(int pos) {
        checkInMode(PROFILE_POSITION);
        enable();
        doReleaseBrake();
        writeTargetPosition(pos);
        writeControlWord(0x3F);
    }

    @Override
    public void enableAndWriteRelativePosition(int pos) {
        enable();
        doReleaseBrake();
        changeMode(EposMode.PROFILE_POSITION);
        writeTargetPosition(pos);
        writeControlWord(0x3F);
    }

    /**
     * Configure this controller to be used as an autochanger linear rail truck
     * controller. This method is to be executed during initialization phase.
     *
     * This configures the digital inputs of the EPOS controller to work with a
     * holding brake.
     *
     * Digital Inputs activated for Linear Rails :
     * ******************************************** Drive Enable Input=01
     * Mask=Enabled Polarity=HighActive QuickStop Input=02 Mask=Enabled
     * Polarity=HighActive
     *
     **********************************************
     * 1 - Configure the Digital Input Functionnality index 0x2070 subindex 01
     * to 04 Functionnality tab Value = 15 General Purpose A Value = 14 General
     * Purpose B Value = 13 General Purpose C Value = 12 General Purpose D Value
     * = 11 General Purpose E Value = 10 General Purpose F Value = 9 General
     * Purpose G Value = 8 General Purpose H Value = 7 General Purpose I Value =
     * 6 General Purpose J Value = 5 QuickStop 1 Value = 4 Drive enable 1 Value
     * = 3 Position marker 0 Value = 2 Home switch 0 Value = 1 Positive Limit
     * switch 0 Value = 0 Negative Limit switch 0
     * ********************************************* 2 - Configure the Digital
     * Input Polarity index 0x2071 subindex 03 Value=0 HighActive Value=1 Low
     * Active ********************************************* 3 - Configure the
     * Digital Input Mask index 0x2071 subindex 02 Value=0 HighActive Value=1
     * Low Active ******************************************** 4 - Configure the
     * Digital Input Execution Mask index 0x2071 subindex 04 Used to suppress
     * execution of the digital input functionalities. The function (Negative or
     * Positive Limit Switch Error Routine) will be executed when the associated
     * bit in functionalities state register goes high and the bit in this
     * execution mask is set. Functionnality tab Value = 15 General Purpose A 0
     * Value = 14 General Purpose B 0 Value = 13 General Purpose C 0 Value = 12
     * General Purpose D 0 Value = 11 General Purpose E 0 Value = 10 General
     * Purpose F 0 Value = 9 General Purpose G 0 Value = 8 General Purpose H 0
     * Value = 7 General Purpose I 0 Value = 6 General Purpose J 0 Value = 5
     * QuickStop 1 Value = 4 Drive enable 1 Value = 3 Position marker 0 Value =
     * 2 Home switch 0 Value = 1 Positive Limit switch 0 Value = 0 Negative
     * Limit switch 0
     *
     *
     *
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the autochanger linear rails controllers.")
    //tested on CPPM testbench in september 2015
    @Override
    public void configureDigitalInputOfLinearRails() {
        //Drive Enable
        writeParameter(Parameter.ConfigurationOfDigitalInput1, 0x4);

        //QuickStop
        writeParameter(Parameter.ConfigurationOfDigitalInput2, 0x5);

        //DigitalInputPolarity 
        //Value=0 HighActive
        //Value=1 Low Active
        writeParameter(Parameter.DigitalInputFonctionnalityPolarity, 0x0);

        //DigitalInputFonctionnalitiesMask
        //Used to display the state of the digital input functionalities. 
        //If a bit is set to “1”, the functionality state will be displayed.
        writeParameter(Parameter.DigitalInputFonctionnalityMask, 0x30);

        //DigitalInputFonctionnalitiesExecutionMask
        writeParameter(Parameter.DigitalInputFonctionnalityExecutionMask, 0x30);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "check configuration of digital input of linear rail controller.")
    public void checkConfigDigitalInputOfLinearRails() {
        configDigitalInputSb = new StringBuilder();
        configDigitalInputSb.append(" Configuration of Digital Input:");
        checkParamInput(Parameter.ConfigurationOfDigitalInput1, "4");
        checkParamInput(Parameter.ConfigurationOfDigitalInput2, "5");
        checkParamInput(Parameter.DigitalInputFonctionnalityPolarity, "0");
        checkParamInput(Parameter.DigitalInputFonctionnalityMask, "30");
        checkParamInput(Parameter.DigitalInputFonctionnalityExecutionMask, "30");
        if (configDigitalInputOK) {
            FCSLOG.info(configDigitalInputSb.toString());
        } else {
            FCSLOG.error(configDigitalInputSb.toString());
        }
    }

    private void checkParamInput(Parameter param, String hexaValue) {
        int paramValue = this.readParameter(param);
        int configValue = Integer.parseInt(hexaValue, 16);
        if (configValue != paramValue) {
            configDigitalInputOK = false;
        }
        configDigitalInputSb.append(param).append("=");
        configDigitalInputSb.append(paramValue).append(";");
    }

    /**
     * Configuration of an EPOS controller to work with a holding brake. This is
     * used for controllers of the Autochanger Linear Rails.
     *
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Digital Outputsactivated
     * for Linear Rails are: ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     * General_A Output=01 Mask=Enabled Polarity=HighActive Ready/Fault
     * Output=02 Mask=Enabled Polarity=HighActive Holding Brake Output=03
     * Mask=Enabled Polarity=LowActive
     * ******************************************* 1- First of all, mask all
     * outputs index 0x2079 subindex 01
     * ********************************************** 2- Configure the Digital
     * Output Fonctionnalities index 0x2079 subindex 01 to 03 Functionnality tab
     * Value = 15 General Purpose Out_A Value = 14 General Purpose Out_B Value =
     * 13 General Purpose Out_C Value = 12 General Purpose Out_D Value = 11
     * General Purpose Out_E Value = 10..8 not used Value = 7..3 reserved Value
     * = 2 Holding Brake Value = 1 Position compare Value = 0 Ready/Fault
     * ************************************************************************************
     * 3 - Configure the Digital Output Functionnality Polarity index 0x2078
     * subindex 0x03 Value = 0 associated output not change => HighActive Value
     * = 1 associated output inverted => LowActive Tab Digital Output
     * Functionnality Mask Value Bit_15 General Purpose Out_A 0 Bit_14 General
     * Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12 General Purpose
     * Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General Purpose Out_F 0
     * Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H 0 Bit_7
     * reserved 0 Bit_6 reserved 0 Bit_5 reserved 0 Bit_4 reserved 0 Bit_3
     * reserved 0 Bit_2 Holding Brake 1 (read only) Bit_1 Position Compare 0
     * (read only) Bit_0 Ready/Fault 0 (read only)
     * ********************************************************************************
     * 4 - Configure the Digital Output Functionnality Mask index 0x2078
     * subindex 0x02 Value = 0 functionnality not activatd Value = 1
     * functionnality activated
     * ********************************************************************************
     * Digital Output Functionnality Mask Value Bit_15 General Purpose Out_A 1
     * Bit_14 General Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12
     * General Purpose Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General
     * Purpose Out_F 0 Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H
     * 0 Bit_7 reserved 0 Bit_6 reserved 0 Bit_5 reserved 0 Bit_4 reserved 0
     * Bit_3 reserved 0 Bit_2 Holding Brake 1 Bit_1 Position Compare 0 Bit_0
     * Ready/Fault 1
     * ********************************************************************************
     * To configure the autochanger linear rails controllers. This method is to
     * be executed during initialization phase.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the autochanger linear rails controllers.")
    //tested on CPPM testbench in september 2015
    @Override
    public void configureDigitalOutputOfLinearRails() {
        //1-Mask all outputs
        writeParameter(Parameter.DigitalOutputFonctionnalityMask, 0);

        //2-Configuration of Digital Output Fonctionnalities
        //General_A
        writeParameter(Parameter.ConfigurationOfDigitalOutput1, 0xF);
        //Ready/Fault
        writeParameter(Parameter.ConfigurationOfDigitalOutput2, 0);
        //Holding Brake
        writeParameter(Parameter.ConfigurationOfDigitalOutput3, 0x2);

        //3-Configuration of Digital Output Fonctionnality Polarity
        writeParameter(Parameter.DigitalOutputFonctionnalityPolarity, 0x4);

        //4-Configuration of Digital Output Fonctionnality Mask
        writeParameter(Parameter.DigitalOutputFonctionnalityMask, 0x8005);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "check configuration of digital output of linear rail controller.")
    public void checkConfigDigitalOutputOfLinearRails() {
        configDigitalOutputSb = new StringBuilder();
        configDigitalOutputSb.append(" Configuration of Digital Output:");
        checkParamOutput(Parameter.ConfigurationOfDigitalOutput1, "f");
        checkParamOutput(Parameter.ConfigurationOfDigitalOutput2, "0");
        checkParamOutput(Parameter.ConfigurationOfDigitalOutput3, "2");
        checkParamOutput(Parameter.DigitalOutputFonctionnalityPolarity, "4");
        checkParamOutput(Parameter.DigitalOutputFonctionnalityMask, "8005");
        if (configDigitalOutputOK) {
            FCSLOG.info(configDigitalOutputSb.toString());
        } else {
            FCSLOG.error(configDigitalOutputSb.toString());
        }
    }

    private void checkParamOutput(Parameter param, String hexaValue) {
        int paramValue = this.readParameter(param);
        int configValue = Integer.parseInt(hexaValue, 16);
        if (configValue != paramValue) {
            configDigitalOutputOK = false;
        }
        configDigitalOutputSb.append(param).append("=");
        configDigitalOutputSb.append(paramValue).append(";");
    }
}
