package org.lsst.ccs.subsystems.fcs.common;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.OperationalState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.framework.SignalLevel;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.FCSCst;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This represents a Mechanical Item which can be moved by a motor. Generaly we
 * want to execute an action on this mechanical item : - start the action, -
 * read sensors to know if the action is completed, - wait for the end of the
 * action. This module provides a general way to start an action and wait until
 * the action is completed. It factorises some code that otherwise would have to
 * be duplicated in each Module that needs such fonctionnalities.
 * 
 * This class raises alerts :
 * - FCS005 in halt, quickStop
 *
 * @author virieux
 */
public abstract class MobileItemModule implements HardwareController, AlertRaiser, SignalHandler, HasLifecycle {

    protected static final Logger FCSLOG = FCSCst.FCSLOG;

    @LookupName
    protected String name;
    
    protected volatile boolean hasToWaitForEndOfAction = false;
    protected volatile MobileItemAction currentAction;
    private volatile boolean moving = false;
    //Used because we have to wait for the update from the sensors to know if the action is completed
    protected final Lock lock = new ReentrantLock();
    private final Condition motionCompleted = lock.newCondition();

    private ScheduledFuture<?> readSensorsHandle;
    protected final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(2);
    //prendre un Scheduler qui peut me permettre de donner un nom au Thread.

    private final AtomicBoolean haltRequired = new AtomicBoolean(false);
    private final AtomicBoolean stopRequired = new AtomicBoolean(false);

    /**
     * Return true if the MobileItemMobile is moving.
     * That means that we are waiting for an action to be completed.
     * @return 
     */
    public boolean isMoving() {
        return moving;
    }

    //for the simulation
    public AtomicBoolean getHaltRequired() {
        return haltRequired;
    }

    /**
     * This stops the thread reading sensors.
     *
     */
    private void cancelReadingSensors() {
        lock.lock();
        try {
            FCSLOG.debug(name + " => stop reading sensors");
            motionCompleted.signal();
        } finally {
            lock.unlock();
        }
        this.readSensorsHandle.cancel(true);
        FCSLOG.debug(name + " => readingSensors canceled");
    }

    /**
     * Return True if all CAN open devices useful for this MobileItemMobile are ready for an action.
     * As an example, if this mobil item is moved by a controller and sensors are read through a sensors gateway,
     * this method will return True when the controller and the gateway are ready to be used (booted, configured, 
     * homing done if needed).
     * @return 
     */
    public abstract boolean myDevicesReady();

    /*
     * This method must returns true if the action is completed, false otherwise.
     * It's abstract because it'up to the sub-classes to know when the action is completed.
     */
    public abstract boolean isActionCompleted(MobileItemAction action);

    /**
     * reads sensors or any device and updates state from sensors
     * it can be also reading a position or a current on a motor controller.
     * It's abstract because it'up to the sub-classes to know what to do.
    */
    public abstract void updateStateWithSensorsToCheckIfActionIsCompleted();

    /**
     * starts the action given as argument.
     * It has to be specified in sub-classes.
     * Usually it consist in sending a command to a controller : give a value of current or give a
     * target position. 
     * @param action 
     */
    public abstract void startAction(MobileItemAction action);

    /**
     * aborts action actually running.
     * It has to be specified in sub-classes.
     * It can be : write a zero value of current or send a stop command to a controller.
     * @param action
     * @param delay 
     */
    public abstract void abortAction(MobileItemAction action, long delay);

    public abstract void quickStopAction(MobileItemAction action, long delay);

    public abstract void postAction(MobileItemAction action);

    public abstract void publishData();
    
    protected void checkReadyForAction() {
        if (!myDevicesReady() && this.getSubsystem().isInState(OperationalState.NORMAL)) {
            throw new FcsHardwareException(name
                    + ": hardware is not ready to execute ACTION commands.");
        }
        
        if (this.getSubsystem().isInState(AlertState.ALARM)) {
            throw new FcsHardwareException(name
                    + ": can't execute ACTION commands when subsystem is in ALARM state.");
        }
    }

    /**
     * This executes an action which moves the MobileItem and waits for the end
     * od this action. The MobileItemAction is an ENUM from the fcs.bus package.
     *
     * @param action, timeout
     * @param timeoutForAction
     * @throws FailedCommandException if the action can't be completed
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    protected void executeAction(MobileItemAction action, long timeoutForAction) {
        
        checkReadyForAction();

        lock.lock();

        try {
            this.hasToWaitForEndOfAction = true;

            this.currentAction = action;

            this.moving = true;

            FCSLOG.info(name + ": ===> STARTING ACTION:" + action.toString());

            startAction(action);

            readSensorsUntilActionIsCompleted(action, System.currentTimeMillis(), timeoutForAction);

            waitForEndOfAction(action);

            //At this stage, the action can be completed with success or a 
            //HALT signal has been received.
            if (haltRequired.get()) {
                abortAction(action, 0);
                moving = false;
                FCSLOG.info(name + "=> Received an ABORT command for action :" + action.toString());

            } else if (stopRequired.get()) {
                quickStopAction(action, 0);
                moving = false;
                FCSLOG.info(name + "=> Received a STOP command for action :" + action.toString());

            } else if (isActionCompleted(action)) {
                moving = false;
                postAction(action);
                FCSLOG.info(name + " ===> ACTION COMPLETED : " + action.doneString());

            } else {
                String msg = getName() + "=>" + action.getFailureMsg();
                FCSLOG.error(msg);
                throw new FailedCommandException(msg);
            }

        } finally {

            FCSLOG.debug(name + ": finally in executeAction.");
            abortAction(action, 0);
            this.moving = false;
            this.hasToWaitForEndOfAction = false;
            this.motionCompleted.signal();
            haltRequired.set(false);
            stopRequired.set(false);
            lock.unlock();
            publishData();
            FCSLOG.info(name + " ===> END ACTION:" + action.toString());

        }
    }

    /**
     * Start reading the sensors at a fixed rate (scheduler.scheduleAtFixedRate)
     * until the action is completed or we received a halt signal or the timeout
     * for this action is past.
     *
     * @param action
     * @param beginTime
     *
     * @param timeout
     */
    public void readSensorsUntilActionIsCompleted(final MobileItemAction action,
            final long beginTime, final long timeout) {

        final Runnable readSensors;
        readSensors = new Runnable() {

            @Override
            public void run() {

                try {
                    FCSLOG.info(name + " : reading sensors");
                    long duration = System.currentTimeMillis() - beginTime;

                    updateStateWithSensorsToCheckIfActionIsCompleted();

                    boolean actionCompleted = isActionCompleted(action);

                    //we got a HALT signal, we have to stop reading the sensors.
                    if (haltRequired.get() || stopRequired.get()) {
                        hasToWaitForEndOfAction = false;
                        FCSLOG.info(name + ":" + action.toString() + " ABORT or STOP REQUESTED FOR ACTION "
                                + "BY ABORT or STOP COMMAND");
                        cancelReadingSensors();

                        //action is completed with success.
                    } else if (actionCompleted) {
                        hasToWaitForEndOfAction = false;
                        FCSLOG.info(name + ":" + action.toString() + " ACTION COMPLETED");
                        cancelReadingSensors();

                        //action is not completed but timeout is over.
                    } else if (duration >= timeout) {
                        hasToWaitForEndOfAction = false;
                        FCSLOG.info(name + ":" + action.toString() + " ACTION NOT COMPLETED during allocated time");
                        cancelReadingSensors();
                        throw new FcsHardwareException(name + ":" + action.toString()
                                + " exceeded timeout for this task: duration=" + duration + ",timeout=" + timeout);

                        //action is not completed, we have to continue reading sensors.
                    } else {
                        FCSLOG.info(name + ":" + action.toString() + " not completed.....");
                        FCSLOG.info(name + ":" + action.name() + "/duration=" + duration);

                    }

                } catch (Exception ex) {

                    hasToWaitForEndOfAction = false;
                    cancelReadingSensors();
                    FCSLOG.error(name+": ERROR during action "+ action.name(),ex);
                }
            }
        };
        this.readSensorsHandle = scheduler.scheduleAtFixedRate(readSensors, 500, 500, TimeUnit.MILLISECONDS);
    }

    /**
     * This method waits until the action is completed. This methods is called
     * by executeAction and has already acquired the lock.
     */
    private void waitForEndOfAction(MobileItemAction action) {
        while (hasToWaitForEndOfAction) {
            try {
                FCSLOG.info(name + " waiting for end of " + action.toString());
                motionCompleted.await();
            } catch (InterruptedException ex) {
                FCSLOG.info(name + ": InterruptedException received=" + ex.toString());
                break;
            }
        }
        FCSLOG.info(name + " STOP WAITING FOR END OF ACTION");
    }

    private void halt(MobileItemAction action, long delay) {
        if (action == null) {
            FCSLOG.warning(name + ": no current action running => nothing to abort.");
            return;
        }
        FCSLOG.debug(name + ": ABORTING ACTION " + action.toString() + "within delay=" + delay);
        abortAction(action, delay);
        moving = false;
    }

    private void quickStop(MobileItemAction action, long delay) {
        if (action == null) {
            FCSLOG.warning(name + ": no current action running => nothing to stop.");
            return;
        }
        FCSLOG.debug(name + ": QUICKSTOP for" + action.toString() + "within delay=" + delay);
        quickStopAction(action, delay);
        moving = false;
    }

    /**
     * just to avoid taping "sendSignal HALT"
     *
     * @param delay
     */
    @Command(type = Command.CommandType.SIGNAL, level = Command.ENGINEERING1,
            description = "Abort the running action within a delay.")
    public void abort(long delay) {
        sendSignalWithTimeLimit("HALT", delay);
    }

    @Command(type = Command.CommandType.SIGNAL, level = Command.ENGINEERING1,
            description = "Abort the running action.")
    public void abort() {
        abort(0);
    }

    /**
     * just to avoid taping "sendSignal STOP"
     *
     * @param delay
     */
    @Command(type = Command.CommandType.SIGNAL, level = Command.ENGINEERING1,
            description = "Stop the running action with a quickstop.")
    public void stop(long delay)  {
        sendSignalWithTimeLimit("STOP", delay);
    }

    /**
     * just to avoid taping "sendSignal STOP"
     *
     */
    @Command(type = Command.CommandType.SIGNAL, level = Command.ENGINEERING1,
            description = "Stop the running action with a quickstop.")
    public void stop()  {
        stop(0);
    }

    /**
     * just to avoid taping "sendSignal STOP"
     *
     */
    @Command(type = Command.CommandType.SIGNAL, level = Command.ENGINEERING1,
            description = "Stop the running action with a quickstop.")
    public void quickstop()  {
        stop(0);
    }

    /**
     * This shutdowns the scheduler. This is executed during CLOSING phase when
     * the MobileItemMobile is stopped. cf checkStopped
     */
    @Override
    public void shutdownNow() {
        scheduler.shutdown();
    }

    @Override
    public TreeWalkerDiag signal(Signal signal) {
        SignalLevel sl = signal.getLevel();
        FCSLOG.info(sl.toString());

        if (!this.moving) {
            FCSLOG.warning(name + " is not moving; nothing to stop.");

        } else {

            switch (signal.getLevel()) {
                case HALT:
                    FCSLOG.info(name + " HALT required");
                    this.haltRequired.set(true);
                    halt(currentAction, signal.getTimeHint());
                    break;
                case STOP:
                    FCSLOG.info(name + " STOP required");
                    this.stopRequired.set(true);
                    quickStop(currentAction, signal.getTimeHint());
                    break;
                case RE_START:
                    this.haltRequired.set(false);
                    break;
                default:
                    assert false;

            }
        }
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    @Override
    public void checkStopped() throws HardwareException {
        if (this.isMoving()) {
            throw new HardwareException(false, getName()
                    + " is moving, has to be stopped before a shutdown.");
        }
    }

}
