package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.LO_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This is the model for the loader in the Filter Exchange System. The loader is
 * used to load a filter into the camera or to unload a filter from the camera.
 *
 * @author virieux
 */
public class LoaderModule implements HardwareController, FilterHolder, AlertRaiser, HasLifecycle {

    @LookupField(strategy = Strategy.TOP)
    private Subsystem s;

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private LoaderCarrierModule carrier;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private LoaderClampModule clamp;

    @LookupField(strategy = Strategy.BYNAME)
    private BridgeToHardware loaderTcpProxy;

    @LookupField(strategy = Strategy.TREE)
    private MainModule main;

    @LookupField(strategy = Strategy.BYNAME)
    private FilterHolder autochanger;

    @LookupField(strategy = Strategy.BYNAME)
    private RedondantSensors loaderFilterPresenceSensors;

    @LookupField(strategy = Strategy.BYNAME)
    private RedondantSensors loaderOnCameraSensors;

    @LookupField(strategy = Strategy.BYNAME)
    private PlutoGatewayInterface loaderPlutoGateway;

    @LookupField(strategy = Strategy.BYNAME)
    private EPOSController hooksController;

    @LookupField(strategy = Strategy.BYNAME)
    private EPOSController carrierController;

    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

    /**
     * Returns carrier.
     *
     * @return carrier
     */
    public LoaderCarrierModule getCarrier() {
        return carrier;
    }

    /**
     * Returns clamp.
     *
     * @return
     */
    public LoaderClampModule getClamp() {
        return clamp;
    }

    /**
     *
     * @return true if hardware (controllers and plutoGateway) is correctly
     * initialized and checkHardwareStateAndDoHomingIfPossible of the
     * controllers is done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if hardware (controllers and plutoGateway) is correctly initialized"
            + "and homing of the controllers is done.")
    public boolean isInitialized() {
        boolean devicesInitialized = loaderPlutoGateway.isInitialized()
                && hooksController.isInitialized()
                && carrierController.isInitialized();
        return devicesInitialized && clamp.isHomingDone() && carrier.isInitialized();
    }

    /**
     * Returns the boolean field empty. If the empty boolean is being updated
     * and waits for a response from a sensor, this methods waits until empty is
     * updated. If the field empty is not being updated, it returns immediatly
     * the field empty.
     *
     * @return empty
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if there is no filter in the loader. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {

        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.", ex);
                }

            }
            return !loaderFilterPresenceSensors.isOn();

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * atHandoff is updated. If the field atHandoff is not being updated, it
     * returns immediatly the field atHandoff.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the loader is connected on the camera. This command doesn't read again the sensors.")
    public boolean isConnectedOnCamera() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.", ex);
                }

            }
            return loaderOnCameraSensors.isOn();

        } finally {
            lock.unlock();
        }
    }

    /**
     * Return true if the autochanger is holding the filter. This command
     * doesn't read again the sensors.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the autochanger is holding the filter. This command doesn't read again the sensors.")
    public boolean isAutochangerHoldingFilter() {
        return autochanger.isHoldingFilter();
    }

    /**
     * Return true if a filter is present and it is held by the loader clamp.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if a filter is present and it is held by the loader clamp.")
    @Override
    public boolean isHoldingFilter() {
        this.updateStateWithSensors();
        this.clamp.updatePosition();
        return this.clamp.isClamped() && !this.isEmpty();
    }

    /**
     * This command has to be executed after the initialization phase and after
     * the checkHardware command. It can't be automaticaly executed during
     * initialization phase because it's not compliant with the CCS requirement
     * which says that the initialization of the subsystem can't make move
     * hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Attention ! this commands does the homing of the controllers and might move hardware. "
            + "Initialize loader hardware.",
            alias = "homing")
    public void locateHardware() {
        /* check that plutoGateway and controllers are booted and initialized, and controllers not in fault.*/
        /* read sensors and update state*/
        /*check that position read on carrier controller is confirmed by sensors.*/
            carrier.initializeHardware();
        boolean devicesInitialized = loaderPlutoGateway.isInitialized()
                && hooksController.isInitialized()
                && carrierController.isInitialized();    
        if (devicesInitialized) {
            updateStateWithSensors();
            /* check that clamp is not empty and closed.*/
            checkNotEmptyAndClosed();
            if (isEmpty()) {
                /* no filter in the loader*/
                homingWhenNoFilter();

            } else {
                /* a filter is detected by filterPresenceSensor*/

            }
        } else {
            throw new FailedCommandException(name + " couldn't locate hardware because devices are not initialized.");
        }
        this.updateFCSStateToReady();
    }

    private void homingWhenNoFilter() {
        if (!carrier.isAtStoragePosition()) {
            carrier.goToStorage();
        }
        clamp.open();
    }

    private void checkNotEmptyAndClosed() {
        /* If the carrier is empty and the clamp is CLOSED, we can't start the loader subsystem.*/
        if (this.isEmpty() && clamp.isClosed()) {
            String msg = name + ": carrier is empty and clamp is CLOSED - can't start.";
            this.raiseAlarm(HARDWARE_ERROR, msg, name);
            throw new FcsHardwareException(msg);
        }
    }

    /**
     * Check that plutoGateway is booted and initialize plutoGateway. 
     * Read sensors and update clamp's lockStatus.
     */
    @Override
    public void postStart() {
        if (loaderPlutoGateway.isBooted()) {
            initializeGateway();
        } else {
            loaderPlutoGateway.raiseAlarmIfMissing();
        }
    }

    public void initializeGateway() {
        try {
            this.loaderPlutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " could not initialize loaderPlutoGateway", ex);
        }
    }

    /**
     * CheckStarted is executed by completeInitialization when a first start has
     * failed. Because end user could have changed many things, we have to check
     * again that all is correct.
     *
     */
    @Override
    public void checkStarted()  {

        FCSLOG.info(name + " BEGIN checkStarted");
        this.initializeGateway();
        FCSLOG.info(name + " END checkStarted");
    }


    /**
     * Check if loader is connected on camera.
     *
     * @throws RejectedCommandException if not connected on camera
     */
    public void checkConnectedOnCamera() {
        if (!isConnectedOnCamera()) {
            throw new RejectedCommandException(name
                    + " Loader not connected - can't execute commands.");
        }
    }

    /**
     * Checks if the loader carrier can move.
     *
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the carrier can move.")
    public void checkConditionsForCarrierMotion() {
        FCSLOG.info(name + " checking pre-conditions for carrier motion");

        updateStateAndCheckSensors();

        if (!this.isEmpty()) {

            if (this.isHoldingFilter() && isAutochangerHoldingFilter()) {
                throw new RejectedCommandException(name
                        + " carrier can't move because a filter is in the loader"
                        + " and it's held by loader AND autochanger.");

            } else if (!this.isHoldingFilter() && !isAutochangerHoldingFilter()) {
                throw new RejectedCommandException(name
                        + " carrier can't move because the filter in the loader is not held"
                        + " neither by loader neither by autochanger. "
                        + "Close loader clamp or autochanger latches. ");

            }
        }
    }

    /**
     * Check if the clamp can be opened.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the clamp can be opened.")
    public void checkConditionsForOpeningHooks() {

        FCSLOG.info(name + " checking pre-conditions for opening hooks");

        updateStateAndCheckSensors();

        if (!this.isEmpty()) {
            if (!carrier.isAtHandoffPosition()) {
                String msg = name + ": carrier is loaded with a filter but not "
                        + "at handoff position - can't open clamp.";
                FCSLOG.error(msg);
                throw new RejectedCommandException(msg);
            }

            if (!isAutochangerHoldingFilter()) {
                String msg = name + ": A filter is in the loader but not held by autochanger "
                        + "- can't open clamp.";
                FCSLOG.error(msg);
                throw new RejectedCommandException(msg);
            }
        }
    }

    public void checkConditionsForUnclampingHooks() {
        if (!carrier.isAtHandoffPosition()) {
            String msg = name + ": carrier is loaded with a filter but not "
                    + "at handoff position - can't unclamp.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
        if (isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(name
                    + " Autochanger is holding filter. Can't unclamp Loader Clamp.");
        }
    }

    /**
     * Check if the clamp can be closed. Clamp can be close if sensors are not
     * in error and a filter is in the loader.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the clamp can be closed.")
    public void checkLoaderNotEmpty() {
        FCSLOG.info(name + " checking pre-conditions for closing hooks");
        updateStateAndCheckSensors();

        this.clamp.updatePosition();

        if (this.isEmpty()) {
            String msg = name + ": no filter in loader - can't execute close, clamp command."
                    + clamp.getName();
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
    }

    public void updateStateAndCheckSensors() {
        updateStateWithSensors();
        checkConnectedOnCamera();
        clamp.checkInitialized();
        clamp.checkSensors(LO_SENSOR_ERROR);
        carrier.checkSensors(LO_SENSOR_ERROR);
    }

    /**
     * This methods updates the carrier and clamp state in reading all the
     * sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors() {

        this.loaderTcpProxy.publishData();

        loaderPlutoGateway.checkBooted();
        loaderPlutoGateway.checkInitialized();

        lock.lock();
        try {
            updatingState = true;
            this.loaderPlutoGateway.updateValues();
            int[] readHexaValues = this.loaderPlutoGateway.getHexaValues();

            this.loaderFilterPresenceSensors.updateValue(readHexaValues);
            this.loaderOnCameraSensors.updateValue(readHexaValues);
            this.clamp.updateStateWithSensors(readHexaValues);
            this.carrier.updateStateWithSensors(readHexaValues);

        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
            this.clamp.publishData();
            this.carrier.publishData();
        }
    }

    /**
     * Update FCS state and FCS readyness state and publish on the status bus.
     * Check that Loader hardware is ready to be operated and moved. This means
     * that : - all CAN open devices are booted, identified and initialized, -
     * checkHardwareStateAndDoHomingIfPossible has been done on the controllers.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update FCS state and FCS readyness state and publishes on the status bus.")
    public void updateFCSStateToReady() {
        if (clamp.isHomingDone() && carrier.isInitialized()) {
            main.updateFCSStateToReady();
        }
    }

    /**
     * This command can be launched when a filter is in the loader and we want
     * to put in inside the camera.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Load a filter from the loader into the camera.")
    public void loadFilterInCamera() {

        checkConnectedOnCamera();
        if (!isHoldingFilter()) {
            throw new RejectedCommandException(name
                    + ":loader is not holding a filter : can't load a filter into camera.");
        }
        if (!carrier.isAtStoragePosition()) //TODO : what do we do in this case ?
        {
            throw new RejectedCommandException(name
                    + ":carrier loader is not at storage position : can't load a filter into camera.");
        }

        //go to Handoff position with the filter
        carrier.goToHandOff();
        if (main.isHaltRequired() || main.isStopRequired()) {
            throw new FailedCommandException(name + ": received HALT or STOP command.");
        }
        //wait until the autochanger hold the filter
        while (!isAutochangerHoldingFilter()) {
            FCSLOG.debug(name + " waiting until autochanger holds the filter...");
            try {
                Thread.sleep(300);
            } catch (InterruptedException ex) {
                throw new FcsHardwareException("loadFilterInCamera interrupted while waiting for the autochanger"
                        + " to take the filter", ex);
            }
            updateStateWithSensors();
        }

        if (isAutochangerHoldingFilter()) {
            clamp.open();
        }
        if (main.isHaltRequired() || main.isStopRequired()) {
            throw new FailedCommandException(name + ": received HALT or STOP command.");
        }

        // empty carrier go to Storage
        carrier.goToStorage();
    }

    /**
     * This command can be launched when the loader is empty and we want to take
     * a filter from the camera. A filter must be at handoff position and held
     * by autochanger.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     * @throws SDORequestException
     * @throws FailedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Unload a filter from the camera into the loader.")
    public void unloadFilterFromCamera() {

        updateStateWithSensors();
        checkConnectedOnCamera();

        if (isHoldingFilter()) {
            throw new RejectedCommandException(name + ":loader is holding a filter : "
                    + "can't unload a filter from camera.");
        }

        //wait for the autochanger to put a filter at handoff position
        //in loader test bench it's just a switch to put on.
        while (!isAutochangerHoldingFilter()) {
            FCSLOG.debug(name + " waiting until autochanger holds the filter...");
            try {
                Thread.sleep(300);
            } catch (InterruptedException ex) {
                throw new FcsHardwareException("unloadFilterFromCamera interrupted while waiting until autochanger holds the filter", ex);
            }
            updateStateWithSensors();
        }

        //go tho Handoff position - carrier empty
        carrier.goToHandOff();
        if (main.isHaltRequired() || main.isStopRequired()) {
            throw new FailedCommandException(name + ": received HALT or STOP command.");
        }

        //at handoff position a filter should be here.
        updateStateWithSensors();
        if (isEmpty()) {
            throw new FailedCommandException(name
                    + ": loader presence filter sensor doesn't detect a filter"
                    + " - can't go on.");
        }
        clamp.close();

        //wait for the autochanger to unlock the filter at handoff position
        //in loader test bench it's just a switch to put off.(switch A22)
        while (isAutochangerHoldingFilter()) {
            FCSLOG.debug(name + " waiting until autochanger releases the filter...");
            try {
                Thread.sleep(300);
            } catch (InterruptedException ex) {
                throw new FcsHardwareException("unloadFilterFromCamera interrupted while waiting until "
                        + "autochanger releases the filter", ex);
            }
            updateStateWithSensors();
        }

        //close more fermely the hooks to hold the filter.
        clamp.clamp();
        if (main.isHaltRequired() || main.isStopRequired()) {
            throw new FailedCommandException(name + ": received HALT or STOP command.");
        }

        //carrier holding filter goes to storage position.
        carrier.goToStorage();
    }

    /**
     * Creates an Object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByLoader createStatusDataPublishedByLoader() {
        StatusDataPublishedByLoader status = new StatusDataPublishedByLoader();
        status.setName(name);
        status.setFilterPresenceSensorValue(loaderFilterPresenceSensors.isOn());
        status.setFilterPresenceSensorsInError(loaderFilterPresenceSensors.isInError());
        status.setLoaderOnCameraSensorValue(loaderOnCameraSensors.isOn());
        status.setLoaderOnCameraSensorsInError(loaderOnCameraSensors.isInError());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        StatusDataPublishedByLoader status = this.createStatusDataPublishedByLoader();
        KeyValueData kvd = new KeyValueData("loaderGeneral", status);

        s.publishSubsystemDataOnStatusBus(kvd);
    }

    @Override
    public boolean isAtHandoff() {
        return carrier.isAtHandoffPosition();
    }

    @Override
    public boolean isAtStandby() {
        return false;
    }

    /**
     * Print list of hardware with initialization information.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Print list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(carrier.printHardwareState());
        sb.append("\n");
        sb.append(clamp.printHardwareState());
        return sb.toString();
    }
}
