
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_TIMEOUT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.LO_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This is model for the clamps mechanism in the loader. 
 * 4 hooks are used for the clamping of the filter in the loader. 
 * This hooks are moved all together by the hooksController.
 * 
 * Raises alert :
 * - CAN_BUS_TIMEOUT in updateStateWithSensorsToCheckIfActionIsCompleted
 *
 * @author virieux
 */
public class LoaderClampModule extends MobileItemModule implements MovedByEPOSController, ControlledBySensors {

    private final LoaderHookModule hook1;
    private final LoaderHookModule hook2;
    private final LoaderHookModule hook3;
    private final LoaderHookModule hook4;
    
    private final ForceSensor forceSensor0;
    private final ForceSensor forceSensor1;
    private final DigitalSensor clampedStatusSensor;
    
    /* a list of hooks to make easier the computation of the LockStatus */
    @LookupField(strategy=Strategy.CHILDREN)
    private LoaderHookModule[] hooks;
    
    @LookupField(strategy=Strategy.BYNAME)
    private EPOSController hooksController;
    
    @LookupField(strategy=Strategy.TREE)
    private LoaderModule loader;
    
    @LookupField(strategy=Strategy.BYNAME)
    private BridgeToHardware loaderTcpProxy;

    private FcsEnumerations.LockStatus lockStatus = LockStatus.UNKNOWN;

    private FcsEnumerations.LockStatus forceStatus;

    @ConfigurationParameter(description="timeout in milliseconds : if closing the clamp last more than "
            + "this amount of time, then the subsystem goes in ERROR.")
    private int timeoutForClosingHooks = 60000;
    
    @ConfigurationParameter(description="timeout in milliseconds : if closing strongly the clamp last "
    + "more than this amount of time, then the subsystem goes in ERROR.")
    private int timeoutForClampingHooks = 60000;
    
    @ConfigurationParameter(description="timeout in milliseconds : if opening the clamp last more than"
    + " this amount of time, then the subsystem goes in ERROR.")
    private int timeoutForOpeningHooks = 60000;
    
    
    private static final int TIMEOUT_FOR_MOVING_CLAMP = 60000;
    
    @ConfigurationParameter(description="target encoder absolute value in qc to open")
    private int targetPositionToOpen = 0;
    
    @ConfigurationParameter(description="target encoder absolute value in qc when hooks are CLOSED")
    private int relativePositionToClose = 515000;
    
    @ConfigurationParameter(description="target encoder absolute value in qc when hooks are CLAMPED")
    private int targetPositionToClamp = 540000;
    
    @ConfigurationParameter(description="relative position in qc to unclamp when hooks are CLAMPED")
    private int relativePositionToUnclamp = -25000;    
    
    @ConfigurationParameter(description="current to clamp hooks, in mA")
    private int currentToClamp = 450;
    
    @ConfigurationParameter(description="current to open hooks, in mA")
    private int currentToOpen = -150;
    
    private int positionToReach = 0;
    
    private volatile boolean homingDone = false;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    private int position;
    private int readCurrent;
    
    /**
     * Build a new LoaderClampModule with 4 hooks and the parameters to configure the EPOS controller
     * in mode CURRENT and in mode HOMING.
     * @param hook1
     * @param hook2
     * @param hook3
     * @param hook4
     * @param forceSensor0
     * @param forceSensor1
     * @param clampedStatusSensor 
     */
    public LoaderClampModule(
            LoaderHookModule hook1,
            LoaderHookModule hook2,
            LoaderHookModule hook3,
            LoaderHookModule hook4,
            ForceSensor forceSensor0,
            ForceSensor forceSensor1,
            DigitalSensor clampedStatusSensor) {
        this.hook1 = hook1;
        this.hook2 = hook2;
        this.hook3 = hook3;
        this.hook4 = hook4;
        this.hooks = new LoaderHookModule[]{hook1, hook2, hook3, hook4};
        this.forceSensor0 = forceSensor0;
        this.forceSensor1 = forceSensor1;
        this.clampedStatusSensor = clampedStatusSensor;
    }
    
    /**
     * Returns controller name.
     * @return 
     */
    @Override
    public String getControllerName() {
        return hooksController.getName();
    }

    @Override
    public boolean isControllerInFault() {
        return hooksController.isInError(); 
    }

    //for the simulator
    public int getTargetPositionToOpen() {
        return targetPositionToOpen;
    }

    //for the simulator
    public int getRelativePositionToClose() {
        return relativePositionToClose;
    }

    //for the simulator
    public int getRelativePositionToUnclamp() {
        return relativePositionToUnclamp;
    }

    //for the simulator
    public int getTargetPositionToClamp() {
        return targetPositionToClamp;
    }

    //for the simulator
    public int getCurrentToClamp() {
        return currentToClamp;
    }

    //for the simulator
    public int getCurrentToOpen() {
        return currentToOpen;
    }

    /**
     * Return position for end user.
     * Do not read again controller.
     * @return position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Display position for end user."
                    + "Do not read again controller.")
    public int getPosition() {
        return position;
    }
    
    

    /**
     * Returns true if loader clamp is homingDone.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if homing of loader clamp has been done.")
    public boolean isHomingDone() {
        return homingDone;
    }

    /**
     * 
     * @return lockStatus
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(name + ": interrupted in getLockStatus.",ex);
                }

            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns true if LockStatus=CLOSED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=CLOSED")
    public boolean isClosed() {
        return this.getLockStatus() == LockStatus.CLOSED;
        //TODO : check also forceStatus.
    }

    /**
     * Returns true if LockStatus=OPENED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=OPENED")
    public boolean isOpened() {
        return this.getLockStatus() == LockStatus.OPENED;
    }
    
    /**
     * Returns true if LockStatus=CLAMPED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=CLAMPED")
    public boolean isClamped() {
        return this.getLockStatus() == LockStatus.CLAMPED;
    }    

    /**
     * Returns true if clamp LockStatus is in ERROR or LockStatus is UNKNOWN.
     * That means that one hook is in ERROR status or one or more of the 4 hooks has a different
     * LockStatus than the other.
     * @return true if clamp LockStatus is in ERROR or LockStatus is UNKNOWN.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=ERROR or UNKNOWN.")
    @Override
    public boolean isInError() {
        return this.getLockStatus() == LockStatus.ERROR || this.getLockStatus() == LockStatus.UNKNOWN;
    }
    
    public boolean isUnderLoad() {
        return getLockStatus() == LockStatus.UNDER_LOAD;
    }


    /**
     * Returns true if loader CANopen devices are booted, identified and homingDone.
     * @return 
     */
    @Override
    public boolean myDevicesReady() {
        return loaderTcpProxy.allDevicesBooted();
    }





    /**
     * If loader is empty, go to home position otherwise do nothing.
     * ATTENTION : this command moves the clamp.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Attention : this commands moves the clamp to do the homing of the controller. "
                    + "If loader is empty, go to home position otherwise do nothing.")
    public void checkHardwareStateAndDoHomingIfPossible()  {

        loader.updateStateWithSensors();
        publishData();
        checkSensors(LO_SENSOR_ERROR);
        hooksController.checkInitialized();

        if (loader.isEmpty()) {
            this.open();
            this.homingDone = true;
            loader.updateFCSStateToReady();
        } 

    }
    
    @Override
    public void postStart() {
        if (hooksController.isBooted()) {
            initializeController();
        } 
    }
    
    public void initializeController() {
        try {
            hooksController.initializeAndCheckHardware();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + ":" + ex);
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " could not initialize loader clamp controller", ex);
        }
    }
    
    
    /**
     * Check if clamp controller is initialized.
     * @throws RejectedCommandException if clamp is not homingDone 
     */
    public void checkInitialized() {
        if (!hooksController.isInitialized()) {
            String msg = getName() + ": clamp is not intialized.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
    }

    /**
     * Open clamp in order to release filter.
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open clamp.")
    public void open()  {
        
        loader.updateStateAndCheckSensors();
        loader.checkConditionsForOpeningHooks();
        
        if (isClosed()) {
            this.executeAction(FcsEnumerations.MobileItemAction.OPENLOADERHOOKS,timeoutForOpeningHooks);
            
        } else if (isOpened()) {
            FCSLOG.info(getName() + " is already OPENED. Nothing to do.");
            
        } else {
            throw new RejectedCommandException(getName() + " has to be CLOSED before an open action.");
        }
    }


    
    /**
     * Closes clamp in order to hold the filter.
     * This action closes softly the clamp on the filter. The filter can't fall although it is not tight fermely.
     * When the clamp is closed, the filter is held by loader so autochanger can open its latches
     * and move trucks back without the filter.
     * Closing hooks consists in setting controller mode to PROFILE_POSITION, and going to relative position
     * relativePositionToClose.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Closes clamp.", timeout = TIMEOUT_FOR_MOVING_CLAMP)
    public void close() {
        
        loader.updateStateAndCheckSensors();
        loader.checkLoaderNotEmpty();
        
        if (this.isOpened()) {
            positionToReach = position + relativePositionToClose;
            this.executeAction(MobileItemAction.CLOSELOADERHOOKS, timeoutForClosingHooks);

        } else if (this.isClosed()) {
            FCSLOG.info(getName() + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(getName() + " has to be OPENED before a close action.");
        }
    }
    
    

    /**
     * Clamps hooks.
     * When clamp is clamped, the filter is held tightly and carrier can go to STORAGE position safely.
     * Clamping consists in setting controller in mode CURRENT and send currentToClamp to controller.
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Clamp to hold tightly a filter.")
    public void clamp()  {
        
        loader.updateStateAndCheckSensors();
        
        if (isClosed()) {
            loader.checkLoaderNotEmpty();
            checkConditionsForClampingHooks();
            this.executeAction(FcsEnumerations.MobileItemAction.CLAMPLOADERHOOKS,timeoutForClampingHooks);
            
        } else if (isClamped()) {
            FCSLOG.info(getName() + " is already CLAMPED. Nothing to do.");
            
        } else {
            throw new RejectedCommandException(getName() + " has to be CLOSED before a clamp action.");
        }       
    }
    
    private void checkConditionsForClampingHooks() {
        if (loader.isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + " Autochanger is holding filter. Open autochanger latches before "
                    + "clamping loader hooks.");
        }
    }
    
    /**
     * unclamp Loader clamp.
     * At the end of this action, hooks are still CLOSED, but doesn't hold tightly filter.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Unclamp filter and return to CLOSED position.")   
    public void unclamp() {
        
        loader.updateStateAndCheckSensors();
        
        if (isClamped() || isUnderLoad()) {
            loader.checkLoaderNotEmpty();
            loader.checkConditionsForUnclampingHooks();
            positionToReach = position + relativePositionToUnclamp;
            FCSLOG.info("positionToReach=" + positionToReach);
            this.executeAction(FcsEnumerations.MobileItemAction.UNCLAMPLOADERHOOKS,
                    timeoutForClampingHooks);
            
        } else if (isClosed()) {
            FCSLOG.info(getName() + " is already CLOSED. Nothing to do.");
            
        } else {
            throw new RejectedCommandException(getName() + " has to be CLAMPED before an unclamp action.");
        }
    }


    /**
     * Check if the action is completed. This method is called after
     * updateStateWithSensorsToCheckIfActionIsCompleted where the different new
     * values of current or position or sensors values have been updated.
     *
     * @param action
     * @return
     */
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        /* A loader clamp motion is completed when the position is in a range of */
        /* 10 microns around the target position. */
        boolean actionCompleted = false;
        if (action == MobileItemAction.OPENLOADERHOOKS){
            actionCompleted = isCurrentReached(currentToOpen) && isOpened();
            
        } else if (action == MobileItemAction.CLOSELOADERHOOKS) {
            actionCompleted = this.position == this.positionToReach && isClosed();
            
        } else if (action == MobileItemAction.CLAMPLOADERHOOKS) {
            actionCompleted = clampedStatusSensor.isOn();    
            
        } else if (action == MobileItemAction.UNCLAMPLOADERHOOKS) {
            actionCompleted = this.position == this.positionToReach && !clampedStatusSensor.isOn();
        }
        
        return actionCompleted;
    }
    
    private boolean isCurrentReached(int currentToReach) {
        int DELTA = 5;
        FCSLOG.debug("readCurrent=" + readCurrent + " currentToReached=" + currentToReach);
        return currentToReach - DELTA <= this.readCurrent
                && this.readCurrent <= currentToReach + DELTA;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        try {
            loader.updateStateWithSensors();
            if (currentAction == MobileItemAction.CLOSELOADERHOOKS
                    || currentAction == MobileItemAction.UNCLAMPLOADERHOOKS) {
                position = this.hooksController.readPosition();
                
            } else if (currentAction == MobileItemAction.OPENLOADERHOOKS
                    || currentAction == MobileItemAction.CLAMPLOADERHOOKS) {
                this.readCurrent = this.hooksController.readCurrent();
                //just to display on the GUI the position of the clamp
                position = this.hooksController.readPosition();
            }
        } catch (CanOpenCallTimeoutException ex) {
            this.raiseWarning(CAN_BUS_TIMEOUT, " error in updateStateWithSensorsToCheckIfActionIsCompleted:" + ex, getName());
        }
    }

    @Override
    public void startAction(MobileItemAction action)  {
        
        if (action == MobileItemAction.OPENLOADERHOOKS) {
            hooksController.enableAndWriteCurrent((short)this.currentToOpen);
            
        } else if (action == MobileItemAction.CLOSELOADERHOOKS) {
            hooksController.enableAndWriteRelativePosition(this.relativePositionToClose);
            
        } else if (action == MobileItemAction.CLAMPLOADERHOOKS) {
            hooksController.enableAndWriteCurrent((short)this.currentToClamp);
        
        } else if (action == MobileItemAction.UNCLAMPLOADERHOOKS) {
            hooksController.enableAndWriteRelativePosition(this.relativePositionToUnclamp);
        }
    }
    
    @Override
    public void abortAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(name + " is ABORTING action " + action.toString()
                + " within delay " + delay);
        this.hooksController.stopAction();
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(name + " is STOPPING action " + action.toString()
                + " within delay " + delay);
        this.hooksController.stopAction();
    }

    @Override
    public void postAction(MobileItemAction action)  {

        //because we don't want to let the controller on power
        this.hooksController.stopAction();
        FCSLOG.info(name + ":" + action.toString() + " completed - doing postAction.");
        
        if (action == MobileItemAction.OPENLOADERHOOKS) {
            homing();
        } 
        this.publishData();
    }
    
    
    private void homing() {
        this.hooksController.defineAbsolutePosition(0);
        // for the GUI
        updatePosition(); 
        this.homingDone = true;
        this.publishData();
        loader.updateFCSStateToReady();
    }
    
    /**
     * reads sensors, computes and state and raises an ALERT if sensors are in error.
     *
     */
    public void updateStateAndCheckSensors() {
        loader.updateStateWithSensors();
        checkSensors(LO_SENSOR_ERROR);
    }


    /**
     * This methods updates the lockStatus from an array of hexaValues.
     *
     * @param readHexaValues
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    void updateStateWithSensors(int[] readHexaValues)  {
        lock.lock();
        try {
            updatingState = true;
            updateHooksSensorsValue(readHexaValues);
            updateForceStatus(readHexaValues);
            clampedStatusSensor.updateValue(readHexaValues);

            if (oneHookInError()) {
                this.lockStatus = LockStatus.ERROR;
                
            } else if (allHooksInState(LockStatus.OPENED)) {
                computeStatusHooksOpened();               
                
            } else if (allHooksInState(LockStatus.CLOSED)) {
                computeStatusHooksClosed();
                
            } else if (allHooksInState(LockStatus.INTRAVEL)) {
                this.lockStatus = LockStatus.INTRAVEL;
                
            } else {
                this.lockStatus = LockStatus.UNKNOWN;
            }

        } finally {
            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
        }
    }
    
    private void updateHooksSensorsValue(int[] hexaValues) {
        for (LoaderHookModule hook: hooks) {
            hook.updateStateWithSensors(hexaValues);
        }
    }
    
    private boolean oneHookInError() {
        boolean bool = false;
        for (LoaderHookModule hook: hooks) {
            bool = bool || hook.getLockStatus() == LockStatus.ERROR;
        }
        return bool;
    }
    
    private boolean allHooksInState(LockStatus status) {
        boolean bool = true;
        for (LoaderHookModule hook: hooks) {
            bool = bool && hook.getLockStatus() == status;
        }
        return bool;
    }
    
    private void updateForceStatus(int[] readHexaValues) {
        forceSensor0.updateForce(readHexaValues);
        forceSensor1.updateForce(readHexaValues);
        if (Math.abs(forceSensor0.getForce() - forceSensor1.getForce()) < 120 ) {
            forceStatus = forceSensor0.getForceStatus();            
        } else {
            forceStatus = LockStatus.ERROR;
        }
    }
    
    
    private void computeStatusHooksClosed() {
        if (clampedStatusSensor.isOn() && forceStatus == LockStatus.CLAMPED) {
            this.lockStatus = LockStatus.CLAMPED;

        } else if (forceStatus == LockStatus.UNCLAMPED) {
            this.lockStatus = LockStatus.CLOSED; 

        } else if (forceStatus == LockStatus.UNDER_LOAD) {
            this.lockStatus = LockStatus.UNDER_LOAD;

        } else {
            this.lockStatus = LockStatus.ERROR;
        }        
    }
    
    private void computeStatusHooksOpened() {
        /* hooks opened - no more load on hooks */
        if (forceStatus == LockStatus.UNCLAMPED) {
            this.lockStatus = LockStatus.OPENED; 
        
        /* hooks opened - forceSensor CLAMPED : weird */
        } else if (forceStatus == LockStatus.CLAMPED) {
            this.lockStatus =  LockStatus.ERROR;
                    
        /* hooks opened - too much load on hooks to be opened */    
        } else {
            this.lockStatus = forceStatus;
        }
    }


    /**
     * Updates the field readCurrent of the clamp in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update clamp current in reading controller.")
    public void updateCurrent() {
        try {
            this.readCurrent = hooksController.readCurrent();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:",ex);
        }
        this.publishData();
    }

    /**
     * Updates the position of the clamp in reading the CPU of the controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Update clamp position in reading controller.")
    public void updatePosition()  {
        try {
            this.position = hooksController.readPosition();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:",ex);
        }
        this.publishData();
    }
    
    
    /**
     * To display position for end user.
     * Updates carrier position in reading controller and returns it.
     * @return position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "To update and display position for end user."
                    + "Updates loader clamp position in reading controller and returns it.")
    public int readPosition() {
        updatePosition();
        return this.position;
    }
   
    /**
     * Return a printed list of hardware with initialization information.
     * For debug purpose.
     * @return 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware with initialization information.")    
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(name);
        if (homingDone) {
            sb.append(" is INITIALIZED.");
        } else {
            sb.append(" is NOT INITIALIZED.");
        }
        return sb.toString();
    }    
    
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "List and display clamp info.")
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("/timeoutForClosingHooks=");
        sb.append(this.timeoutForClosingHooks);
        sb.append("/timeoutForClosingHooksStrongly=");
        sb.append(this.timeoutForClampingHooks);
        sb.append("/timeoutForOpeningHooks=");
        sb.append(this.timeoutForOpeningHooks);
        return sb.toString();
    }


    /**
     * Creates and returns the object to be published on the STATUS bus by the LoaderClampModule.
     * @return 
     */
    public StatusDataPublishedByLoaderClamp createStatusDataPublishedByLoaderClamp() {
        StatusDataPublishedByLoaderClamp status = new StatusDataPublishedByLoaderClamp();
        status.setName(name);
        status.setPosition(position);
        status.setCurrent(readCurrent);
        status.setClampState(lockStatus);
        status.setForce0(forceSensor0.getForce());
        status.setForce1(forceSensor1.getForce());
        status.setForceStatusOn(clampedStatusSensor.isOn());
        status.setForceStatus(forceStatus);
        status.setHomingDone(homingDone);
        status.setStatusPublishedByHook1(hook1.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook2(hook2.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook3(hook3.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook4(hook4.createStatusDataPublishedByLoaderHook());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        KeyValueData kvd = new KeyValueData("loaderClamp", this.createStatusDataPublishedByLoaderClamp());
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }

}
