
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_TIMEOUT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This is a model for the carrier in the loader.
 *
 * @author virieux
 */
public class LoaderCarrierModule extends MobileItemModule implements MovedByEPOSController, ControlledBySensors {

    
    private final RedondantSensors handoffSensors;    
    private final RedondantSensors storageSensors;


    private int position = 0;
    private int ssiPosition = 0;
    private int absoluteTargetPosition = 0;
    
    @ConfigurationParameter(description="delta position in micron : used to know if carrier position "
            + "is in a range of 2*deltaPosition around a given position")
    private int deltaPosition = 50;
    
    @ConfigurationParameter(description="Loader Handoff position in micron")
    private int handoffPosition = 1037500;
    
    @ConfigurationParameter(description="Loader Storage position in micron")
    private int storagePosition = 0; 
    
    @ConfigurationParameter(description=
            "For the Loader GUI : Maximum current to be sent to the Loader Carrier controller (in mA).")
    private int maxCurrent = 1000;
        
    @ConfigurationParameter(description="For the Loader GUI : Loader Carrier Maximum speed (in rpm/mn).")
    private int maxSpeed = 227;

    @ConfigurationParameter(description="timeout in milliseconds to go from storage to handoff on loader")
    private long timeoutForGoingToHandOff = 120000;
    
    @ConfigurationParameter(description="timeout in milliseconds to go from handoff to storage on loader")
    private long timeoutForGoingToStorage = 120000;
    
    private int speed;
    private int current;

    @LookupField(strategy=Strategy.BYNAME)
    private EPOSController carrierController;
    
    @LookupField(strategy=Strategy.TREE)
    private LoaderModule loader;
    
    @LookupField(strategy=Strategy.BYNAME)
    private BridgeToHardware loaderTcpProxy;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    private volatile boolean initialized = false;

    /**
     * Buils a LoaderCarrierModule with 2 couples of redondants sensors to detect carrier position.
     * @param handoffSensors
     * @param storageSensors 
     */
    public LoaderCarrierModule(RedondantSensors handoffSensors, RedondantSensors storageSensors) {
        this.handoffSensors = handoffSensors;
        this.storageSensors = storageSensors;
    }
    
    

    /**
     * 
     * @return carrier position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns carrier position.")    
    public int getPosition() {
        return position;
    }

    /**
     * returns current for tests
     * @return 
     */
    protected int getCurrent() {
        return current;
    }
    
    

    /**
     * Returns true if carrierController is in Fault.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if carrierController is in Fault.")
    @Override
    public boolean isControllerInFault() {
        return carrierController.isInError();
    }
    
    /**
     * @return controllerName
     */
    @Override
    public String getControllerName() {
        return this.carrierController.getName();
    }

    /**
     * 
     * @return true if carrier is initialized and ready to receive commands.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if carrier is initialized and ready to receive commands.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * Returns max speed - Used by GUI
     * @return max speed in rpm/mn (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the max speed in rpm/mn (format decimal).")
    public int getMaxSpeed() {
        return maxSpeed;
    }

    /**
     * Returns max current - Used by GUI
     * @return max current in mA (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the max current in mA (format decimal).")
    public int getMaxCurrent() {
        return maxCurrent;
    }

    /**
     *
     * @return handoff position in microns (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the handoff position in microns (format decimal).")
    public int getHandoffPosition() {
        return handoffPosition;
    }

    /**
     *
     * @return storage position in microns (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the storage position in microns (format decimal).")
    public int getStoragePosition() {
        return storagePosition;
    }

    /**
     * Used by GUI
     * @return timeout for going to Handoff action in millis (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the timeout for going to Handoff in millis (format decimal).")
    public long getTimeoutForGoingToHandOff() {
        return timeoutForGoingToHandOff;
    }

    /**
     * Used by GUI
     * @return timeout for going to Storage action in millis (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the timeout for going to Storage in millis (format decimal).")
    public long getTimeoutForGoingToStorage() {
        return timeoutForGoingToStorage;
    }

    /**
     * 
     * @return false if the 2 redondant position sensors at Storage are equal
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Storage are equal.")
    public boolean isStorageSensorsInError() {
        return this.storageSensors.isInError();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at HandOff are equal.")
    public boolean isHandoffSensorsInError() {
        return handoffSensors.isInError();
    }

    /**
     * returns true if handoff sensors or storage sensors are in error.
     * @return 
     */
    @Override
    public boolean isInError() {
        return this.handoffSensors.isInError() || this.storageSensors.isInError();
    }

    
    public void initializeController() {
        try {
            carrierController.initializeAndCheckHardware();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + ":",ex);
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(FcsAlert.HARDWARE_ERROR, " could not initialize controller", name, ex);
        }
    }
    
    @Override
    public void postStart() {
        if (carrierController.isBooted()) {
            initializeController();
        } 
    }

    /**
     * Returns true if loader CANopen devices are booted, identified and initialized.
     * @return 
     */
    @Override
    public boolean myDevicesReady() {
        return loaderTcpProxy.allDevicesBooted();
    }

    /**
     * Returns the boolean field atStorage. If the atStorage boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * atStorage is updated. If the field atStorage is not being updated, it
     * returns immediatly the field atStorage.
     *
     * @return atStorage
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at storage position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtStoragePosition() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.",ex);
                }

            }
            return this.storageSensors.isOn();

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * atHandoff is updated. If the field atHandoff is not being updated, it
     * returns immediatly the field atHandoff.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at HANDOFF position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtHandoffPosition() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.",ex);
                }

            }
            return this.handoffSensors.isOn();

        } finally {
            lock.unlock();
        }
    }



    /**
     * Move the carrier to Handoff position.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            alias = "goToHandoff",
            description = "Move the carrier to Handoff position.")
    public void goToHandOff()  {
        if (this.isAtHandoffPosition()) {
            FCSLOG.info(name + " is already at Handoff position. Nothing to do.");
        } else {
            loader.checkConditionsForCarrierMotion();
            this.executeAction(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_HANDOFF,
                    timeoutForGoingToHandOff);
        }
    }

    /**
     * Moves loader carrier to storage position.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the carrier to STORAGE position.")
    public void goToStorage()  {
        if (this.isAtStoragePosition()) {
            FCSLOG.info(name + " is already at Storage position. Nothing to do.");
        } else {
            loader.checkConditionsForCarrierMotion();
            this.executeAction(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_STORAGE,
                    timeoutForGoingToStorage);
        }
    }
    
    /**
     * For end user.
     * Move the carrier to absolute position given as argument (in decimal format).
     * @param absolutePosition 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the carrier to absolute position given as argument (in decimal format).")    
    public void goToAbsolutePosition(int absolutePosition) {
        this.updatePosition();
        if (position == absolutePosition) {
            FCSLOG.info(name + " is already at position "+absolutePosition);
        } else {
            loader.checkConditionsForCarrierMotion();
            this.absoluteTargetPosition = absolutePosition;
            this.executeAction(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION,
                    timeoutForGoingToStorage);
        }
    }

    /**
     * Tests if action is completed. 
     * During motion of the carrier, this methods tests if the desired position has been reached.
     * If yes, it returns true, else false.
     * @param action to be tested
     * @return true if required position has been reached
     */
    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {

            case MOVE_LOADERCARRIER_TO_HANDOFF:
                return this.position == this.handoffPosition;

            case MOVE_LOADERCARRIER_TO_STORAGE:
                return this.position == this.storagePosition;
                
            case MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION:
                return this.position == this.absoluteTargetPosition;
                
            default:
                assert false;
        }
        return false;
    }

    /**
     * Updates the field position of the carrier in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update carrier position in reading controller.")
    public void updatePosition()  {
        this.position = carrierController.readPosition();
        this.publishData();
    }
    
    @Override
    public void updateCurrent() {
        this.current = this.carrierController.readCurrent();
    }

    /**
     * To display position for end user.
     * Updates carrier position in reading controller and returns it.
     * @return position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "To update and display position for end user."
                    + "Updates carrier position in reading controller and returns it.")
    public int readPosition() {
        updatePosition();
        return this.position;
    }


    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            carrierController.checkFault();
            this.position = this.carrierController.readPosition();
            updateCurrent();
            //for the GUI : is it the right place ?
            loader.updateStateWithSensors();
            this.speed = this.carrierController.readProfileVelocity();
        } catch (ShortResponseToSDORequestException | SDORequestException ex) {
            this.raiseWarning(SDO_ERROR,
                    "error in updateStateWithSensorsToCheckIfActionIsCompleted:" + ex,getName());
        } catch (CanOpenCallTimeoutException ex) {
            this.raiseWarning(CAN_BUS_TIMEOUT, 
                    " error in updateStateWithSensorsToCheckIfActionIsCompleted:" + ex,getName());
        }
    }


    /**
     * This methods updates the lockStatus from an array of hexaValues.
     *
     * @param readHexaValues
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    protected void updateStateWithSensors(int[] readHexaValues){
        lock.lock();

        try {
            updatingState = true;
            this.handoffSensors.updateValue(readHexaValues);
            this.storageSensors.updateValue(readHexaValues);

        } finally {
            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }

    }

    /**
     * Starts an action on the carrier.
     * On the loader carrier, the action in normal mode are motion of the carrier to go to STORAGE position 
     * or HANDOFF position.
     * In engineering mode, the end user can move the carrier to a given absolute position.
     * @param action to be started
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        carrierController.checkFault();

        if (action.equals(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_HANDOFF)) {
            carrierController.enableAndWriteAbsolutePosition(this.handoffPosition);

        } else if (action.equals(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_STORAGE)) {
            carrierController.enableAndWriteAbsolutePosition(this.storagePosition);

        } else if (action.equals(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION)) {
            carrierController.enableAndWriteAbsolutePosition(this.absoluteTargetPosition);
        }
    }
    
    
    /**
     * Aborts an action.
     * @param action
     * @param delay 
     */
    @Override
    public void abortAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(name + " is ABORTING action " + action.toString()
                + " within delay " + delay);
        this.carrierController.stopAction();
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(name + " is STOPPING action " + action.toString()
                + " within delay " + delay);
        //doesn't work : we can't do a quickStop when in PROFILE_POSITION
        //this.carrierController.quickStop();
        this.carrierController.stopPosition();

    }

    /**
     * Puts controller stopAction after an action and tests if the result of the action is correct.
     * The result of the action is correct if the position given by the controller is confirmed by the position sensors.
     * @param action 
     */
    @Override
    public void postAction(FcsEnumerations.MobileItemAction action)  {
        //because we don't want to let the controller on power
        this.carrierController.stopPosition();
        this.carrierController.disable();
        FCSLOG.info(name + ":" + action.toString() + " completed - doing postAction.");

        switch (action) {

            case MOVE_LOADERCARRIER_TO_STORAGE:
                if (!this.isAtStoragePosition()) {
                    throw new FailedCommandException(name + ": storage sensors don't confirm carrier position.");
                }
                break;

            case MOVE_LOADERCARRIER_TO_HANDOFF:
                if (!this.isAtHandoffPosition()) {
                    throw new FailedCommandException(name + ": handoff sensors don't confirm carrier position.");
                }
                break;
                
            case MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION:
                this.absoluteTargetPosition = 0;
                break;    
                
            default:
                assert false;
        }

    }

    /**
     * Command for ENGINEERING mode. Not used during INITIALIZATION.
     * Check that carrier controller is initialized.
     * Read sensors and update state.
     * Read carrier position on controller.
     * If position sensors and position read on carrier controller give inconsistent information
     * throw a FcsHardwareException.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Check if hardware is ready to be started.")
    public void initializeHardware()  {
        carrierController.checkInitialized();

        loader.updateStateWithSensors();
        try {
            updatePosition();
        } catch (SDORequestException ex) {
            String msg = getName() + ": couldn't update position";
            FCSLOG.error(msg);
            throw new FcsHardwareException(name, ex);
        }
        if (this.isAtHandoffPosition() && isCarrierAlmostAtPosition(this.handoffPosition)) {
            throw new FcsHardwareException(name
                    + ": handoff sensors don't confirm position read on carrierController.");
        }
        if (this.isAtStoragePosition() && isCarrierAlmostAtPosition(this.storagePosition)) {
            throw new FcsHardwareException(name
                    + ": storage sensors don't confirm position read on carrierController.");
        }
        this.initialized = true;
        loader.updateFCSStateToReady();
    }
    
    /**
     * Returns true if carrier position is in a range of 2*deltaPosition around a given position.
     * @param position
     * @return 
     */
    private boolean isCarrierAlmostAtPosition(int position) {
        return this.position < position - deltaPosition || this.position > position + deltaPosition;
    }
    
    /**
     * Return a printed list of hardware with initialization information.
     * For debug purpose.
     * @return 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware with initialization information.")    
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(name);
        if (this.isInitialized()) {
            sb.append(" is INITIALIZED.");
        } else {
            sb.append(" is NOT INITIALIZED.");
        }
        return sb.toString();
    }
    
    /**
     * Creates an object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByLoaderCarrier createStatusDataPublishedByLoaderCarrier() {
        StatusDataPublishedByLoaderCarrier status = new StatusDataPublishedByLoaderCarrier();
        status.setName(name);
        status.setPosition(position);
        status.setSsiPosition(ssiPosition);
        status.setSpeed(speed);
        status.setCurrent(current);
        status.setHandoffSensorValue0(handoffSensors.isOn());
        status.setStorageSensorValue0(storageSensors.isOn());
        status.setHandoffInError(handoffSensors.isInError());
        status.setStorageInError(storageSensors.isInError());
        status.setControllerInError(carrierController.isInError());
//        status.setLimitSwitchDownInError(limitSwitchDownInError);
//        status.setLimitSwitchUpInError(limitSwitchUpInError);
        status.setHomingDone(initialized);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        StatusDataPublishedByLoaderCarrier status = this.createStatusDataPublishedByLoaderCarrier();
        KeyValueData kvd = new KeyValueData("loaderCarrier", status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }





}
