package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.ClearAlertHandler;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CA_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.ClampsOrLatchesDisagreeException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This is a socket on the carousel : there is 5 sockets on a carousel. When a
 * filter is on the carousel, it is attached at a socket. A socket has a clamp
 * on each side : - clampX- (clampXminus) - clampX+ (clampXplus) clampX+ and
 * clampX- are the same except for the method isLocked() See
 * FilterClampXminusModule and FilterClampXplusModule
 *
 * The main job of this class CarouselSocket is to synchronize the actions on
 * the 2 clamps. The state of the CarouselSocket is the state of the clamps when
 * both clamps state are identical. 
 * 
 *
 * During an action on both clamps, we want to lock the CarouselSocket so no
 * other thread can try to access this object before the completion of action.
 * We do that with a Lock and Condition. (see java.util.concurrent).
 *
 * CarouselSocket extends MobileItemModule because we want to lock or release
 * the clamps when the carousel is at standby position and we want to
 * wait until the action is completed. We know if the action is completed in
 * reading the clamps sensors. So we use the general mechanism provided by the
 * MobileItemModule.
 *
 * this class raises ALERT FCS004 in method
 * updateStateWithSensorsToCheckIfActionIsCompleted
 *
 * @author virieux
 *
 */
public class CarouselSocket extends MobileItemModule implements ClearAlertHandler {

    @LookupField(strategy = Strategy.TREE)
    private MainModule mainModule;

    @LookupField(strategy = Strategy.TREE)
    private CarouselModule carousel;

    @LookupField(strategy = Strategy.BYNAME)
    private BridgeToHardware tcpProxy;

    private CarouselClampModule clampXminus;

    private CarouselClampModule clampXplus;

    @LookupField(strategy = Strategy.BYNAME)
    private EPOSController clampXminusController;
    
    @LookupField(strategy = Strategy.BYNAME)
    private EPOSController clampXplusController;

    /*To be able to know if the autochanger holds a filter. */
    /*This can't be a AutochangerModule because when the carousel is in Standalone mode
     /*there is not autochanger.
     /*FilterHolder is an Interface. */
    @LookupField(strategy = Strategy.BYNAME)
    private FilterHolder autochanger;


    /**
     * carousel positionOnCarousel (angle) when this socket is at standby
     * standbyPosition should be : 360 - positionOnCarousel, but could be
     * different
     */
    @ConfigurationParameter(range = "-4500000..4500000")
    private int standbyPosition;

    /**
     * When the socket holds a filter, this field is initialized whith this
     * filter. When the socket is empty, this field is null.
     */
    private Filter filter;

    private volatile boolean updatingClamps = false;
    private volatile FilterClampState clampsState = FilterClampState.UNDEFINED;

    //When we update the 2 clamps, we want to take a lock on the socket until
    //the end of the update, so any other thread has to wait for this update.
    private final Condition stateUpdated = lock.newCondition();

    //TODO : there is also a timeout for each clamp, it's a lot of timeouts !
    //here we could take the maximum of the timeout of the clampXminus and clampXplus.
    //this is done in initModule 
    private long timeoutForUnlocking = 4000;
    private long timeoutForReleasing = 4000;

    /**
     * Build a new CarouselSocket with a tickMillis of 5000.
     *
     * @param clampXminus
     * @param clampXplus
     * @param standbyPosition
     */
    public CarouselSocket(
            CarouselClampModule clampXminus, CarouselClampModule clampXplus,
            int standbyPosition) {
        this.clampXminus = clampXminus;
        this.clampXplus = clampXplus;
        this.standbyPosition = standbyPosition;
    }

    public CarouselClampModule getClampXminus() {
        return clampXminus;
    }

    public void setClampXminus(CarouselClampModule clampXminus) {
        this.clampXminus = clampXminus;
    }

    public CarouselClampModule getClampXplus() {
        return clampXplus;
    }

    public void setClampXplus(CarouselClampModule clampXplus) {
        this.clampXplus = clampXplus;
    }

    public Filter getFilter() {
        return filter;
    }

    private void setFilter(Filter filter) {
        this.filter = filter;
    }

    public int getStandbyPosition() {
        return standbyPosition;
    }

    /**
     * Return true if this socket is at STANDBY position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if this socket is at STANDBY position on the carousel.")
    public boolean isAtStandby() {
        return Math.abs(standbyPosition - carousel.getPosition()) <= 1000;
    }

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** END OF SETTERS AND GETTERS
     * **********************************************
     */
    /**
     * ***********************************************************************************************
     */
    @Override
    public void init() {
        this.timeoutForReleasing = java.lang.Math.max(this.clampXminus.timeoutForReleasing,
                this.clampXplus.timeoutForReleasing);
        this.timeoutForUnlocking = java.lang.Math.max(this.clampXminus.timeoutForUnlocking,
                this.clampXplus.timeoutForUnlocking);

        this.clampXminus.setController(clampXminusController);
        this.clampXplus.setController(clampXplusController);

    }

    /**
     * Checking if autochanger is holding the filter is delegated to
     * autochanger.
     *
     * @return true if autochanger is at STANDBY and holds a filter.
     */
    private boolean isAutochangerHoldingFilter() {
        return autochanger.isAtStandby() && autochanger.isHoldingFilter();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return mainModule.allDevicesBooted();
    }

    /**
     * Returns the state of the clamps. If the state is being updated and
     * waiting for a response from a sensor, this methods waits until the state
     * is updated. If the state is not being updated, it returns immediatly the
     * state.
     *
     * @return state
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return clamps state - doesn't read again sensors.")
    public FilterClampState getClampsState() {

        lock.lock();
        try {
            while (updatingClamps) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(name + " was interrupted during getClampsState.");
                }

            }
            return clampsState;

        } finally {
            lock.unlock();
        }
    }

    /**
     * This method updates the state of the 2 clamps in reading the values sent
     * by the sensors. The values of the sensors are read clamp by clamp with
     * read SDO commands.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Deprecated
    public void updateClampsStateWithSensorsFromSDO() {

        lock.lock();
        this.updatingClamps = true;
        try {
            clampXminus.updateStateWithSensorsFromSDO();
            clampXplus.updateStateWithSensorsFromSDO();

            if (clampXminus.getClampState() == clampXplus.getClampState()) {
                this.clampsState = clampXminus.getClampState();
                //System.out.println("XXXXXXXXSocket at standby : clamps are updated");

            } else {
                throw new FcsHardwareException("Error in filter presence detection at standby position : "
                        + "the clamps don't agree.");
            }

        } finally {
            this.updatingClamps = false;
            this.stateUpdated.signal();
            lock.unlock();
        }
    }

    public void updateClampsStateWithSensors(PDOStorage pdoStorage) {
        lock.lock();
        this.updatingClamps = true;
        try {
            clampXminus.updateStateWithSensors(pdoStorage);
            clampXplus.updateStateWithSensors(pdoStorage);

            if (clampXminus.getClampState() == clampXplus.getClampState()) {
                this.clampsState = clampXminus.getClampState();
                FCSLOG.info(name + " : clamps are updated");

            } else {
                //TODO put here the clampName
                String msg = name + ":Error in carousel at standby position : "
                        + "\nclampState at Xminus side is different from clampState at Xplus side.";
                msg = msg + clampXminus.getName() + " state=" + clampXminus.getClampState();
                msg = msg + clampXplus.getName() + " state=" + clampXplus.getClampState();
                this.raiseAlarm(CA_SENSOR_ERROR, msg);
                throw new ClampsOrLatchesDisagreeException(msg);
            }

        } finally {
            this.updatingClamps = false;
            this.stateUpdated.signal();
            lock.unlock();
        }
        publishData();
    }

    /**
     * Update clamps state in reading sensors.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Update clamps state in reading sensors.")
    public void updateClampsStateWithSensors() {

        FCSLOG.debug("updateClampsStateWithSensors/" + name + "/bridge=" + this.tcpProxy.toString());
        updateClampsStateWithSensors(this.tcpProxy.readPDOs());
    }

    /*
     * The carousel socket is empty if there is no filter in the socket.
     * That appends when clamps of both side of the socket are empty.
     * A clamp is empty when no filter is engaged in the clamp.
     * If the 2 clamps are not in the same state this method throws
     * a FcsHardwareException : this should not append except if 
     * the sensors are broken.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns true if there is no filter in the socket.")
    public boolean isEmpty() {
        return clampsState == FilterClampState.READYTOCLAMP
                || clampsState == FilterClampState.UNCLAMPEDEMPTY;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns true if there is a filter in the socket and the clamps"
            + "are LOCKED.")
    public boolean isClampedOnFilter() {
        return clampsState == FilterClampState.CLAMPEDONFILTER;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns true if there is a filter in the socket and the clamps"
            + "are UNLOCKED.")
    public boolean isUnclampedOnFilter() {
        return clampsState == FilterClampState.UNCLAMPEDONFILTER;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns true if there is NO filter in the socket and the clamps"
            + "are UNLOCKED.")
    public boolean isUnclampedEmpty() {
        return clampsState == FilterClampState.UNCLAMPEDEMPTY;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns true if this socket is ready to clamp a filter.")
    public boolean isReadyToClamp() {
        return clampsState == FilterClampState.READYTOCLAMP;
    }

    /**
     * Releases the 2 clamps of the socket.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Release the 2 clamps of this socket if the socket is at STANDBY position.")
    public void releaseClamps() {
        updateClampsStateWithSensors();

        FCSLOG.info("Checking conditions for release clamp " + name + " on socket "
                + "at standby position.");
        if (!this.isAtStandby()) {
            throw new RejectedCommandException(name + " is NOT AT STANDBY - can't unlock clamps.");
        }

        if (clampsState != FilterClampState.UNCLAMPEDEMPTY) {
            FCSLOG.fine(name + " clampsState=" + clampsState);
            FCSLOG.fine(this.clampXminus.getName() + " clampState=" + this.clampXminus.getClampState());
            FCSLOG.finest(this.clampXminus.getName() + " lockStatus=" + this.clampXminus.getLockStatus());
            FCSLOG.finest(this.clampXminus.getName() + " filterPresenceStatus=" + this.clampXminus.getFilterPresenceStatus());

            FCSLOG.fine(this.clampXplus.getName() + " clampState=" + this.clampXplus.getClampState());
            FCSLOG.finest(this.clampXplus.getName() + " lockStatus=" + this.clampXplus.getLockStatus());
            FCSLOG.finest(this.clampXplus.getName() + " filterPresenceStatus=" + this.clampXplus.getFilterPresenceStatus());
            throw new RejectedCommandException(name + ": Can't release clamps if socket is not UNCLAMPED and EMPTY."
                    + " ");
        }

        FCSLOG.info(name + ": Releasing clamps at standby position.");
        this.executeAction(MobileItemAction.RELEASECLAMPS, timeoutForReleasing);
    }

    /**
     * This method unclamps the 2 clamps at standby position.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Unlock the 2 clamps of this socket if the socket is at STANDBY position.")
    public void unlockClamps() {

        if (!this.isAtStandby()) {
            throw new RejectedCommandException(name + " is NOT AT STANDBY - can't unlock clamps.");
        }

        updateClampsStateWithSensors();
        if (clampsState != FilterClampState.CLAMPEDONFILTER) {
            throw new RejectedCommandException(name + ":Can't unlock clamps if isn't clamped on filter.");
        }

        if (!this.isAutochangerHoldingFilter()) {
            throw new RejectedCommandException("CANNOT UNLOCK CLAMPS if FILTER is not HELD by autochanger.");
        }

        this.executeAction(MobileItemAction.UNLOCKCLAMPS, timeoutForUnlocking);
    }

    /**
     * Software removing. Removes a filter from carousel. This method has no
     * action on hardware. It updates the objects Socket and Filter. A filter
     * can be removed from Carousel only when the socketAtStandby is at standby
     * filterPosition.
     *
     */
    public void removeFilter() {
        //just for the GUI
        this.filter.removeFromCarousel();

        this.filter = null;
        this.publishData();
    }

    /**
     * This is only a software operation. It updates the filter when a filter is
     * seen by the sensors.
     *
     * @param filter
     */
    public synchronized void putFilterOnSocket(Filter filter) {
        setFilter(filter);
        this.publishData();
        filter.putFilterOnSocket(name);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns a String representation of a CarouselSocket.")
    @Override
    public synchronized String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append(",Filter in socket : ");
        if (filter == null) {
            sb.append(" NO FILTER").append("\n");
        } else {
            sb.append(filter.getName()).append("\n");
        }
        return sb.toString();
    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action == MobileItemAction.UNLOCKCLAMPS) {
            return this.clampsState == FilterClampState.UNCLAMPEDONFILTER;

        } else if (action == MobileItemAction.RELEASECLAMPS) {
            return this.clampsState == FilterClampState.READYTOCLAMP;

        } else {
            throw new IllegalArgumentException("Action on clamps on socket must "
                    + "be UNLOCKCLAMPS or RELEASECLAMPS");
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            this.updateClampsStateWithSensors();
        } catch (ClampsOrLatchesDisagreeException ex) {
            String msg = name + " a little delai between the update of the 2 clamps : have to wait ...";
            this.raiseWarning("FCS004", msg, ex);
        }
    }

    /**
     * Start action of unlocking or releasing clamps.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void startAction(MobileItemAction action) {
        if (action == MobileItemAction.UNLOCKCLAMPS) {
            if (!this.clampXminusController.isEnabled()) {
                this.clampXminusController.enable();
            }
            if (!this.clampXplusController.isEnabled()) {
                this.clampXplusController.enable();
            }
            if (!clampXminusController.isEnabled()) {
                throw new RejectedCommandException(name
                        + " ControllerXminus has to be enabled first.");
            }
            if (!clampXplusController.isEnabled()) {
                throw new RejectedCommandException(name
                        + " ControllerXplus has to be enabled first.");
            }
            this.clampXminusController.writeCurrent(this.clampXminus.getCurrentToUnlock());
            this.clampXplusController.writeCurrent(this.clampXplus.getCurrentToUnlock());

        } else if (action == MobileItemAction.RELEASECLAMPS) {
            this.clampXminusController.stopAction();
            this.clampXplusController.stopAction();
        } else {
            throw new IllegalArgumentException("Action on clamps on socket must be UNLOCKCLAMPS or RELEASECLAMPS");
        }
    }

    @Override
    public void postAction(MobileItemAction action) {
        if (action == MobileItemAction.UNLOCKCLAMPS) {
            FCSLOG.info("Just about to remove filter from carousel (software update)");
            this.removeFilter();
            FCSLOG.info("Command unlockClamps completed");
        }
        this.publishData();
    }

    @Override
    public void abortAction(MobileItemAction action, long delay) {
        //TODO complete this
        FCSLOG.info(name + " stopAction : nothing to be done.");
    }

    @Override
    public void publishData() {
        getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(name,
                createStatusDataPublishedByCarouselSocket()));
    }

    /**
     * Create an object to be published on the STATUS bus by a CarouselSocket.
     *
     * @return
     */
    public StatusDataPublishedByCarouselSocket createStatusDataPublishedByCarouselSocket() {
        StatusDataPublishedByCarouselSocket status = new StatusDataPublishedByCarouselSocket();
        status.setAtStandby(this.isAtStandby());
        status.setClampsState(clampsState);
        status.setEmpty(this.isEmpty());
        if (this.filter == null) {
            status.setFilterName(NO_FILTER);
        } else {
            status.setFilterName(this.filter.getName());
        }
        return status;
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert) {
        switch (alert.getAlertId()) {

            case "FCS004":
                if (clampXminus.getClampState() == clampXplus.getClampState()) {
                    return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
                } else {
                    return ClearAlertHandler.ClearAlertCode.DONT_CLEAR_ALERT;
                }

            default:
                return ClearAlertHandler.ClearAlertCode.UNKWNOWN_ALERT;
        }
    }

}
