
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Observable;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SENSOR14BITS_MAX;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SENSOR14BITS_MIN;

/**
 * This class simulate a filter presence sensor for the carousel.
 * In the carousel the new values of the filter presence sensor are read from a PDO.
 * @author virieux
 */
public class SimuCarouselClampFilterPresenceSensor extends SimuCanOpenSensor14bits {
    
    
    /**
     * Create a new simulated filter presence sensor for the Carousel.
     * @param adcName
     * @param numOfAnalogInput 
     */
    public SimuCarouselClampFilterPresenceSensor( 
            String adcName, int numOfAnalogInput) {
        super(adcName, numOfAnalogInput);
    }
    
    @Override
    public void initModule() {
        super.initModule();
        simulateFilterIsLockable();

    }
    
    
    /**
     * We observe SimuAutochangerPlutoGateway.
     * When it changes trucks position we are notified, we can then simulate that carousel is holding filter or not.
     * @param source
     * @param v 
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        FCSLOG.finest(getName()+" :old Value=" + this.printValue());
        if (this.isAtStandby()) {
            FCSLOG.finest(getName() + " is AT STANDBY.");
        }
        if (this.getSubsystem().isInState(PhaseState.OPERATIONAL) && this.isAtStandby()
                && autochanger.isHoldingFilter()) {
            if ((Integer) v.getValue() > 500) {
                /**
                 * filter is on autochanger trucks.
                 * So if position is not 0, the clamp filter presence sensor doesn't detect presence filter.
                 */
                simulateNoFilter();
                
            } else if ((Integer) v.getValue() == 0) {
                simulateFilterIsLockable();
            }
        }
        FCSLOG.finest(getName()+" :new Value=" + this.printValue());
    }
    
    /**
     * Simulates the filter is seen by the sensor.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsLockable() {
        this.setNewRange(clamp.getFilterPresenceValueA(), clamp.getFilterPresenceValueB());
        this.updateFakePDOStorage();
    }

    /**
     * Simulates the filter is seen by the sensor but is not Lockable.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsNotLockable() {
        this.setNewRange(clamp.getFilterPresenceValueB(), clamp.getFilterPresenceValueC());
        this.updateFakePDOStorage();
    }

    /**
     * Simulates no filter is seen by the sensor.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateNoFilter() {
        this.setNewRange(clamp.getFilterPresenceValueC(), SENSOR14BITS_MAX);
        this.updateFakePDOStorage();
    }

    /**
     * Simulates the sensor is in error.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsInError() {
        this.setNewRange(SENSOR14BITS_MIN, clamp.getFilterPresenceValueA());
        this.updateFakePDOStorage();
    }

}
