
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.AutoChangerModule;
import org.lsst.ccs.subsystems.fcs.AutoChangerTwoTrucksModule;
import org.lsst.ccs.subsystems.fcs.AutochangerTruckModule;
import org.lsst.ccs.subsystems.fcs.CanOpenNumericSensor;
import org.lsst.ccs.subsystems.fcs.CarouselModule;
import org.lsst.ccs.subsystems.fcs.ComplementarySensors;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.FilterIdentificator;
import org.lsst.ccs.subsystems.fcs.FilterManager;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;

/**
 * To simulate the plutoGateway on the autochanger.
 * @author virieux
 */
public class SimuAutochangerPlutoGateway extends SimuPlutoGateway implements SimuFilterIDPlutoGatewayInterface {
    
    /* To read filterID hall effect sensors.*/
    private FilterIdentificator filterIdentificator;
    private FilterHolder carousel;

    private AutoChangerModule autochanger;
    //trucks controllers
    private AutoChangerTwoTrucksModule trucks;
    private SimuAutochangerLinearRailController trucksController;
    private SimuAutochangerLinearRailController trucksSlaveController;

    //latches controllers
    private SimuAutochangerLatchController latchXminusController;
    private SimuAutochangerLatchController latchXplusController;

    //clamps controllers
    private SimuAutochangerOnlineClampController onlineClampXminusController;
    private SimuAutochangerOnlineClampController onlineClampXplusController;
    private SimuAutochangerOnlineClampController onlineClampYminusController;

    //trucks sensors
    private AutochangerTruckModule acTruckXminus;
    private AutochangerTruckModule acTruckXplus;
    
    //loader connected sensors 
    private CanOpenNumericSensor loaderConnectedSensor0;
    private CanOpenNumericSensor loaderConnectedSensor1;

    //latches sensors
    private ComplementarySensors lockSensorsLatchXminus;
    private ComplementarySensors lockSensorsLatchXplus;
    private ComplementarySensors unlockSensorsLatchXminus;
    private ComplementarySensors unlockSensorsLatchXplus;
    private ComplementarySensors filterEngagedSensorsLatchXminus;
    private ComplementarySensors filterEngagedSensorsLatchXplus;


    //clamps sensors
    private CanOpenNumericSensor onlineClampXminusCloseSensor;
    private CanOpenNumericSensor onlineClampXminusCloseSensorC;
    private CanOpenNumericSensor onlineClampXminusOpenSensor;
    private CanOpenNumericSensor onlineClampXminusOpenSensorC;
    private CanOpenNumericSensor onlineClampXplusCloseSensor;
    private CanOpenNumericSensor onlineClampXplusCloseSensorC;
    private CanOpenNumericSensor onlineClampXplusOpenSensor;
    private CanOpenNumericSensor onlineClampXplusOpenSensorC;
    private CanOpenNumericSensor onlineClampYminusCloseSensor;
    private CanOpenNumericSensor onlineClampYminusCloseSensorC;
    private CanOpenNumericSensor onlineClampYminusOpenSensor;
    private CanOpenNumericSensor onlineClampYminusOpenSensorC;
    

    /**
     * Build a new SimuAutochangerPlutoGateway
     * @param nodeID
     * @param serialNB 
     */
    public SimuAutochangerPlutoGateway(
            String nodeID, String serialNB) {
        super(nodeID, serialNB);
    }
    
    @Override
    public List<CanOpenNumericSensor> getFilterIDSensors() {
        return filterIdentificator.getFilterIDSensors();
    }

    @Override
    public void initModule() {
        super.initModule();
        ComponentLookup lookup = getComponentLookup();
        filterIdentificator = (FilterIdentificator) lookup.getComponentByName("filterIdentificator");
        carousel = (FilterHolder) lookup.getComponentByName("carousel");
        //trucks
        this.acTruckXminus = (AutochangerTruckModule) lookup.getComponentByName("acTruckXminus");
        this.acTruckXplus = (AutochangerTruckModule) lookup.getComponentByName("acTruckXplus");
        this.loaderConnectedSensor0 = (CanOpenNumericSensor) lookup.getComponentByName("loaderConnectedSensor0");
        this.loaderConnectedSensor1 = (CanOpenNumericSensor) lookup.getComponentByName("loaderConnectedSensor1");
        this.trucksController = (SimuAutochangerLinearRailController) lookup.getComponentByName("linearRailMasterController");
        this.trucksSlaveController = (SimuAutochangerLinearRailController) lookup.getComponentByName("linearRailSlaveController");
        this.trucks = (AutoChangerTwoTrucksModule) lookup.getComponentByName("autochangerTrucks");

        //latches
        this.lockSensorsLatchXminus = (ComplementarySensors) lookup.getComponentByName("lockSensorsLatchXminus");
        this.unlockSensorsLatchXminus = (ComplementarySensors) lookup.getComponentByName("unlockSensorsLatchXminus");
        this.filterEngagedSensorsLatchXminus = (ComplementarySensors) lookup.getComponentByName("filterEngagedSensorsLatchXminus");
        this.lockSensorsLatchXplus = (ComplementarySensors) lookup.getComponentByName("lockSensorsLatchXplus");
        this.unlockSensorsLatchXplus = (ComplementarySensors) lookup.getComponentByName("unlockSensorsLatchXplus");
        this.filterEngagedSensorsLatchXplus = (ComplementarySensors) lookup.getComponentByName("filterEngagedSensorsLatchXplus");


        
        this.latchXminusController = (SimuAutochangerLatchController) lookup.getComponentByName("latchXminusController");
        this.latchXplusController = (SimuAutochangerLatchController) lookup.getComponentByName("latchXplusController");
        

        //ONLINE clamps
        this.onlineClampXminusCloseSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXminusCloseSensor");
        this.onlineClampXminusCloseSensorC = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXminusCloseSensorC");
        this.onlineClampXminusOpenSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXminusOpenSensor");
        this.onlineClampXminusOpenSensorC = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXminusOpenSensorC");
        this.onlineClampXplusCloseSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXplusCloseSensor");
        this.onlineClampXplusCloseSensorC = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXplusCloseSensorC");
        this.onlineClampXplusOpenSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXplusOpenSensor");
        this.onlineClampXplusOpenSensorC = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXplusOpenSensorC");
        this.onlineClampYminusCloseSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampYminusCloseSensor");
        this.onlineClampYminusCloseSensorC = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampYminusCloseSensorC");
        this.onlineClampYminusOpenSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampYminusOpenSensor");
        this.onlineClampYminusOpenSensorC = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampYminusOpenSensorC");

        simulateAutochangerTrucksIsAtHandoff();
        simulateAutochangerIsEmpty();
        simulateAutochangerLatchesAreUnlocked();
        simulateAutochangerOnlineClampsAreUnlocked();
    }
    
    protected void simuComplementarySensorsOn(ComplementarySensors sensors) {
        replaceDigitalValues((CanOpenNumericSensor) sensors.getSensor(), "1");
        replaceDigitalValues((CanOpenNumericSensor) sensors.getSensorC(), "0");
    }
    
    protected void simuComplementarySensorsNotOn(ComplementarySensors sensors) {
        replaceDigitalValues((CanOpenNumericSensor) sensors.getSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) sensors.getSensorC(), "1");
    }
    
    protected void simuComplementarySensorsInError(ComplementarySensors sensors) {
        replaceDigitalValues((CanOpenNumericSensor) sensors.getSensor(), "1");
        replaceDigitalValues((CanOpenNumericSensor) sensors.getSensorC(), "1");
    }
    
    
    protected void simulateACTruckIsAtHandoff(AutochangerTruckModule truck) {
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensor(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensorB(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensorB(), "1");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksIsAtHandoff() {
        FCSLOG.info("simulateAutochangerTrucksIsAtHandoff");
        simulateACTruckIsAtHandoff(acTruckXminus);
        simulateACTruckIsAtHandoff(acTruckXplus);
        trucksController.setPosition(trucks.getHandoffPosition());
        trucksController.setSsiPosition(trucks.getEncoderRibbonMinValue()
                + trucks.getHandoffPosition());
        trucksSlaveController.setPosition(trucks.getHandoffPosition());
        trucksSlaveController.setSsiPosition(trucks.getEncoderRibbonMinValue()
                + trucks.getHandoffPosition()); 
        setChanged();
        this.notifyObservers(new ValueUpdate(name,trucks.getHandoffPosition()));
    }
    
    private void simulateACTruckIsOnline(AutochangerTruckModule truck) {
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensor(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensorB(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensorB(), "1");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksIsOnline() {
        simulateACTruckIsOnline(acTruckXminus);
        simulateACTruckIsOnline(acTruckXplus);
        trucksController.setPosition(trucks.getOnlinePosition());
        trucksController.setSsiPosition(trucks.getEncoderRibbonMinValue()
                + trucks.getOnlinePosition());
        trucksSlaveController.setPosition(trucks.getOnlinePosition());
        trucksSlaveController.setSsiPosition(trucks.getEncoderRibbonMinValue()
                + trucks.getOnlinePosition());   
        setChanged();
        this.notifyObservers(new ValueUpdate(name,trucks.getOnlinePosition()));
    }
    
    private void simulateACTruckIsAtStandby(AutochangerTruckModule truck) {
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensor(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensorB(), "0");        
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksIsAtStandby() {
        simulateACTruckIsAtStandby(acTruckXminus);
        simulateACTruckIsAtStandby(acTruckXplus);
        trucksController.setSsiPosition(trucks.getEncoderRibbonMinValue());
        trucksController.setPosition(trucks.getStandbyPosition());
        trucksSlaveController.setSsiPosition(trucks.getEncoderRibbonMinValue());
        trucksSlaveController.setPosition(trucks.getStandbyPosition());        
        setChanged();
        this.notifyObservers(new ValueUpdate(name,trucks.getStandbyPosition()));
        simulateFilterIDAtStandby();
    }
    
    private void simulateACTruckIsInTravel(AutochangerTruckModule truck) {
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensorB(), "1"); 
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksIsInTravel() {
        simulateACTruckIsInTravel(acTruckXminus);
        simulateACTruckIsInTravel(acTruckXplus);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsConnected() {
        replaceDigitalValues(loaderConnectedSensor0, "1");
        replaceDigitalValues(loaderConnectedSensor1, "0");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsDisconnected() {
        replaceDigitalValues(loaderConnectedSensor0, "0");
        replaceDigitalValues(loaderConnectedSensor1, "1");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsOnAutochanger() {
        simuComplementarySensorsOn(filterEngagedSensorsLatchXminus);
        simuComplementarySensorsOn(filterEngagedSensorsLatchXplus);        
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerIsEmpty() {
        simuComplementarySensorsNotOn(filterEngagedSensorsLatchXminus);
        simuComplementarySensorsNotOn(filterEngagedSensorsLatchXplus);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchesAreLocked() {
        simuComplementarySensorsOn(lockSensorsLatchXminus);
        simuComplementarySensorsOn(lockSensorsLatchXplus);
        simuComplementarySensorsNotOn(unlockSensorsLatchXminus);
        simuComplementarySensorsNotOn(unlockSensorsLatchXplus);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchesAreUnlocked() {
        simuComplementarySensorsNotOn(lockSensorsLatchXminus);
        simuComplementarySensorsNotOn(lockSensorsLatchXplus);
        simuComplementarySensorsOn(unlockSensorsLatchXminus);
        simuComplementarySensorsOn(unlockSensorsLatchXplus);        
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchIsLocked(String latchName) {
        if (latchName.contains("Xminus")) {
            simuComplementarySensorsOn(lockSensorsLatchXminus);
            simuComplementarySensorsNotOn(unlockSensorsLatchXminus);
        } else if (latchName.contains("Xplus")) {
            simuComplementarySensorsOn(lockSensorsLatchXplus);
            simuComplementarySensorsNotOn(unlockSensorsLatchXplus);            
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchIsUnlocked(String latchName) {
        if (latchName.contains("Xminus")) {
            simuComplementarySensorsNotOn(lockSensorsLatchXminus);
            simuComplementarySensorsOn(unlockSensorsLatchXminus);
        } else if (latchName.contains("Xplus")) {
            simuComplementarySensorsNotOn(lockSensorsLatchXplus);
            simuComplementarySensorsOn(unlockSensorsLatchXplus);           
        }
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLatchLockSensorsInError(String latchName) {
        if (latchName.contains("Xminus")) {
            simuComplementarySensorsInError(lockSensorsLatchXminus);
        } else if (latchName.contains("Xplus")) {
            simuComplementarySensorsInError(lockSensorsLatchXplus);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampsAreLocked() {
        simulateAutochangerOnlineClampXminusIsLocked();
        simulateAutochangerOnlineClampXplusIsLocked();
        simulateAutochangerOnlineClampYminusIsLocked();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampsAreUnlocked() {
        simulateAutochangerOnlineClampXminusIsUnlocked();
        simulateAutochangerOnlineClampXplusIsUnlocked();
        simulateAutochangerOnlineClampYminusIsUnlocked();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampIsLocked(CanOpenNumericSensor lockSensor,
            CanOpenNumericSensor lockSensorC,
            CanOpenNumericSensor unlockSensor,
            CanOpenNumericSensor unlockSensorC) {
        replaceDigitalValues(lockSensor, "1");
        replaceDigitalValues(lockSensorC, "0");
        replaceDigitalValues(unlockSensor, "0");
        replaceDigitalValues(unlockSensorC, "1");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampIsUnlocked(CanOpenNumericSensor lockSensor,
            CanOpenNumericSensor lockSensorC,
            CanOpenNumericSensor unlockSensor,
            CanOpenNumericSensor unlockSensorC) {
        replaceDigitalValues(lockSensor, "0");
        replaceDigitalValues(lockSensorC, "1");
        replaceDigitalValues(unlockSensor, "1");
        replaceDigitalValues(unlockSensorC, "0");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampXminusIsLocked() {
        simulateAutochangerOnlineClampIsLocked(onlineClampXminusCloseSensor,
                onlineClampXminusCloseSensorC,
                onlineClampXminusOpenSensor,
                onlineClampXminusOpenSensorC);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampXminusIsUnlocked() {
        simulateAutochangerOnlineClampIsUnlocked(onlineClampXminusCloseSensor,
                onlineClampXminusCloseSensorC,
                onlineClampXminusOpenSensor,
                onlineClampXminusOpenSensorC);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampXplusIsLocked() {
        simulateAutochangerOnlineClampIsLocked(onlineClampXplusCloseSensor,
                onlineClampXplusCloseSensorC,
                onlineClampXplusOpenSensor,
                onlineClampXplusOpenSensorC);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampXplusIsUnlocked() {
        simulateAutochangerOnlineClampIsUnlocked(onlineClampXplusCloseSensor,
                onlineClampXplusCloseSensorC,
                onlineClampXplusOpenSensor,
                onlineClampXplusOpenSensorC);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampYminusIsLocked() {
        simulateAutochangerOnlineClampIsLocked(onlineClampYminusCloseSensor,
                onlineClampYminusCloseSensorC,
                onlineClampYminusOpenSensor,
                onlineClampYminusOpenSensorC);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampYminusIsUnlocked() {
        simulateAutochangerOnlineClampIsUnlocked(onlineClampYminusCloseSensor,
                onlineClampYminusCloseSensorC,
                onlineClampYminusOpenSensor,
                onlineClampYminusOpenSensorC);
    }
    
    /**
     * Changes the values of the gateway in order to simulate that the filter read by autochanger 
     * at STANDBY is the filterID of the filter on carousel at STANDBY.
     * This has to be done only for wholeFCS.
     * In autochanger-standalone mode this is not needed.
     */
    private void simulateFilterIDAtStandby() {
        if ((carousel instanceof CarouselModule) && ((CarouselModule)carousel).getFilterAtStandby() != null) {
                simulateFilterID(((CarouselModule)carousel).getFilterAtStandby().getFilterID());
        }
    }
    
    /**
     * Here we want to simulate the autochanger sensors to be consistent with the filter location.
     * If a filter is on autochanger, we suppose that it's at ONLINE position and onlineClamps are LOCKED,
     * and latches are LOCKED, and filter presence sensors detect a filter.
     */
    @Override
    public void start() {
        boolean aFilterIsOnTrucks = false;
        autochanger = (AutoChangerModule) getComponentLookup().getComponentByName("autochanger");
        FilterManager filterManager = (FilterManager) getComponentLookup().getComponentByName("filterManager");
        Map<String, Filter> filtersMap = filterManager.getFiltersMapByName();
        for (Map.Entry<String, Filter> entry : filtersMap.entrySet()) {
            Filter filter = entry.getValue();
            
            if (filter.isOnAutoChanger()) {
                FCSLOG.fine(filter.getName() + " is on autochanger trucks.");
                aFilterIsOnTrucks = true;
                autochanger.setFilterOnTrucksID(filter.getFilterID());
            } 
        }
        if (aFilterIsOnTrucks) {
                simulateAutochangerTrucksIsOnline();
                simulateFilterIsOnAutochanger();
                simulateAutochangerLatchesAreLocked();
                simulateAutochangerOnlineClampsAreLocked();
                
        } else {
                simulateAutochangerTrucksIsAtHandoff();
                simulateAutochangerIsEmpty();
                simulateAutochangerLatchesAreUnlocked();
                simulateAutochangerOnlineClampsAreUnlocked();
        }
    }
}
