/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.simulation;

import static java.lang.Math.abs;
import java.util.Map;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.AutoChangerModule;

import org.lsst.ccs.subsystems.fcs.AutoChangerTwoTrucksModule;
import org.lsst.ccs.subsystems.fcs.AutochangerTwoLatches;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * To simulate a controller which moves autochanger trucks along its linear rails.
 * @author virieux
 */
public class SimuAutochangerLinearRailController extends SimuEPOSControllerModule {

    @ConfigurationParameter(isFinal = true)
    private String plutoGatewayName;
    
    private AutoChangerModule autochanger;
    private AutoChangerTwoTrucksModule trucks;
    private SimuAutochangerPlutoGateway plutoGateway;
    private FilterHolder carousel;
    private AutochangerTwoLatches latches;
    
    private SimuAutochangerLinearRailController slaveController;

    /**
     * Build a new SimuAutochangerLinearRailController
     * @param nodeID
     * @param serialNB
     * @param plutoGatewayName
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming 
     */
    public SimuAutochangerLinearRailController(
            String nodeID, String serialNB,
            String plutoGatewayName,
            Map<String, Integer> paramsForCurrent,
            Map<String, Integer> paramsForProfilePosition,
            Map<String, Integer> paramsForHoming) {
        super(nodeID, serialNB,
                paramsForCurrent, paramsForProfilePosition, paramsForHoming);
        this.plutoGatewayName = plutoGatewayName;
    }

    public void setSsiPosition(int ssiPosition) {
        this.ssiPosition = ssiPosition;
    }

    @Override
    public void initModule() {
        super.initModule();
        ComponentLookup lookup = getComponentLookup();
        this.trucks = (AutoChangerTwoTrucksModule) lookup.getComponentByName("autochangerTrucks");
        this.plutoGateway = (SimuAutochangerPlutoGateway) lookup.getComponentByName(plutoGatewayName);
        this.carousel = (FilterHolder) lookup.getComponentByName("carousel");
        this.autochanger = (AutoChangerModule) lookup.getComponentByName("autochanger");
        this.latches = (AutochangerTwoLatches) lookup.getComponentByName("latches");
        this.setSsiPosition(trucks.getEncoderRibbonMinValue());
        this.slaveController = (SimuAutochangerLinearRailController) lookup.getComponentByName("linearRailSlaveController");
    }

    /**
     * Simulates a command writeTargetPosition :
     * the position given as argument is the position to be reached.
     * This method updates position incremently to simulate a real trucks with a motor controller.
     * @param positionToReached 
     */
    @Override
    public void writeTargetPosition(int positionToReached) {
        this.targetPosition = positionToReached;
        int positionInitiale = this.position;
        FCSLOG.debug("loader trucks is moving.");
        FCSLOG.debug(getName() + "=>position intitiale=" + position);
        int stepsNB;
        if (abs(targetPosition - this.position) > 10000) {
            stepsNB = 20;
        } else {
            stepsNB = 5;
        }

        int stepHeight = (targetPosition - this.position) / stepsNB;

        simulateTrucksMotion(positionInitiale,stepsNB,stepHeight);

        this.position = targetPosition;
        this.slaveController.setPosition(this.position);

        trucks.updatePosition();
        trucks.publishData();
        FCSLOG.info("plutoGateway=" + plutoGateway.toString());

        if (position == trucks.getHandoffPosition()) {
            FCSLOG.info("simulateAutochangerTrucksIsAtHandoff");
            this.plutoGateway.simulateAutochangerTrucksIsAtHandoff();
            
        } else if (position == trucks.getStandbyPosition()) {
            this.plutoGateway.simulateAutochangerTrucksIsAtStandby();
            if (carousel.isHoldingFilter()) {
                FCSLOG.info("simulateFilterIsOnAutochanger");
                this.plutoGateway.simulateFilterIsOnAutochanger();
            } 
            
        } else if (position == trucks.getOnlinePosition()) {
            FCSLOG.info("simulateAutochangerTrucksIsOnline");
            this.plutoGateway.simulateAutochangerTrucksIsOnline();
            
        } else {
            FCSLOG.info("simulateAutochangerTrucksIsInTravel");
            this.plutoGateway.simulateAutochangerTrucksIsInTravel();
        }
        FCSLOG.info("plutoGateway=" + plutoGateway.toString());
    }
    
    /**
     * simulates FilterPresenceSensors according to the position of the trucks
     * and publishes trucks data.
     * @param positionInitiale
     * @param stepHeight 
     */
    private void simulateFilterPresenceSensors(int positionInitiale, int stepHeight) {
        /* on se rapproche de STANDBY s'il y a un filtre sur le carousel il faut que l'autochnager en tienne compte*/
        if (positionInitiale > 0 && position <= 100 && carousel.isHoldingFilter()) {
            FCSLOG.info("simulateFilterIsOnAutochanger");
            this.plutoGateway.simulateFilterIsOnAutochanger();

        /* on s'eloigne de STANDBY il faut mettre a jour l'autochanger*/    
        } else if (positionInitiale == 0 && latches.isUnlocked() && position > 100 && position < (2*stepHeight)) {
            FCSLOG.info("simulateAutochangerIsEmpty");
            this.plutoGateway.simulateAutochangerIsEmpty();
        }
        autochanger.updateStateWithSensors();
        trucks.publishData();
    }
    
    /**
     * simulates trucks motion : increment or decrement position value from an initialPosition,
     * with a number of steps and a step height.
     * @param positionInitiale
     * @param stepsNB number of steps
     * @param stepHeight
     */
    private void simulateTrucksMotion(int positionInitiale, int stepsNB, int stepHeight) {
        for (int i = 1; i < stepsNB; i++) {
            this.position = positionInitiale + (i * stepHeight);
            this.slaveController.setPosition(this.position);
            trucks.updatePosition();
            FCSLOG.info("trucks Position="+trucks.readPosition());
            simulateFilterPresenceSensors(positionInitiale, stepHeight);
            FCSLOG.debug(getName() + "i=" + i + ",position=" + position);
            try {
                Thread.sleep(100);
                if (trucks.getHaltRequired().get()) {
                    FCSLOG.debug(getName() + " STOP simulated trucks motion.");
                    return;
                }
            } catch (InterruptedException ex) {
                FCSLOG.error(ex);
                throw new RejectedCommandException(getName() + " sleep was interrupted.");
            }
        }
    }
}
