package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Map;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.AutochangerLatchModule;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * To simulate a controller which can open or close a latch in the autochanger.
 * @author virieux
 */
public class SimuAutochangerLatchController extends SimuEPOSControllerModule {

    @ConfigurationParameter(isFinal = true)
    private String latchName;
    
    @ConfigurationParameter(isFinal = true)
    private String plutoGatewayName;

    private AutochangerLatchModule latch;
    private SimuAutochangerPlutoGateway plutoGateway;

    /**
     * Build a SimuAutochangerLatchController.
     * @param nodeID
     * @param serialNB
     * @param latchName
     * @param plutoGatewayName
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming 
     */
    public SimuAutochangerLatchController(
            String nodeID, String serialNB,
            String latchName,
            String plutoGatewayName,
            Map<String, Integer> paramsForCurrent,
            Map<String, Integer> paramsForProfilePosition,
            Map<String, Integer> paramsForHoming) {
        super(nodeID, serialNB,
                paramsForCurrent, paramsForProfilePosition, paramsForHoming);
        this.mode = EPOSEnumerations.EposMode.CURRENT;
        this.latchName = latchName;
        this.plutoGatewayName = plutoGatewayName;
    }

    @Override
    public void initModule() {
        super.initModule();
        FCSLOG.debug(getName() + ":latchName=" + latchName);
        ComponentLookup lookup = getComponentLookup();
        latch = (AutochangerLatchModule) lookup.getComponentByName(latchName);
        plutoGateway = (SimuAutochangerPlutoGateway) lookup.getComponentByName(plutoGatewayName);
    }

    @Override
    public void writeCurrent(int aValue)  {
        this.targetCurrent = aValue;
        int currentInitial = this.actualCurrent;
        FCSLOG.debug("=>initial current=" + actualCurrent);
        int stepsNB = 10;
        int step = (targetCurrent - this.actualCurrent) / stepsNB;

        for (int i = 1; i <= stepsNB; i++) {
            if (i < stepsNB) {
                this.actualCurrent = currentInitial + (i * step);
            } //because in the last iteration the actual current could be 
            //different from the target value.
            else {
                this.actualCurrent = aValue;
            }
            latch.updateCurrent();
            if (actualCurrent == latch.getCurrentToOpen()) {
                FCSLOG.debug("plutoGateway=" + plutoGateway.toString());
                this.plutoGateway.simulateAutochangerLatchIsUnlocked(latchName);

            } else if (actualCurrent == -latch.getCurrentToOpen()) {
                this.plutoGateway.simulateAutochangerLatchIsLocked(latchName);
            }
            latch.publishData();
            FCSLOG.debug("i=" + i + ",actualCurrent=" + actualCurrent);
            try {
                Thread.sleep(200);
                if (latch.getHaltRequired().get()) {
                    FCSLOG.debug(getName() + " STOP simulated latch motion.");
                    return;
                }
            } catch (InterruptedException ex) {
                throw new FcsHardwareException(getName() + " sleep was interrupted.",ex);
            }

        }
    }

    @Override
    public void off()  {
        writeCurrent(0);
    }


}
