
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import org.lsst.ccs.subsystems.fcs.common.Sensor14bits;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This is a Model for a sensor on a can open bus which communicates over Can
 * Open protocol. The sensor is supposed to be connected to a ADC with analogics
 * input.(CAN-CBX-AI814)
 *
 * @author virieux
 */
public class CanOpenSensor14bits extends Sensor14bits {

    /**
     * The ADC on which this sensor is plugged.
     */
    protected CanOpenADC adc;

    /**
     * This is the name of the ADC in the groovy file which is used in
     * initModule.
     *
     */
    @ConfigurationParameter(isFinal=true,description="The name of the ADC where this sensor is plugged on.")
    private final String adcName;

    /**
     * The numero of analog input where the sensor is plugged.
     */
    @ConfigurationParameter(range="1..8",description="The numero of the input of ADC where this sensor is plugged on.")
    protected int numOfAnalogInput;

    /**
     * Build a CanOpenSensor14bits with an ADC name a numero of input and a tickMillis of 1000.
     * @param adcName
     * @param numOfAnalogInput 
     */
    public CanOpenSensor14bits(String adcName, int numOfAnalogInput) {
        super(1000); //initialize tickMillis to 1000
        this.adcName = adcName;
        this.numOfAnalogInput = numOfAnalogInput;
    }

    @Override
    public void initModule() {
        this.adc = (CanOpenADC) getComponentLookup().getComponentByName(adcName);
    }

    /**
     * Read a new value of this sensor on the ADC by sending a readSDO request.
     * This was used in single-filter-test with real hardware at the beginning but it's no more used
     * with real hardware because we read new values in reading PDO request. 
     * But it's used for simulation.
     * //TODO move this method in SimuCanOpenSensor14bits if we don't use it for scale1 prototype 's carousel
     * //real hardware neither.
     * @return
     * @throws FcsHardwareException 
     */
    @Override
    public int readNewValue()  {

        String valueInHexa = adc.readAnalogInput(numOfAnalogInput);

        value = Integer.parseInt(valueInHexa, 16);
        return value;
    }

    /**
     * Update the value of the sensor by retrieving it in a PDOStorage given as argument.
     * @param pdoStorage
     * @return 
     */
    @Override
    public int updateValue(PDOStorage pdoStorage) {
        String hexaValue = pdoStorage.getVarValue(this.adc.nodeID, 
                Integer.toString(this.numOfAnalogInput));
        this.value = Integer.parseInt(hexaValue, 16);
        return this.value;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append("/adcName=");
        sb.append(this.adcName);
        sb.append("/inputNumero=");
        sb.append(this.numOfAnalogInput);
        return sb.toString();
    }

}
