
package org.lsst.ccs.subsystems.fcs;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import static org.lsst.ccs.subsystems.fcs.utils.FcsUtils.TICKMILLIS;

/**
 * This is the Main Module for every software related to the Filters Exchanger :
 * - single-filter-test control-command - scale 1 prototype control-command -
 * real Filters Exchanger This class is designed to gather the commons methods
 * to all these softwares.
 *
 * @author virieux
 */
public class MainModule extends Module implements AlertRaiser {

    protected BridgeToHardware bridge;
    protected AtomicBoolean haltRequired;
    protected AtomicBoolean stopRequired;

    public MainModule(BridgeToHardware bridge) {
        super(TICKMILLIS);
        this.bridge = bridge;
    }

    public boolean isHaltRequired() {
        return haltRequired.get();
    }

    public boolean isStopRequired() {
        return stopRequired.get();
    }

    @Override
    public void initModule() {
        haltRequired = new AtomicBoolean(false);
        stopRequired = new AtomicBoolean(false);
        String msg = controlRealHardware() ? "Real Hardware" : "Simulation";
        FCSLOG.info(getName() + "==>" + msg);
    }


    /**
     * This happens when the hardware has been initialized. (after completeInitialization).
     * It starts the ticking in super.postStart().
     * @throws HardwareException 
     */
    @Override
    public void postStart() throws HardwareException {
        super.postStart();
        FCSLOG.fine(getName() + ":postStart begins.");
        try {
            this.updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }
        FCSLOG.fine(getName() + ":postStart ends.");
    }

    /**
     * This methods updates the boolean hardwareReady from the hardware bridge.
     *
     * @return true if hardware is booted, identified and initialized
     */
    @Command(level = Command.ENGINEERING1, type = Command.CommandType.QUERY,
            description = "Return true if hardware is booted, identified and initialized.")
    public boolean isCANDevicesReady() {
        return this.bridge.isCANDevicesReady();
    }

    @Command(type = Command.CommandType.QUERY, 
            description = "Return true if this main module controls real hardware.")
    public boolean controlRealHardware() {
        return this.bridge.isRealHardware();
    }
    
    /**
     * Returns list of hardware names.
     * Used by GUI.
     * @return list of hardware this MainModule manages.
     */
    @Command(level = Command.ENGINEERING1, description = "Returns the list of CANopen hardware manages by this subsystem."
            + "that this subsystem manages.", type = Command.CommandType.QUERY)
    public List<String> listHardwareNames() {
        return this.bridge.listHardwareNames();
    }
    
    /**
     * For Whole FCS GUI. 
     * Has to be overriden in FcsMainModule.
     * @return 
     */
    @Command(level = Command.ENGINEERING1, description = "Return the list of LOADER CANopen hardware "
            + "that this subsystem manages.", type = Command.CommandType.QUERY)
    public List<String> listLoaderHardwareNames() {
        return Collections.emptyList();
    }    
    
    
    /**
     * Return a list of filter names that the subsystem manages.
     * For the GUI or the Console.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY,
            description = "Return the list of names of filters  "
            + "that this subsystem manages.",             
            level = Command.ENGINEERING1,
            alias="listFilterNames")
    public List<String> getFilterNames() {
        FilterManager filterManager = (FilterManager) getComponentLookup().getComponentByName("filterManager");
        if (filterManager == null) {
            return Collections.emptyList();
        } else {
            return filterManager.getFilterNamesSortedByFilterID();
        }
    }

    /**
     * Update state in reading sensors.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors()  {
        this.bridge.publishData();
    }
    
    /**
     * Updates the FCS state and FCS readyness state and publishes on the status bus.
     * Checks that hardware is ready to be operated and moved.
     * This means that :
     * - all CAN open devices are booted, identified and initialized,
     * - homing has been done on the controllers.
     * This updates the FCS state and FCS readyness state and publishes on the status bus.
     * This has to be overriden if there is other initializations to do on the hardware like homing.
     * 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update FCS state and FCS readyness state and publishes on the status bus.")    
    public void updateFCSState() {
        if (this.isCANDevicesReady()) {
            /* The initialization has been done, so now the hardware is ready */
            this.getSubsystem().updateAgentState(FcsEnumerations.FilterState.HARDWARE_READY, 
                FcsEnumerations.FilterReadinessState.READY);
        }
    }    


    @Override
    public TreeWalkerDiag signal(Signal signal) {
        switch (signal.getLevel()) {
            case HALT:
                FCSLOG.debug("HALT required");
                this.haltRequired.set(true);
                break;
            case STOP:
                FCSLOG.debug("STOP required");
                this.stopRequired.set(true);
                break;
            case RE_START:
                this.haltRequired.set(false);
                this.stopRequired.set(false);
                break;
            default:
                assert false;
        }
        //we want the signal to go to all my children.
        return TreeWalkerDiag.GO;
    }

    

}
