
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This is the model for the loader in the Filter Exchange System. 
 * The loader is used to load a filter into the camera or to unload a filter 
 * from the camera.
 *
 * @author virieux
 */
public class LoaderModule extends Module implements HardwareController, FilterHolder {

    @ConfigurationParameter(isFinal = true)
    private final String plutoGatewayName;
    
    private final LoaderCarrierModule carrier;
    private final LoaderClampModule clamp;
    private BridgeToHardware bridgeToLoader;
    private MainModule main;
    
    private FilterHolder autochanger;

    private final NumericSensor filterPresenceSensor0;
    private final NumericSensor filterPresenceSensor1;
    private final NumericSensor loaderOnCameraSensor0;
    private final NumericSensor loaderOnCameraSensor1;
    
    private final RedondantSensors filterPresenceSensors;
    private final RedondantSensors loaderOnCameraSensors;

    private PlutoGatewayInterface plutoGateway;
    private EPOSController hooksController;
    private EPOSController carrierController;

    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

    /**
     * Build a LoaderModule with a carrier, a clamp and sensors.
     * @param carrier
     * @param clamp
     * @param plutoGatewayName
     * @param filterPresenceSensor0
     * @param filterPresenceSensor1
     * @param loaderOnCameraSensor0
     * @param loaderOnCameraSensor1 
     */
    public LoaderModule(
            LoaderCarrierModule carrier,
            LoaderClampModule clamp,
            String plutoGatewayName,
            NumericSensor filterPresenceSensor0,
            NumericSensor filterPresenceSensor1,
            NumericSensor loaderOnCameraSensor0,
            NumericSensor loaderOnCameraSensor1) {
        this.carrier = carrier;
        this.clamp = clamp;
        this.plutoGatewayName = plutoGatewayName;
        this.filterPresenceSensor0 = filterPresenceSensor0;
        this.filterPresenceSensor1 = filterPresenceSensor1;
        this.loaderOnCameraSensor0 = loaderOnCameraSensor0;
        this.loaderOnCameraSensor1 = loaderOnCameraSensor1;
        filterPresenceSensors = new RedondantSensors(filterPresenceSensor0, filterPresenceSensor1);
        loaderOnCameraSensors = new RedondantSensors(loaderOnCameraSensor0, loaderOnCameraSensor1);
    }




       

    @Override
    public void initModule() {
        ComponentLookup lookup = getComponentLookup();
        this.plutoGateway = (PlutoGatewayInterface) lookup.getComponentByName(plutoGatewayName);
        this.bridgeToLoader = (BridgeToHardware) lookup.getComponentByName("loaderTcpProxy");
        this.main = (MainModule) lookup.getComponentByName("main");
        this.hooksController = (EPOSController) getComponentLookup().getComponentByName("hooksController");
        this.carrierController = (EPOSController) getComponentLookup().getComponentByName("carrierController");
        
        if (lookup.getComponentByName("autochanger") instanceof FilterHolder) {
            autochanger = (FilterHolder) lookup.getComponentByName("autochanger");
        } else {
            final String MSG = getName() + " ==> autochanger doesn't implements FilterHolder -"
                    + " Please fix groovy description file or LoaderModule implementation.";
            FCSLOG.error(MSG);
            throw new IllegalArgumentException(MSG);
        }
    }
    
    /**
     * 
     * @return true if hardware (controllers and plutoGateway) is correctly initialized
     *          and homing of the controllers is done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if hardware (controllers and plutoGateway) is correctly initialized"
                + "and homing of the controllers is done.")
    public boolean isInitialized() {
        boolean devicesInitialized = plutoGateway.isInitialized()
                && hooksController.isInitialized()
                && carrierController.isInitialized();
        return devicesInitialized && clamp.isInitialized() && carrier.isInitialized();
    }
    
    
    

    /**
     * Returns the boolean field empty. If the empty boolean is being updated
     * and waits for a response from a sensor, this methods waits until empty is
     * updated. If the field empty is not being updated, it returns immediatly
     * the field empty.
     *
     * @return empty
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if there is no filter in the loader. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {

        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(getName() + ": has been interrupted while waiting for end of update.",ex);
                }

            }
            return !filterPresenceSensors.isOn();

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * atHandoff is updated. If the field atHandoff is not being updated, it
     * returns immediatly the field atHandoff.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the loader is connected on the camera. This command doesn't read again the sensors.")
    public boolean isConnectedOnCamera() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(getName() + ": has been interrupted while waiting for end of update.",ex);
                }

            }
            return loaderOnCameraSensors.isOn();

        } finally {
            lock.unlock();
        }
    }

    /**
     * Return true if the autochanger is holding the filter.
     * This command doesn't read again the sensors.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the autochanger is holding the filter. This command doesn't read again the sensors.")
    public boolean isAutochangerHoldingFilter()  {       
        return autochanger.isHoldingFilter();
    }

    /**
     * Return true if a filter is present and it is held by the loader clamp.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if a filter is present and it is held by the loader clamp.")
    @Override
    public boolean isHoldingFilter()   {
        this.updateStateWithSensors();
        this.clamp.updatePosition();
        return this.clamp.isAtClampedPosition() && !this.isEmpty();
    }

    /**
     * 
     * @return true if carrier is at STORAGE position.
     */
    public boolean isCarrierAtStoragePosition() {
        return carrier.isAtStoragePosition();
    }

    /**
     * This command has to be executed after the initialization phase and after
     * the checkHardware command.
     * It can't be automaticaly executed during initialization phase because it's not compliant with the
     * CCS requirement which says that the initialization of the subsystem can't make move hardware.
     * @throws HardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Attention ! this commands does the homing of the controllers and might move hardware. "
                    + "Initialize loader hardware.",
            alias="homing")
    public void initializeHardware() throws HardwareException {
        checkHardware();
        try {
            clamp.initializeHardware();
            
        } catch (FailedCommandException | FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }
        /* If the carrier is empty and the clamp is LOCKED, we can't start the loader subsystem.*/
        /* The test clamp.isLocked can be done only after the clamp has been initialized.*/
        if (this.isEmpty() && clamp.isLocked()) {
            String msg = getName() + ": carrier is empty and clamp is LOCKED - can't start.";
            FCSLOG.error(msg);
            throw new HardwareException(true, msg);
        }
        this.updateFCSState();
    }

    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {

        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }

        clamp.checkHardware();
        carrier.checkHardware();
        /*NOT TO BE DONE here : initialization of the hardware because we don't want 
        to move hardware during initialization phase.*/
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * CheckStarted is executed by completeInitialization when a first start has
     * failed. Because end user could have changed many things, we have to check
     * again that all is correct.
     *
     * @throws org.lsst.ccs.HardwareException
     *
     */
    @Override
    public void checkStarted() throws HardwareException {

        FCSLOG.info(getName() + " BEGIN checkStarted");
        this.checkHardware();        
        FCSLOG.info(getName() + " END checkStarted");
    }

    /**
     * This method is executed during a shutdown of the subystem to prevent a
     * shutdown when some hardware is moving. Nothing to do here because all the
     * moving hardware of the loader is represented by the class MobilItemModule
     * where the method checkStopped is implemented.
     *
     * @throws org.lsst.ccs.HardwareException
     *
     */
    @Override
    public void checkStopped() throws HardwareException {
        //done already in my children which are MobilItemModule and my other children don't need that.
    }

    @Override
    public void tick() {
        //Don't execute here methods which take a lock.
        this.publishData();
    }
    

    
    /**
     * Check if loader is connected on camera.
     * @throws RejectedCommandException if not connected on camera
     */
    public void checkConnectedOnCamera() {
        if (!isConnectedOnCamera()) {
            throw new RejectedCommandException(getName()
                    + " Loader not connected - can't execute commands.");
        }
    }

    /**
     * Check if the loader carrier can move.
     * @throws FcsHardwareException
     * @throws FailedCommandException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the carrier can move.")
    public void checkConditionsForCarrierMotion()  {
        FCSLOG.info(getName() + " checking pre-conditions for carrier motion");

        updateStateWithSensors();
        checkConnectedOnCamera();
        clamp.checkInitialized();

        if (clamp.isInError()) {
            throw new RejectedCommandException(getName() + ": carrier can't move because clamp is in ERROR.");
        }

        if (!this.isEmpty() && this.isHoldingFilter() && isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + " carrier can't move because a filter is in the loader"
                    + " and it's held by loader AND autochanger.");

        } else if (!this.isEmpty() && !this.isHoldingFilter() && !isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + " carrier can't move because the filter in the loader is not held"
                    + " neither by loader neither by autochanger. "
                    + "Close loader clamp or autochanger latches");

        }
    }

    /**
     * Check if the clamp can be opened.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the clamp can be opened.")
    public void checkConditionsForOpeningHooks()  {
        
        FCSLOG.info(getName() + " checking pre-conditions for opening hooks");

        updateStateWithSensors();
        checkConnectedOnCamera();
        clamp.checkInitialized();

        clamp.checkSensorsOK();

        if (!this.isEmpty()) {
            if (!carrier.isAtHandoffPosition()) {
                String msg = getName() + ": carrier is loaded with a filter but is not "
                        + "at handoff position - can't open clamp.";
                FCSLOG.error(msg);
                throw new RejectedCommandException(msg);
            }

            if (!isAutochangerHoldingFilter()) {
                String msg = getName() + ": A filter is in the loader but autochanger and "
                        + "not holding it - can't open clamp.";
                FCSLOG.error(msg);
                throw new RejectedCommandException(msg);
            } 
        }


    }

    /**
     * Check if the clamp can be closed.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the clamp can be closed.")
    public void checkPreConditionsForClosingHooks()  {
        FCSLOG.info(getName() + " checking pre-conditions for closing hooks");
        updateStateWithSensors();
        checkConnectedOnCamera();
        clamp.checkInitialized();
        clamp.checkSensorsOK();

        this.clamp.updatePosition();
        
        if (this.isEmpty()) {
            String msg = getName() + ": no filter in carrier - can't execute close command for Module "
                    + clamp.getName();
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
        //at this point carrier is NOT empty
        if (this.clamp.isAtClampedPosition()) {
            String msg = getName() + ": a filter is already clamped - can't execute close or clamp command for Module "
                    + clamp.getName();
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
    }

    /**
     * This method closes the clamps. When this action is completed the filter
     * is clamped softly on the loader and can't fall.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Close hooks.")
    public void closeHooks()  {
        clamp.close();
    }

    /**
     * This method closes the clamps strongly. When this action is completed,
     * the filter is clamped softly on the loader and is tight fermely on the
     * carrier.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Clamp hooks.")
    public void clampHooks()  {
        clamp.clamp();
    }

    /**
     * This command opens the clamp.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(level = Command.ENGINEERING1, type = Command.CommandType.ACTION,
            description = "Open the hooks")
    public void openHooks()  {
        clamp.open();
    }

    /**
     * Moves the loader carrier to the handoff position.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the loader carrier to HANDOFF position")
    public void goToHandoff()  {
        //TODO check the preconditions
        carrier.goToHandOff();
    }

    /**
     * Moves the loader carrier to the storage position.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the loader carrier to STORAGE position")
    public void goToStorage()  {
        //TODO check the preconditions
        carrier.goToStorage();
    }

    /**
     * This methods updates the carrier and clamp state in reading all the
     * sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors()  {
        
        this.bridgeToLoader.publishData();
        
        plutoGateway.checkBooted();
        plutoGateway.checkInitialized();

        lock.lock();
        try {
            updatingState = true;
            this.plutoGateway.updateValues();
            String[] readHexaValues = this.plutoGateway.getHexaValues();

            this.filterPresenceSensors.updateValues(readHexaValues);
            this.loaderOnCameraSensors.updateValues(readHexaValues);
            this.clamp.updateStateWithSensors(readHexaValues);
            this.carrier.updateStateWithSensors(readHexaValues);

        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
            this.clamp.publishData();
            this.carrier.publishData();
        }
    }
    
    /**
     * Check that Loader hardware is ready to be operated and moved.
     * This means that :
     * - all CAN open devices are booted, identified and initialized,
     * - homing has been done on the controllers.
     * This updates the FCS state and FCS readyness state and publishes on the status bus.
     * 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update FCS state and FCS readyness state and publishes on the status bus.")
    public void updateFCSState() {
        if (clamp.isInitialized() && carrier.isInitialized()) {
            main.updateFCSState();

        } 
    }

    /**
     * This command can be launched when a filter is in the loader and we want
     * to put in inside the camera.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Load a filter from the loader into the camera.")
    public void loadFilterInCamera()  {
        
        checkConnectedOnCamera();
        if (!isHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + ":loader is not holding a filter : can't load a filter into camera.");
        }
        if (!carrier.isAtStoragePosition()) //TODO : what do we do in this case ?
        {
            throw new RejectedCommandException(getName()
                    + ":carrier loader is not at storage position : can't load a filter into camera.");
        }

        //go to Handoff position with the filter
        carrier.goToHandOff();
        if (main.isHaltRequired() || main.isStopRequired()) {
            throw new FailedCommandException(getName() + ": received HALT or STOP command.");
        }
        //wait until the autochanger hold the filter
        while (!isAutochangerHoldingFilter()) {
            FCSLOG.debug(getName() + " waiting until autochanger holds the filter...");
            try {
                Thread.sleep(300);
            } catch (InterruptedException ex) {
                throw new FcsHardwareException("loadFilterInCamera interrupted while waiting for the autochanger"
                        + " to take the filter",ex);
            }
            updateStateWithSensors();
        }

        if (isAutochangerHoldingFilter()) {
            openHooks();
        }
        if (main.isHaltRequired() || main.isStopRequired()) {
            throw new FailedCommandException(getName() + ": received HALT or STOP command.");
        }

        // empty carrier go to Storage
        goToStorage();
    }

    /**
     * This command can be launched when the loader is empty and we want to take
     * a filter from the camera. A filter must be at handoff position and held
     * by autochanger.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     * @throws SDORequestException
     * @throws FailedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Unload a filter from the camera into the loader.")
    public void unloadFilterFromCamera()  {

        updateStateWithSensors();
        checkConnectedOnCamera();
        
        if (isHoldingFilter()) {
            throw new RejectedCommandException(getName() + ":loader is holding a filter : "
                    + "can't unload a filter from camera.");
        }

        //wait for the autochanger to put a filter at handoff position
        //in loader test bench it's just a switch to put on.
        while (!isAutochangerHoldingFilter()) {
            FCSLOG.debug(getName() + " waiting until autochanger holds the filter...");
            try {
                Thread.sleep(300);
            } catch (InterruptedException ex) {
                throw new FcsHardwareException("unloadFilterFromCamera interrupted while waiting until autochanger holds the filter",ex);
            }
            updateStateWithSensors();
        }

        //go tho Handoff position - carrier empty
        goToHandoff();
        if (main.isHaltRequired() || main.isStopRequired()) {
            throw new FailedCommandException(getName() + ": received HALT or STOP command.");
        }

        //at handoff position a filter should be here.
        updateStateWithSensors();
        if (isEmpty()) {
            throw new FailedCommandException(getName()
                    + ": loader presence filter sensor doesn't detect a filter"
                    + " - can't go on.");
        }
        closeHooks();

        //wait for the autochanger to unlock the filter at handoff position
        //in loader test bench it's just a switch to put off.(switch A22)
        while (isAutochangerHoldingFilter()) {
            FCSLOG.debug(getName() + " waiting until autochanger releases the filter...");
            try {
                Thread.sleep(300);
            } catch (InterruptedException ex) {
                throw new FcsHardwareException("unloadFilterFromCamera interrupted while waiting until "
                        + "autochanger releases the filter",ex);
            }
            updateStateWithSensors();
        }

        //close more fermely the hooks to hold the filter.
        clampHooks();
        if (main.isHaltRequired() || main.isStopRequired()) {
            throw new FailedCommandException(getName() + ": received HALT or STOP command.");
        }

        //carrier holding filter goes to storage position.
        goToStorage();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "List and display info on the hooks.")
    public String listHooks() {
        return this.clamp.listHooks();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "List and display hooks sensors values.")
    public String listSensorsValues() {
        return this.clamp.listSensorsValues();
    }

    
    /**
     * Creates an Object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByLoader createStatusDataPublishedByLoader() {
        StatusDataPublishedByLoader status = new StatusDataPublishedByLoader();
        status.setName(getName());
        status.setFilterPresenceSensorValue0(filterPresenceSensor0.getDigitalValue());
        status.setFilterPresenceSensorValue1(filterPresenceSensor1.getDigitalValue());
        status.setFilterPresenceSensorsInError(filterPresenceSensors.isInError());
        status.setLoaderOnCameraSensorValue0(loaderOnCameraSensor0.getDigitalValue());
        status.setLoaderOnCameraSensorValue1(loaderOnCameraSensor1.getDigitalValue());
        status.setLoaderOnCameraSensorsInError(loaderOnCameraSensors.isInError());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        StatusDataPublishedByLoader status = this.createStatusDataPublishedByLoader();
        KeyValueData kvd = new KeyValueData("loaderGeneral", status);
        
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }

    @Override
    public boolean isAtHandoff() {
        return carrier.isAtHandoffPosition();
    }

    @Override
    public boolean isAtStandby() {
        return false;
    }

    /**
     * Print list of hardware with initialization information.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, 
            level = Command.ENGINEERING1, 
            description = "Print list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(carrier.printHardwareState());
        sb.append("\n");
        sb.append(clamp.printHardwareState());
        return sb.toString();
    }
}
