
package org.lsst.ccs.subsystems.fcs;

import java.util.Map;
import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_TIMEOUT;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This is model for the clamps mechanism in the loader. 
 * 4 hooks are used for the clamping of the filter in the loader. 
 * This hooks are moved all together by the hooksController.
 * 
 * Raises alert :
 * - CAN_BUS_TIMEOUT in updateStateWithSensorsToCheckIfActionIsCompleted
 *
 * @author virieux
 */
public class LoaderClampModule extends MobileItemModule implements MovedByEPOSController {

    private final LoaderHookModule hook1;
    private final LoaderHookModule hook2;
    private final LoaderHookModule hook3;
    private final LoaderHookModule hook4;
    
    /* a list of hooks to make easier the computation of the LockStatus */
    private final LoaderHookModule[] hooks;
    
    private EPOSController hooksController;
    private LoaderModule loader;
    private BridgeToHardware bridgeToLoader;

    private FcsEnumerations.LockStatus lockStatus;

    @ConfigurationParameter(description="timeout in milliseconds : if closing the clamp last more than "
            + "this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForClosingHooks = 60000;
    
    @ConfigurationParameter(description="timeout in milliseconds : if closing strongly the clamp last "
    + "more than this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForClampingHooks = 60000;
    
    @ConfigurationParameter(description="timeout in milliseconds : if opening the clamp last more than"
    + " this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForOpeningHooks = 60000;
    
    @ConfigurationParameter(description="timeout in milliseconds : if the action of hoing to home position "
            + "last more than this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForGoingToHomePosition = 60000;
    
    @ConfigurationParameter(description="target encoder absolute value in qc to open")
    private int targetPositionToOpen = 5000;
    
    @ConfigurationParameter(description="target encoder absolute value in qc when hooks are LOCKED")
    private int targetPositionToClose = 492000;
    
    @ConfigurationParameter(description="target encoder absolute value in qc when hooks are LOCKED STRONGLY")
    private int targetPositionToClamp = 515000;
    
    @ConfigurationParameter(description="current to close in initialisation phase, in mA")
    private int currentToClamp = 280;
    
    @ConfigurationParameter(description="current to go to home position in initialisation phase, in mA")
    private int currentToGoHome = -200;
    
    /* for GUI */
    @ConfigurationParameter(description="min position value used by the GUI")
    private int minPosition = 0;
    
    /* for GUI */
    @ConfigurationParameter(description="max position value used by the GUI")
    private int maxPosition = 515000; 
    
    /* for GUI */
    @ConfigurationParameter(description="min current value used by the GUI")
    private final int minCurrent = -300;
            
    /* for GUI */
    @ConfigurationParameter(description="max current value used by the GUI")
    private final int maxCurrent = 400;
    
    @ConfigurationParameter
    private Map<String, Integer> paramsForCurrentToGoHome;
    
    @ConfigurationParameter
    private Map<String, Integer> paramsForCurrentToClamp;
            
    private volatile boolean initialized = false;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    private int position;
    private int readCurrent;
    
    /**
     * This is the value returned by the force sensor.
     */
    private int force;
    

    /**
     * FOR THE GUI
     * true if the controller is in fault
     * false if the user does a faultReset
     */
    private boolean controllerInFault;

    /**
     * Build a new LoaderClampModule with 4 hooks and the parameters to configure the EPOS controller
     * in mode CURRENT and in mode HOMING.
     * @param hook1
     * @param hook2
     * @param hook3
     * @param hook4
     * @param paramsForCurrentToClamp
     * @param paramsForCurrentToGoHomePosition 
     */
    public LoaderClampModule(
            LoaderHookModule hook1,
            LoaderHookModule hook2,
            LoaderHookModule hook3,
            LoaderHookModule hook4,
            Map<String, Integer> paramsForCurrentToClamp,
            Map<String, Integer> paramsForCurrentToGoHomePosition) {
        this.hook1 = hook1;
        this.hook2 = hook2;
        this.hook3 = hook3;
        this.hook4 = hook4;
        this.hooks = new LoaderHookModule[]{hook1, hook2, hook3, hook4};
        this.paramsForCurrentToClamp = paramsForCurrentToClamp;
        this.paramsForCurrentToGoHome = paramsForCurrentToGoHomePosition;
    }
    
    /**
     * Returns controller name.
     * @return 
     */
    @Override
    public String getControllerName() {
        return hooksController.getName();
    }

    @Override
    public boolean isControllerInFault() {
        return this.controllerInFault; 
    }

    @Override
    public void setControllerInFault(boolean controllerInFault) {
        this.controllerInFault = controllerInFault;
    }

    //for the simulator
    public int getTargetPositionToOpen() {
        return targetPositionToOpen;
    }

    //for the simulator
    public int getTargetPositionToClose() {
        return targetPositionToClose;
    }

    //for the simulator
    public int getTargetPositionToClamp() {
        return targetPositionToClamp;
    }

    //for the simulator
    public int getCurrentToClamp() {
        return currentToClamp;
    }

    //for the simulator
    public int getCurrentToGoHome() {
        return currentToGoHome;
    }

    /**
     * Returns true if loader clamp is initialized.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if loader clamp is initialized.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * 
     * @return lockStatus
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(getName() + ": interrupted in getLockStatus.",ex);
                }

            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns true if LockStatus=LOCKED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=LOCKED")
    public boolean isLocked() {
        return this.getLockStatus() == LockStatus.LOCKED;
    }

    /**
     * Returns true if LockStatus=UNLOCKED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=UNLOCKED")
    public boolean isUnlocked() {
        return this.getLockStatus() == LockStatus.UNLOCKED;
    }

    /**
     * Returns true if clamp LockStatus is in ERROR or LockStatus is UNKNOWN.
     * That means that one hook is in ERROR status or one or more of the 4 hooks has a different
     * LockStatus than the other.
     * @return true if clamp LockStatus is in ERROR or LockStatus is UNKNOWN.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=UNLOCKED")
    public boolean isInError() {
        return this.getLockStatus() == LockStatus.ERROR || this.getLockStatus() == LockStatus.UNKNOWN;
    }


    /**
     * Returns true if loader CANopen devices are booted, identified and initialized.
     * @return 
     */
    @Override
    public boolean isCANDevicesReady() {
        return bridgeToLoader.isCANDevicesReady();
    }

    @Override
    public void initModule() {
        ComponentLookup lookup = getComponentLookup();
        this.hooksController = (EPOSController) lookup.getComponentByName("hooksController");
        this.loader = (LoaderModule) lookup.getComponentByName("loader");
        this.bridgeToLoader = (BridgeToHardware) lookup.getComponentByName("loaderTcpProxy");
        this.lockStatus = LockStatus.UNKNOWN;

        //listens to hooksController to detect the controller's faultReset
        //or the emergency messages coming from the controller.
        if (hooksController instanceof Observable) {
            this.listens((Observable) hooksController);
        }
    }

    /**
     * Returns true if hook is open and at home position
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if clamp is open and at home position")
    public boolean isAtHomePosition()  {
        return position == 0;
    }

    /**
     * Returns true if hook is closed and at clamped position
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if hook is closed and at clamped position")
    public boolean isAtClampedPosition() {
        return (this.targetPositionToClamp - 1000 <= this.position)
                && (this.targetPositionToClamp <= this.position + 1000);
    }

    /**
     * Returns true if hook is at closed position.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if hook is at closed position.")
    public boolean isAtClosedPosition()  {
        return position == this.targetPositionToClose;
    }

    /**
     * Returns true if hook is at open position.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if hook is at open position.")
    public boolean isAtOpenPosition() {
        return position == this.targetPositionToOpen;
    }

    /**
     * If loader is empty, go to home position otherwise go to clamped position.
     * ATTENTION : this command moves the clamp.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Attention : this commands moves the clamp to do the homing of the controller. "
                    + "If loader is empty, go to home position otherwise go to clamped position.",
            alias = "homing")
    public void initializeHardware()  {

        loader.updateStateWithSensors();
        publishData();
        checkSensorsOK();
        hooksController.checkInitialized();

        if (loader.isEmpty()) {
            this.goToHomePosition();

        } else {
            this.goToClampedPosition();

        }
        this.initialized = true;
        loader.updateFCSState();
    }

    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        try {
            hooksController.initializeAndCheckHardware();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + ":" + ex);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false, ex);
        }
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }
    
    /**
     * check if clamp sensors have correct values.
     * @throws FcsHardwareException if in error.
     */
    public void checkSensorsOK() {
        if (this.isInError()) {
            String msg = getName() + " in ERROR state - can't execute commands.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }
    }
    
    /**
     * Check if clamp is initialized.
     * @throws RejectedCommandException if clamp is not initialized 
     */
    public void checkInitialized() {
        if (!isInitialized()) {
            String msg = getName() + ": clamp is not intialized.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
    }

    @Override
    public void tick() {
        //Don't execute here a method which takes a lock.
        publishData();
    }

    /**
     * In the initialisation phase, this close the clamp to be able to define
     * the absolute position of the encoder.
     *
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Go to clamped position to intialize hardware.")
    public void goToClampedPosition()  {

        loader.updateStateWithSensors();
        loader.checkConnectedOnCamera();

        checkSensorsOK();

        if (loader.isEmpty()) {
            throw new RejectedCommandException(getName()
                    + "No filter in carrier -  can't execute command goToClampedPosition.");
        }

        updatePosition();
        if (initialized && isAtClampedPosition()) {
            throw new RejectedCommandException(getName() + " is already at clamped position - nothing to do.");
        }

        this.executeAction(FcsEnumerations.MobileItemAction.GOTOCLAMPEDPOSITION,
                timeoutForClampingHooks);
    }

    /**
     * In the initialisation phase, this open the clamp to be able to define the
     * absolute position of the encoder. September 2014 : this command is also
     * used to open the clamp.
     *
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Go to home position to initialize hardware.")
    public void goToHomePosition()  {
        loader.updateStateWithSensors();
        loader.checkConnectedOnCamera();
        checkConditionsToGoToHomePosition();
        updatePosition();
        this.executeAction(FcsEnumerations.MobileItemAction.GOTOHOMEPOSITION,timeoutForGoingToHomePosition);
    }
    
    public void checkConditionsToGoToHomePosition() {
        checkSensorsOK();
        if (!initialized && !loader.isEmpty()) {
            throw new RejectedCommandException(getName()
                    + " There is a filter in carrier - can't execute command goToHomePosition.");
        }

        if (initialized && !loader.isEmpty() && !loader.isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + " There is a filter in carrier and Autochanger is not holding filter - "
                    + "can't execute command goToHomePosition. ");
        }
    }

    /**
     * Close clamp in order to hold the filter.
     * When the clamp is closed, the filter is held by loader so autochanger can open its latches
     * and move trucks back without the filter.
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Close clamp.")
    public void close()  {

        loader.checkPreConditionsForClosingHooks();
        if (this.isAtClosedPosition()) {
            String msg = getName() + ": is already closed.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
        this.executeAction(FcsEnumerations.MobileItemAction.CLOSELOADERHOOKS,timeoutForClosingHooks);
    }

    /**
     * Clamp clamp or strongly close.
     * When the clamp is clamped, the filter is held strongly and so the carrier can go to STORAGE position safely.
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Clamp clamp.")
    public void clamp()  {

        loader.checkPreConditionsForClosingHooks();
        if (!isLocked()) {
            throw new RejectedCommandException(getName()
                    + " has to be LOCKED first. Close hooks prior clamp.");
        }
        if (loader.isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + " Autochanger is holding filter. Open autochanger latches before "
                    + "clamping loader hooks.");
        }
        this.executeAction(FcsEnumerations.MobileItemAction.CLAMPLOADERHOOKS,timeoutForClampingHooks);
    }

    /**
     * Open clamp in order to release filter.
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open clamp.")
    public void open()  {
        loader.checkConditionsForOpeningHooks();
        this.executeAction(FcsEnumerations.MobileItemAction.OPENLOADERHOOKS,timeoutForOpeningHooks);
    }

    /**
     * Check if the action is completed. This method is called after
     * updateStateWithSensorsToCheckIfActionIsCompleted where the different new
     * values of current or position or sensors values have been updated.
     *
     * @param action
     * @return
     */
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        /* A loader clamp motion is completed when the position is in a range of */
        /* 10 microns around the target position. */
        boolean actionCompleted = false;
        if (action == MobileItemAction.GOTOHOMEPOSITION) {
            actionCompleted = isCurrentReached(currentToGoHome);
            
        } else if (action == MobileItemAction.OPENLOADERHOOKS){
            actionCompleted = this.position == this.targetPositionToOpen;
            
        } else if (action == MobileItemAction.CLOSELOADERHOOKS) {
            actionCompleted = this.position == this.targetPositionToClose;
            
        } else if (action == MobileItemAction.GOTOCLAMPEDPOSITION ||
                action == MobileItemAction.CLAMPLOADERHOOKS) {
            //TODO read the force sensor
            actionCompleted = isCurrentReached(currentToClamp);
        }
        return actionCompleted;
    }
    
    private boolean isCurrentReached(int currentToReach) {
        int DELTA = 5;
        FCSLOG.debug("readCurrent=" + readCurrent + " currentToReached=" + currentToReach);
        return currentToReach - DELTA <= this.readCurrent
                && this.readCurrent <= currentToReach + DELTA;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        try {
            loader.updateStateWithSensors();
            if (currentAction == MobileItemAction.CLOSELOADERHOOKS
                    || currentAction == MobileItemAction.OPENLOADERHOOKS) {
                position = this.hooksController.readPosition();
                
            } else if (currentAction== MobileItemAction.GOTOHOMEPOSITION
                    || currentAction == MobileItemAction.GOTOCLAMPEDPOSITION
                    || currentAction == MobileItemAction.CLAMPLOADERHOOKS) {
                this.readCurrent = this.hooksController.readCurrent();
                //just to display on the GUI the position of the clamp
                position = this.hooksController.readPosition();
                force = this.readForceSensor();
            }
        } catch (CanOpenCallTimeoutException ex) {
            this.raiseWarning(getName() + ":" + CAN_BUS_TIMEOUT, CAN_BUS_TIMEOUT.getLongDescription(),ex);
        }
        //TODO read the force sensor for the actions GOTOCLAMPEDPOSITION and CLAMP
    }

    @Override
    public void startAction(MobileItemAction action)  {
        
        if (action == MobileItemAction.GOTOCLAMPEDPOSITION) {
            controllerWriteCurrent(this.currentToClamp, paramsForCurrentToClamp);

        } else if (action == MobileItemAction.GOTOHOMEPOSITION) {
            controllerWriteCurrent(this.currentToGoHome, paramsForCurrentToGoHome);

        } else if (action == MobileItemAction.CLOSELOADERHOOKS) {
            controllerWriteAbsoluteTargetPosition(this.targetPositionToClose);

        } else if (action == MobileItemAction.CLAMPLOADERHOOKS) {
            if (!(this.lockStatus.equals(LockStatus.LOCKED))) {
                throw new RejectedCommandException(getName() + 
                        ": Hooks have to be LOCKED before being LOCKED STRONGLY");
            }
            controllerWriteCurrent(this.currentToClamp, paramsForCurrentToClamp);
            
        } else if (action == MobileItemAction.OPENLOADERHOOKS) {
            controllerWriteAbsoluteTargetPosition(this.targetPositionToOpen);

        }
    }
    
    private void controllerWriteCurrent(int current, Map<String, Integer> paramsForCurrent) {
        hooksController.enable();
        hooksController.changeMode(EposMode.CURRENT);
        hooksController.writeParameters(paramsForCurrent);
        hooksController.writeCurrent(current);
    }
    
    private void controllerWriteAbsoluteTargetPosition(int pos) {
        hooksController.enable();
        hooksController.changeMode(EposMode.PROFILE_POSITION);
        hooksController.writeParameters(EposMode.PROFILE_POSITION);
        hooksController.writeTargetPosition(pos);
        hooksController.writeControlWord("3F");
    }

    @Override
    public void abortAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(getName() + " is ABORTING action " + action.toString()
                + " within delay " + delay);
        this.hooksController.off();
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(getName() + " is STOPPING action " + action.toString()
                + " within delay " + delay);
        this.hooksController.quickStop();
    }

    @Override
    public void postAction(MobileItemAction action)  {

        //because we don't want to let the controller on power
        this.hooksController.off();
        FCSLOG.info(getName() + ":" + action.toString() + " completed - doing postAction.");
        
        if (action == MobileItemAction.GOTOCLAMPEDPOSITION) {
            checkLOCKED();
            postActionGOTOCLAMPEDPOSITION();
            
        } else if (action == MobileItemAction.GOTOHOMEPOSITION) {
            checkUNLOCKED();
            postActionGOTOHOMEPOSITION();
       
        } else if (action == MobileItemAction.CLOSELOADERHOOKS) {
            checkLOCKED();
            
        } else if (action == MobileItemAction.OPENLOADERHOOKS) {
            checkUNLOCKED();
            
        } else if (action == MobileItemAction.CLAMPLOADERHOOKS &&
                this.isLocked() && this.isAtClampedPosition()) {
            this.lockStatus = FcsEnumerations.LockStatus.CLAMPED;
            this.publishData();
        }
    }
    
    private void postActionGOTOCLAMPEDPOSITION() {
        this.hooksController.defineAbsolutePosition(this.targetPositionToClamp);
        //TODO check is useful to do it twice
        this.hooksController.off();
        // for the GUI
        updatePosition(); 
        //for the GUI
        this.lockStatus = LockStatus.CLAMPED; 
        this.initialized = true;
        this.publishData();
        loader.updateFCSState();
    }
    
    private void postActionGOTOHOMEPOSITION() {
        this.hooksController.defineAbsolutePosition(0);
        //TODO check is useful to do it twice
        this.hooksController.off();
        // for the GUI
        updatePosition(); 
        this.initialized = true;
        this.publishData();
        loader.updateFCSState();
    }
    
    private void checkLOCKED() {
        if (!this.isLocked()) {
            throw new FailedCommandException(getName()
                    + ": check with sensors: clamp should be LOCKED.");
        }
    }
     
    private void checkUNLOCKED() {
        if (!this.isUnlocked()) {
            throw new FailedCommandException(getName()
                    + ": check with sensors: clamp should be UNLOCKED.");
        }
    }


    /**
     * This methods updates the lockStatus from an array of hexaValues.
     *
     * @param readHexaValues
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    void updateStateWithSensors(String[] readHexaValues)  {
        lock.lock();
        try {
            updatingState = true;
            updateHooksSensorsValue(readHexaValues);
            boolean locked = allHooksInState(LockStatus.LOCKED);

            if (oneHookInError()) {
                this.lockStatus = LockStatus.ERROR;
                
            } else if (locked) {
                this.lockStatus = LockStatus.LOCKED;
                
            } else if (allHooksInState(LockStatus.UNLOCKED)) {
                this.lockStatus = LockStatus.UNLOCKED;
                
            } else if (allHooksInState(LockStatus.INTRAVEL)) {
                this.lockStatus = LockStatus.INTRAVEL;
                
            } else {
                this.lockStatus = LockStatus.UNKNOWN;
            }

            if (initialized && locked && isAtClampedPosition()) {
                this.lockStatus = LockStatus.CLAMPED;
            }

        } finally {
            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
        }
    }
    
    private void updateHooksSensorsValue(String[] hexaValues) {
        for (LoaderHookModule hook: hooks) {
            hook.updateStateWithSensors(hexaValues);
        }
    }
    
    private boolean oneHookInError() {
        boolean bool = false;
        for (LoaderHookModule hook: hooks) {
            bool = bool || hook.getLockStatus() == LockStatus.ERROR;
        }
        return bool;
    }
    
    private boolean allHooksInState(LockStatus status) {
        boolean bool = true;
        for (LoaderHookModule hook: hooks) {
            bool = bool && hook.getLockStatus() == status;
        }
        return bool;
    }
    
    
    //TODO put here the right code to read force sensor.
    private int readForceSensor() {
        return 0;
    }   

    /**
     * Updates the field readCurrent of the clamp in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update clamp current in reading controller.")
    public void updateCurrent() {
        try {
            this.readCurrent = hooksController.readCurrent();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> ERROR IN READING CONTROLLER:",ex);
        }
        this.publishData();
    }

    /**
     * Updates the position of the clamp in reading the CPU of the controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Update clamp position in reading controller.")
    public void updatePosition()  {
        try {
            this.position = hooksController.readPosition();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> ERROR IN READING CONTROLLER:",ex);

        }
        this.publishData();
    }

    /**
     * Lists and display hooks ino.
     * For end user at the console.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "List and display clamp info.")
    public String listHooks() {
        StringBuilder sb = new StringBuilder("Hooks status:");
        sb.append("\n");
        for (LoaderHookModule hook: hooks) {
        sb.append(hook.toString());
        sb.append("/");
        sb.append(hook.getLockStatus());
        sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * List and display clamp sensors values.
     * for end user
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "List and display clamp sensors values.")
    public String listSensorsValues() {
        StringBuilder sb = new StringBuilder(getName() + "Sensors values=");
        sb.append("/");
        for (LoaderHookModule hook: hooks) {
        sb.append(hook.listSensorsValues());
        sb.append("/");
        }
        return sb.toString();
    }

    
    /**
     * Return a printed list of hardware with initialization information.
     * For debug purpose.
     * @return 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware with initialization information.")    
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(getName());
        if (initialized) {
            sb.append(" is INITIALIZED.");
        } else {
            sb.append(" is NOT INITIALIZED.");
        }
        return sb.toString();
    }    

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "List and display clamp info.")
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("/timeoutForClosingHooks=");
        sb.append(this.timeoutForClosingHooks);
        sb.append("/timeoutForClosingHooksStrongly=");
        sb.append(this.timeoutForClampingHooks);
        sb.append("/timeoutForOpeningHooks=");
        sb.append(this.timeoutForOpeningHooks);
        return sb.toString();
    }

    
    /**
     * Creates and returns the object to be published on the STATUS bus by the LoaderClampModule.
     * @return 
     */
    public StatusDataPublishedByLoaderClamp createStatusDataPublishedByLoaderClamp() {
        StatusDataPublishedByLoaderClamp status = new StatusDataPublishedByLoaderClamp();
        status.setName(getName());
        status.setPosition(position);
        status.setCurrent(readCurrent);
        status.setClampState(lockStatus);
        status.setForce(force);
        status.setHomingDone(initialized);
        status.setStatusPublishedByHook1(hook1.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook2(hook2.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook3(hook3.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook4(hook4.createStatusDataPublishedByLoaderHook());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        StatusDataPublishedByLoaderClamp status = this.createStatusDataPublishedByLoaderClamp();
        KeyValueData kvd = new KeyValueData("loaderClamp", status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }
}
