
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This is a model for the Inclinometer used in CPPM test bench and on 
 * whole size prototype.
 * 
 * Because for mechanical raisons, the filter changer system X axis has to be horizontal
 * in order to be able to rotate carousel or move autochanger trucks along its rails.
 * So we have 2 inclinometers which give redondant information. We must read permanently
 * the 2 inclinometers to check that inclinaison is correct. 
 * We read the inclinaison on the plutoGateway.
 *
 * @author virieux
 */
public class InclinometerModule extends Module {

    private final String plutoGatewayName;
    private PlutoGatewayInterface plutoGateway;
    private double inclinaison;

    /**
     * Build an InclinometerModule with the name of the plutoGateway.
     * 
     * @param plutoGatewayName 
     */
    public InclinometerModule(String plutoGatewayName) {
        this.plutoGatewayName = plutoGatewayName;
    }

    @Override
    public void initModule() {
        this.plutoGateway = (PlutoGatewayInterface) getComponentLookup().getComponentByName(this.plutoGatewayName);
    }

    /**
     * Read the inclinaison on the plutoGateway, updates the filed inclinaison
     * and returns its value in degrees.
     *
     * @return inclinaison
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads the inclinometer and returns inclinaison.")
    public double readInclinaison()  {
        updateInclinaison();
        return inclinaison;
    }

    /**
     * Read the inclinaison on the plutoGateway, updates the field inclinaison.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads the inclinometer and returns inclinaison.")
    public void updateInclinaison()  {
        //the value read from the plutoGateway is in tenth of volts
        //so it has to be divided by ten to obtain the value in volt
        inclinaison = convertVoltage2Degree(plutoGateway.readNewAnalogValue() / 10);
    }

    /**
     * Concert a voltage in volt between 0.5 and 4.5 volts.
     *
     * @param voltage
     * @return
     */
    public static double convertVoltage2Degree(int voltage) {
        if (voltage < 0 || voltage > 5) {
            throw new IllegalArgumentException("Voltage has to be between 0.5 and 4.5.");
        }
        //voltage is in volt
        double sin_angle = (voltage - 2.5) / 11.52;
        return Math.toDegrees(Math.asin(sin_angle));
    }

}
