
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;

/**
 * This class is a model for the fakeAutochanger that we use on loader and carousel test bench 
 * in standalone mode.
 * 
 * As there is no autochanger, a numeric sensor is setup to simulate that the autochanger holds the filter
 * at STANDBY or HANDOFF position.
 * It's also a model for the fakeCarousel and the fakeLoader in autochanger-standalone subsystem.
 * 
 * FakeFilterHolder implements interface FilterHolder.
 * In the whole FCS, carousel, autochanger and loader implements FilterHolder too.
 * 
 * @author virieux
 */
public class FakeFilterHolder extends Module implements FilterHolder {
    
    @ConfigurationParameter(isFinal = true)
    private String plutoGatewayName;
    
    private PlutoGatewayInterface plutoGateway;
        
    private ComplementarySensors holdingFilterSensors;
    private NumericSensor holdingFilterSensor;
    
    private boolean holdingFilter = false;
    
    private final boolean onlyOneSensor;

    //Used because we have to wait for the update from the sensors
    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

    /**
     * Build a FakeFilterHolder with a pluto gateway name and a numeric sensor.
     * @param plutoGatewayName
     * @param holdingFilterSensor0 
     * @param holdingFilterSensor1 
     */
    public FakeFilterHolder(String plutoGatewayName,
            NumericSensor holdingFilterSensor0,
            NumericSensor holdingFilterSensor1) {
        this.plutoGatewayName = plutoGatewayName;
        this.holdingFilterSensors = new ComplementarySensors(holdingFilterSensor0,holdingFilterSensor1);
        this.onlyOneSensor = false;
    }

    /**
     * A constructor with only one filter
     * @param plutoGatewayName
     * @param holdingFilterSensor 
     */
    public FakeFilterHolder(String plutoGatewayName, NumericSensor holdingFilterSensor) {
        this.plutoGatewayName = plutoGatewayName;
        this.holdingFilterSensor = holdingFilterSensor;
        this.onlyOneSensor = true;
    }
    
    

    /**
     * initialize plutoGateway
     */
    @Override
    public void initModule() {
        this.plutoGateway = (PlutoGatewayInterface) getComponentLookup().getComponentByName(plutoGatewayName);
    }
    
    /**
     * Return true if the fake filter holder is holding the filter.
     * This command doesn't read again the sensor.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the fake autochanger is holding the filter. "
                    + "This command doesn't read again the sensor.")
    @Override
    public boolean isHoldingFilter() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(getName() + ": has been interrupted while waiting for end of update.");
                }

            }
            return holdingFilter;

        } finally {
            lock.unlock();
        }
    }
    
    
    /**
     * This methods updates the field holdingFilter in reading the holding filter sensor.
     * Then it published status data on the STATUS bus.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors()  {
        
        plutoGateway.checkBooted();
        plutoGateway.checkInitialized();
        
        lock.lock();
        try {
            updatingState = true;
            this.plutoGateway.updateValues();
            computeState(this.plutoGateway.getHexaValues());
        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
        }
    }
    
    private void computeState(String[] hexaValues) {
        if (onlyOneSensor) {
            this.holdingFilterSensor.updateValue(hexaValues);
            this.holdingFilter = this.holdingFilterSensor.getDigitalValue() == 1;
        } else {
            this.holdingFilterSensors.updateValues(hexaValues);
            this.holdingFilter = holdingFilterSensors.isOn();
        }
    }
    
    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        KeyValueData kvd;
        if (onlyOneSensor) {
            kvd = new KeyValueData(getName(), holdingFilterSensor.getDigitalValue());
        } else {
            kvd = new KeyValueData(getName(), holdingFilterSensors.getDigitalValue());
        }
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }

    /**
     * For the fake autochanger there is no autochanger trucks, so the position has not a real meaning.
     * The autochanger is supposed to be at HANDOFF.
     * Idem for fakeCarousel and fakeLoader.
     * @return 
     */
    @Override
    public boolean isAtHandoff() {
        return true;
    }

    /**
     * For the fake autochanger there is no autochanger trucks, so the position has not a real meaning.
     * The autochanger is supposed to be at STANDBY.
     * Idem for fakeCarousel and fakeLoader.
     * @return 
     */
    @Override
    public boolean isAtStandby() {
        return true;
    }

}
