
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;

/**
 * A model for an autochanger truck. 
 * The goal of this class is to factorise the code which reads AC truck sensors.
 * On the Autochanger there are 2 trucks : one on each side : 
 * truckXminus and truckXplus.
 * @author virieux
 */
public class AutochangerTruckModule extends Module implements MovedByEPOSController {
    
    //Used because we have to wait for the update from the sensors to know if the action is completed.
    protected final Lock lock = new ReentrantLock();
    
    protected NumericSensor handoffPositionSensor;
    protected NumericSensor handoffPositionSensorB;
    protected NumericSensor onlinePositionSensor;
    protected NumericSensor onlinePositionSensorB;
    protected NumericSensor standbyPositionSensor;
    protected NumericSensor standbyPositionSensorB;
    private final ComplementarySensors handoffPositionSensors;
    private final ComplementarySensors onlinePositionSensors;
    private final ComplementarySensors standbyPositionSensors;
    
    private int position;
    private EPOSController controller;
    
    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    
    /**
     * true if the 2 position sensors value at Handoff is 1 *
     */
    private boolean atHandoff = false;

    /**
     * true if the 2 position sensors value at Online is 1 *
     */
    private boolean atOnline = false;

    /**
     * true if the 2 position sensors value at Standby is 1 *
     */
    private boolean atStandby = false;

    /**
     * true if the 2 position sensors value at Standby differ *
     */
    private boolean standbySensorsInError = false;

    /**
     * true if the 2 position sensors value at Handoff differ *
     */
    private boolean handoffSensorsInError = false;

    /**
     * true if the 2 position sensors value at Handoff differ *
     */
    private boolean onlineSensorsInError = false;  
    
    private boolean controllerInFault;


    public AutochangerTruckModule(NumericSensor handoffPositionSensor, 
            NumericSensor handoffPositionSensorB, 
            NumericSensor onlinePositionSensor, 
            NumericSensor onlinePositionSensorB, 
            NumericSensor standbyPositionSensor, 
            NumericSensor standbyPositionSensorB) {
        this.handoffPositionSensor = handoffPositionSensor;
        this.handoffPositionSensorB = handoffPositionSensorB;
        this.onlinePositionSensor = onlinePositionSensor;
        this.onlinePositionSensorB = onlinePositionSensorB;
        this.standbyPositionSensor = standbyPositionSensor;
        this.standbyPositionSensorB = standbyPositionSensorB;
        handoffPositionSensors = new ComplementarySensors(this.handoffPositionSensor, 
                this.handoffPositionSensorB);
        onlinePositionSensors = new ComplementarySensors(this.onlinePositionSensor, 
                this.onlinePositionSensorB); 
        standbyPositionSensors = new ComplementarySensors(this.standbyPositionSensor, 
                this.standbyPositionSensorB);        
    }

    /**
     * @return true if truck is at HANDOFF position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if truck is at HANDOFF position.")    
    public boolean isAtHandoff() {
        return atHandoff;
    }
    
    /**
     * @return true if truck is at ONLINE position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if truck is at ONLINE position.")
    public boolean isAtOnline() {
        return atOnline;
    }

    /**
     * @return true if truck is at STANDBY position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if truck is at STANDBY position.")    
    public boolean isAtStandby() {
        return atStandby;
    }
    
    

    /**
     * Return false if the 2 redondant position sensors at Standby are equal.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Standby are equal. "
                    + "Doesn't read again sensors.")
    public boolean isStandbySensorsInError() {
        return standbySensorsInError;
    }

    /**
     * Return false if the 2 redondant position sensors at Handoff are equal.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Handoff are equal."
                    + "Doesn't read again sensors.")
    public boolean isHandoffSensorsInError() {
        return handoffSensorsInError;
    }

    /**
     * Return false if the 2 redondant position sensors at Online are equal.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Online are equal."
                    + "Doesn't read again sensors.")
    public boolean isOnlineSensorsInError() {
        return onlineSensorsInError;
    }
    
    /**
     * Return true if position sensors are in error.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
        description = "Return true if position sensors are in error."
                + "Doesn't read again sensors.")
    public boolean isPositionSensorsInError() {
        return onlineSensorsInError || handoffSensorsInError || standbySensorsInError;
    }   

    public NumericSensor getHandoffPositionSensor() {
        return handoffPositionSensor;
    }

    public NumericSensor getHandoffPositionSensorB() {
        return handoffPositionSensorB;
    }

    public NumericSensor getOnlinePositionSensor() {
        return onlinePositionSensor;
    }

    public NumericSensor getOnlinePositionSensorB() {
        return onlinePositionSensorB;
    }

    public NumericSensor getStandbyPositionSensor() {
        return standbyPositionSensor;
    }

    public NumericSensor getStandbyPositionSensorB() {
        return standbyPositionSensorB;
    }

    /**
     * Return truck position enregistred in field position.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return truck position.")    
    public int getPosition() {
        return position;
    }
    
    /**
     * @return controller name.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return controller name.")    
    @Override
    public String getControllerName() {
        return this.controller.getName();
    }

    /**
     * Return true if controller is in fault.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if the controller is in fault.")    
    @Override
    public boolean isControllerInFault() {
        return controllerInFault;
    }

    @Override
    public void setControllerInFault(boolean controllerInFault) {
        this.controllerInFault = controllerInFault;
    }

    
    @Override
    public void initModule() {
        ComponentLookup lookup = getComponentLookup();
        if (getName().contains("Xminus")) {
            controller = (EPOSController) lookup.getComponentByName("linearRailSlaveController");
        } else if (getName().contains("Xplus")) {
            controller = (EPOSController) lookup.getComponentByName("linearRailMasterController");
        } else {
            FCSLOG.error("ERROR in Autochanger trucks name. Can't find a controller for truck: " + this.getName());
        }
    }
    
    /**
     * Updates the field position of the truck in reading the CPU of the
     * controller.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update truck position in reading controller.")
    public void updatePosition()  {
        this.position = controller.readPosition();
        this.publishData();
    }    
    
    /**
     * This methods updates the lockStatus from an array of hexaValues.
     *
     * @param readHexaValues
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    //TODO test with real hardware
    protected void updateStateWithSensors(String[] readHexaValues) {
        lock.lock();

        try {
            updatingState = true;

            this.handoffPositionSensors.updateValues(readHexaValues);
            this.onlinePositionSensors.updateValues(readHexaValues);
            this.standbyPositionSensors.updateValues(readHexaValues);

            //TODO compute a state 
            this.atHandoff = handoffPositionSensors.isOn();
            this.atOnline = onlinePositionSensors.isOn();
            this.atStandby = standbyPositionSensors.isOn();

            this.handoffSensorsInError = handoffPositionSensors.isInError();
            this.onlineSensorsInError = onlinePositionSensors.isInError();
            this.standbySensorsInError = standbyPositionSensors.isInError();

        } finally {
            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }
    }
    

    
    /**
     * Creates an object to be published on the status bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerTruck createStatusDataPublishedByAutoChangerTruck() {
        StatusDataPublishedByAutochangerTruck s = new StatusDataPublishedByAutochangerTruck();
        s.setHandoffInError(this.handoffSensorsInError);
        s.setOnlineInError(this.onlineSensorsInError);
        s.setStandbyInError(this.standbySensorsInError);
        s.setHandoffSensorValue(handoffPositionSensors.isOn());
        s.setOnlineSensorValue(onlinePositionSensors.isOn());
        s.setStandbySensorValue(standbyPositionSensors.isOn());
        s.setPosition(position);
        s.setControllerInFault(controllerInFault);
        return s;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), 
                createStatusDataPublishedByAutoChangerTruck()));
    }    
    
}
