
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * The Three Online clamps which holds a filter when it is at ONLINE position
 * can ve viewed as a single object. It's the goal of this class to represent
 * this object. Opening or closing the 3 clamps at ONLINE is not the same action
 * as opening or closing online clamps one by one.
 *
 * @author virieux
 */
public class AutochangerThreeOnlineClamps extends MobileItemModule {

    private AutoChangerModule autochanger;
    private final AutochangerOnlineClampModule onlineClampXminus;
    private final AutochangerOnlineClampModule onlineClampXplus;
    private final AutochangerOnlineClampModule onlineClampYminus;

    private FcsEnumerations.LockStatus lockStatus = LockStatus.UNKNOWN;


    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamps
     state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    @ConfigurationParameter(description="timeout in milliseconds : if closing the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForLockingClamps = 15000;
    
    @ConfigurationParameter(description="timeout in milliseconds : if unlocking the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.")
    private long  timeoutForUnlockingClamps = 15000;

    /**
     * Create a AutochangerThreeOnlineClamps with 3 AutochangerOnlineClampModule.
     * @param onlineClampXminus
     * @param onlineClampXplus
     * @param onlineClampY
     */
    public AutochangerThreeOnlineClamps(
            AutochangerOnlineClampModule onlineClampXminus,
            AutochangerOnlineClampModule onlineClampXplus,
            AutochangerOnlineClampModule onlineClampY) {
        this.onlineClampXminus = onlineClampXminus;
        this.onlineClampXplus = onlineClampXplus;
        this.onlineClampYminus = onlineClampY;
    }

    /**
     * Return lockStatus
     * If lockStatus is being updated, wait until update completion else return immediately lockStatus.
     * @return 
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(getName() + ": interrupted in getLockStatus.",ex);
                }

            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }
    
    /**
     * Returns true if LockStatus=LOCKED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the 3 clamps are CLOSED.")
    public boolean isLocked() {
        return onlineClampXminus.isClosed()
                && onlineClampXplus.isClosed()
                && onlineClampYminus.isClosed();
    }
    
    /**
     * Returns true if LockStatus=UNLOCKED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the 3 clamps are OPENED.")
    public boolean isUnlocked() {
        return onlineClampXminus.isOpened()
                && onlineClampXplus.isOpened()
                && onlineClampYminus.isOpened();
    }
    
    /**
     * @return true if the 3 clamps are in Travel between opened position and closed position.
     */
    private boolean isInTravel() {
        return onlineClampXminus.isInTravel()
                && onlineClampXplus.isInTravel()
                && onlineClampYminus.isInTravel();
    }
    
    /**
     * Returns true if LockStatus=ERROR
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if one of the clamp is in error.")
    public boolean isInError() {
        return onlineClampXminus.isInError()
                || onlineClampXplus.isInError()
                || onlineClampYminus.isInError();
    }
    
    

    @Override
    public void initModule() {
        this.autochanger = (AutoChangerModule) getComponentLookup().getComponentByName("autochanger");
        this.lockStatus = FcsEnumerations.LockStatus.UNKNOWN;
    }
    
    /**
     * Return true if the 3 onlineClamps hardware is initialized. 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if the 3 onlineClamps hardware is ready : controllers ready, "
                + "controllers parameters checked and controllers configured.")
    public boolean isInitialized() {
        return onlineClampXminus.isInitialized() && onlineClampXplus.isInitialized() 
                && onlineClampYminus.isInitialized();
    }
    
    /**
     * Lock the online clamps.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Lock the online clamps.")
    public void lockClamps()  {
        if (isLocked()) {
            throw new RejectedCommandException(name + " is already LOCKED");
        }
        this.executeAction(MobileItemAction.LOCK_ONLINECLAMPS, timeoutForLockingClamps);
    }

    /**
     * Unlock the online clamps.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Unlock the online clamps.")
    public void unlockClamps()  {
        if (isUnlocked()) {
            throw new RejectedCommandException(name + " is already UNLOCKED");
        }        
        this.executeAction(MobileItemAction.UNLOCK_ONLINECLAMPS, timeoutForUnlockingClamps);
    }
    
    /**
     * To Unlock 3 clamps.
     * Open progressively the 3 clamps.
     * onlineClampYminus last.
     * @throws HardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Unlock the 3 online clamps.")   
    public void testUnlockClamps() throws HardwareException {
        /* do a ramp of current until currentToClamp/2*/
        long period = 500;
        
        onlineClampYminus.checkControllerBeforeAction();
        onlineClampXminus.checkControllerBeforeAction();
        onlineClampXplus.checkControllerBeforeAction();
        onlineClampYminus.executeCurrentRamp(0, onlineClampYminus.getCurrentToClamp()/2, 4, period);
        
        onlineClampXminus.executeCurrentRamp(0, onlineClampXminus.getCurrentToClamp()/2, 4, period);
        onlineClampXplus.executeCurrentRamp(0, onlineClampXplus.getCurrentToClamp()/2, 4, period);
        autochanger.updateStateWithSensors();
        /* continue until currentToRamp is reached for the 3 clamps in 2 steps*/
        onlineClampYminus.executeCurrentRamp(onlineClampYminus.getCurrentToClamp()/2, 
                onlineClampYminus.getCurrentToClamp(), 2, period);
        onlineClampXminus.executeCurrentRamp(onlineClampXminus.getCurrentToClamp()/2, 
                onlineClampXminus.getCurrentToClamp(), 2, period);
        onlineClampXplus.executeCurrentRamp(onlineClampXplus.getCurrentToClamp()/2, 
                onlineClampXplus.getCurrentToClamp(), 2, period);
        autochanger.updateStateWithSensors();
        onlineClampYminus.postAction(MobileItemAction.CLOSE_ONLINECLAMP);
        onlineClampXminus.postAction(MobileItemAction.CLOSE_ONLINECLAMP);
        onlineClampXplus.postAction(MobileItemAction.CLOSE_ONLINECLAMP);
    }
    
    
    /**
     * Lock the 3 online clamps progressively.
     * OnlineClampYminus first.
     * @throws HardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open the 3 online clamps.")   
    //tested in december 2016
    //TODO do same thing with MobileItemModule.
    public void testLockClamps() throws HardwareException {
            long period = 500;
            onlineClampYminus.checkControllerBeforeAction();
            onlineClampXminus.checkControllerBeforeAction();
            onlineClampXplus.checkControllerBeforeAction();
            
            onlineClampYminus.sendCurrentToController(onlineClampYminus.getCurrentToClose());
            onlineClampXminus.sendCurrentToController(onlineClampXminus.getCurrentToClose());
            onlineClampXplus.sendCurrentToController(onlineClampXplus.getCurrentToClose());
            
            autochanger.updateStateWithSensors();
            
            /* do a ramp of current until currentToClamp/2*/
            onlineClampYminus.executeCurrentRamp(onlineClampYminus.getCurrentToClose(), 
                    onlineClampYminus.getCurrentToClamp()/2, 4, period);
            
            currentRampTwoClamps(-2000,2000, 4, period);
            
            autochanger.updateStateWithSensors();
            
            try {
                /* continue until currentToOpen is reached for the 3 clamps in 2 steps*/
                onlineClampXminus.sendCurrentToController(-2500);
                onlineClampXplus.sendCurrentToController(2500);
                onlineClampYminus.sendCurrentToController(-3000);
                Thread.sleep(period);
                onlineClampXminus.sendCurrentToController(-3000);
                onlineClampXplus.sendCurrentToController(3000);
                onlineClampYminus.sendCurrentToController(-3000);
            } catch (InterruptedException ex) {
               FCSLOG.error(getName()+ "interrupted during sleep:"+ex);
            }
            autochanger.updateStateWithSensors();
            onlineClampXminus.postAction(MobileItemAction.OPEN_ONLINECLAMP);
            onlineClampXplus.postAction(MobileItemAction.OPEN_ONLINECLAMP);
            onlineClampYminus.postAction(MobileItemAction.OPEN_ONLINECLAMP);

    }   
    
    private void currentRampTwoClamps(int finalValueXminus, int finalValueXplus, int nbSteps,long period) {
        int stepHeightXminus = finalValueXminus/nbSteps;
        int currentValueXminus;
        int stepHeightXplus = finalValueXplus/nbSteps;
        int currentValueXplus;
        for (int i = 1; i < nbSteps; i++) {
            try {
                /* slowly goes to -2A*/
                currentValueXminus = stepHeightXminus * i;
                onlineClampXminus.sendCurrentToController(currentValueXminus);
                
                /* slowly goes to +2A*/
                currentValueXplus = stepHeightXplus * i;
                onlineClampXplus.sendCurrentToController(currentValueXplus);
                Thread.sleep(period);
            } catch (InterruptedException ex) {
                throw new FcsHardwareException("testUnlockClamps was interrupted while sleeping "+ex);
            }
        }
    }

    /**
     * Closes the 3 clamps with a small pressure.
     * @throws HardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Close the 3 online clamps.")
    public void testCloseClamps() throws HardwareException {
        onlineClampXminus.close();
        onlineClampXplus.close();
        onlineClampYminus.close();
    }
    
    /**
     * Opens the 3 clamps with a small pressure.
     * @throws HardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Opens the 3 online clamps with a small pressure.")
    public void testOpenClamps() throws HardwareException {
        onlineClampXminus.open();
        onlineClampXplus.open();
        onlineClampYminus.open();
    }

    /**
     * Return true if the 3 onlineClamps hardware is ready. 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if the 3 onlineClamps hardware is ready.")
    @Override
    public boolean isCANDevicesReady() {
        return onlineClampXminus.isCANDevicesReady() && onlineClampXplus.isCANDevicesReady() 
                && onlineClampYminus.isCANDevicesReady();
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (action == MobileItemAction.LOCK_ONLINECLAMPS) {
            return isLocked(); 
            
        } else if (action == MobileItemAction.UNLOCK_ONLINECLAMPS) {
            return isUnlocked(); 
            
        } else {
            throw new IllegalArgumentException(name + " invalid action:"+action);
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        autochanger.updateStateWithSensors();
    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        if (action == MobileItemAction.LOCK_ONLINECLAMPS) {
            onlineClampXminus.getController().enable();
            onlineClampXplus.getController().enable();
            onlineClampYminus.getController().enable();
            
            onlineClampXminus.getController().releaseBrake();
            onlineClampXplus.getController().releaseBrake();
            onlineClampYminus.getController().releaseBrake();
            
            onlineClampXminus.getController().writeCurrent(onlineClampXminus.getCurrentToClose());
            onlineClampXplus.getController().writeCurrent(onlineClampXplus.getCurrentToClose());
            onlineClampYminus.getController().writeCurrent(onlineClampYminus.getCurrentToClose());
            
        } else if (action == MobileItemAction.UNLOCK_ONLINECLAMPS) {
            onlineClampXminus.getController().enable();
            onlineClampXplus.getController().enable();
            onlineClampYminus.getController().enable();

            onlineClampXminus.getController().writeCurrent(onlineClampXminus.getCurrentToClamp());
            onlineClampXplus.getController().writeCurrent(onlineClampXplus.getCurrentToClamp());
            onlineClampYminus.getController().writeCurrent(onlineClampYminus.getCurrentToClamp());
            
            onlineClampXminus.getController().releaseBrake();
            onlineClampXplus.getController().releaseBrake();
            onlineClampYminus.getController().releaseBrake();
            //TODO: slowDownCurrent ?

            onlineClampXminus.getController().writeCurrent(onlineClampXminus.getCurrentToOpen());
            onlineClampXplus.getController().writeCurrent(onlineClampXplus.getCurrentToOpen());
            onlineClampYminus.getController().writeCurrent(onlineClampYminus.getCurrentToOpen());
        } else {
            throw new IllegalArgumentException(name + " invalid action:"+action);
        } 
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay)  {
            onlineClampXminus.abortAction(action,delay);
            onlineClampXplus.abortAction(action,delay);
            onlineClampYminus.abortAction(action,delay);
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay)  {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action)  {
        //nothing to do 
    }
    
    /**
     * Creates an object to be published on the status bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerThreeClamps createStatusDataPublishedByThreeClamps() {
        StatusDataPublishedByAutochangerThreeClamps status = new StatusDataPublishedByAutochangerThreeClamps();
        status.setName(getName());
        status.setLockStatus(lockStatus);
        return status;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(
                new KeyValueData("autochangerClamps", createStatusDataPublishedByThreeClamps()));
    }


    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the
     * method.
     *
     * @param hexaValues
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors(String[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;
            this.onlineClampXminus.updateStateWithSensors(hexaValues);
            this.onlineClampXplus.updateStateWithSensors(hexaValues);
            this.onlineClampYminus.updateStateWithSensors(hexaValues);

            computeLockStatus();

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }
    }
    
    /**
     * Compute lockStatus.
     */
    private void computeLockStatus() {
        if (isInError()) {
            this.lockStatus = LockStatus.ERROR;
            
        } else if (isLocked()) {
            this.lockStatus = LockStatus.LOCKED;
            
        } else if (isUnlocked()) {
            this.lockStatus = LockStatus.UNLOCKED;
            
        } else if (isInTravel()) {
            this.lockStatus = LockStatus.INTRAVEL;
            
        } else {
            /* if lockStatus is not the same for the 3 clamps we are in this case.
            It can happen during the closing (or the opening) of the 3 clamps when a clamp is already closed 
            but not the 2 others. 
            */
            this.lockStatus = LockStatus.UNKNOWN;
        }
    }

}
