
package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import static org.lsst.ccs.subsystems.fcs.utils.FcsUtils.TICKMILLIS;

/**
 * This is a model for online clamp which holds a filter when it is at ONLINE
 * position. The online clamps are part of the autochanger subsystem. There is 3
 * online clamps.
 *
 * @author virieux
 */
public class AutochangerOnlineClampModule extends MobileItemModule implements MovedByEPOSController {
    
    private static final int CURRENT_RAMP_PERIOD = 500;

    private final int timeoutForClosing = 10000;
    private final int timeoutForOpening = 10000;
    private AutoChangerModule autochanger;
    private final EPOSController controller;
    
    /* couple of close sensors*/
    private final ComplementarySensors closeSensors;
    
    /* couple of open sensors*/
    private final ComplementarySensors openSensors;
    
    @ConfigurationParameter(description="current to close ONLINE clamp in mA")
    private int currentToClose = 0; 
    
    @ConfigurationParameter(description="current to clamp ONLINE clamp, in mA")
    private int currentToClamp = 0;
         
    @ConfigurationParameter(description="current to close ONLINE clamp in mA")
    private int currentToOpen = - currentToClose;
    
    @ConfigurationParameter(description="time to hold clamp, in millis")
    private long holdTime = 5000;
    
    @ConfigurationParameter(description="time for clamp to lock, in millis")
    private long travelTime = 5000;

    /**
     * For the onlineClamp the lockStatus is OPENED or CLOSED.
     * We don't known if the clamp is locked or not because there is no force sensor.
     */
    private FcsEnumerations.LockStatus lockStatus = LockStatus.UNKNOWN;
    
    private boolean controllerInFault;

    private volatile boolean initialized;
    private boolean controllerConfigured = false;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the latch state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    
    /* Tools for newCurrentValue ramp*/
    private ScheduledFuture<?> currentRampHandle;
    protected final Condition currentRampEnded = lock.newCondition();
    protected boolean hasToWaitForEndOfCurrentRamp;
    /* end of Tools for newCurrentValue ramp*/
    
    /**
     * Build an AutochangerOnlineClampModule with a controller, 4 sensors and 
     * 2 maps of parameters for the controller.
     * @param controller
     * @param closeSensor
     * @param closeSensorC
     * @param openSensor
     * @param openSensorC
     * @param currentToClose
     */
    public AutochangerOnlineClampModule(
            EPOSController controller,
            NumericSensor closeSensor,
            NumericSensor closeSensorC,
            NumericSensor openSensor,
            NumericSensor openSensorC,
            int currentToClose) {
        super(TICKMILLIS);
        this.controller = controller;
        this.closeSensors = new ComplementarySensors(closeSensor, closeSensorC);
        this.openSensors = new ComplementarySensors(openSensor, openSensorC);
        this.currentToClose = currentToClose;
        this.currentToOpen = - currentToClose;
    }

    /**
     * Return ONLINE clamp controller
     * @return 
     */
    public EPOSController getController() {
        return controller;
    }

    /**
     * For simulator
     * @return 
     */
    public int getCurrentToClose() {
        return currentToClose;
    }

    /**
     * For simulator
     * @return 
     */
    public int getCurrentToOpen() {
        return currentToOpen;
    }

    /**
     * For simulator
     * @return 
     */
    public int getCurrentToClamp() {
        return currentToClamp;
    }

    /**
     * For simulator
     * @return 
     */
    public long getTravelTime() {
        return travelTime;
    }
    
    

    /**
     * Returns true if closeSensor and closeSensorC return the same value.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if lockSensor and lockSensorC retun the same value. "
                    + "Doesn't read again sensors.")
    public boolean isCloseSensorsInError() {
        return closeSensors.isInError();
    }

    /**
     * Returns true if openSensor and openSensorC retun the same value.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if unlockSensor and unlockSensorC retun the same value. "
                    + "Doesn't read again sensors.")
    public boolean isOpenSensorsInError() {
        return openSensors.isInError();
    }

    /**
     * Returns true if LockStatus=ERROR, this means closeSensor or openSensor is in ERROR 
     * or openSensor and closeSensor return non consistant values.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=ERROR, this means that "
            + "closeSensor or openSensor is in ERROR or that"
            + "openSensor and closeSensor return non consistant values. Doesn't read again sensors.")
    public boolean isInError() {
        return lockStatus == FcsEnumerations.LockStatus.ERROR;
    }

    /**
     * Returns true if LockStatus=CLOSED.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=CLOSED. Doesn't read again sensors.")
    public boolean isClosed() {
        return lockStatus == FcsEnumerations.LockStatus.CLOSED;
    }

    /**
     * Returns true if LockStatus=OPENED.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=OPENED. Doesn't read again sensors.")
    public boolean isOpened() {
        return lockStatus == FcsEnumerations.LockStatus.OPENED;
    }

    /**
     * Returns true if LockStatus=INTRAVEL.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=INTRAVEL. Doesn't read again sensors.")
    public boolean isInTravel() {
        return lockStatus == FcsEnumerations.LockStatus.INTRAVEL;
    }

    /**
     * Returns true if controller is in fault.
     * Doesn't read controller CPU.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if controller is in fault. Doesn't read controller CPU.")
    @Override
    public boolean isControllerInFault() {
        return controllerInFault;
    }

    /**
     * set controllerInFault
     * @param controllerInFault 
     */
    @Override
    public void setControllerInFault(boolean controllerInFault) {
        this.controllerInFault = controllerInFault;
    }
    
    /**
     * return controller name
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
        description = "Returns controller name.")
    @Override
    public String getControllerName() {
        return controller.getName();
    }

    /**
     * Returns true if clamp is initialized : hardware is ready (at least booted) and
     * clamp controller is initialized (parameters in the controller CPU have been checked and 
     * controller has been configured.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if clamp is initialized : controller is booted, parameters in the controller "
                    + "CPU have been checked and controller is configured.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * Initialize fields autochanger, lockStatus and listens to its controller.
     */
    @Override
    public void initModule() {
        super.initModule();
        this.autochanger = (AutoChangerModule) getComponentLookup().getComponentByName("autochanger");
        this.lockStatus = FcsEnumerations.LockStatus.UNKNOWN;
        
        /**
         * Because AutochangerOnlineClampModule implements MovedByEPOSController, it has to listen to its controller
         * to known when the controller is in Fault or when a Fault reset has been done.
         */
        if (controller == null) {
            FCSLOG.error(getName() + "==>>> onlineController == null - Please fix groovy description file.");
            throw new IllegalArgumentException(getName() + "==>>> null onlineClampController - fix groovy description file.");
        } else {
            //listens to my Controller to detect the controller's faultReset
            //or the emergency messages coming from the controller.
            if (controller instanceof Observable) {
                this.listens((Observable) controller);
            }
        }
    }
       

    /**
     * Return true when ONLINE clamp hardware is ready. That is when controller is initialized and configured.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY,
            level = Command.ENGINEERING1,
            description = "Returns true if controller is initialized and configured.")
    @Override
    //tested on CPPM testbench in september 2015
    public boolean isCANDevicesReady() {
        return ((MainModule) getComponentLookup().getComponentByName("main")).isCANDevicesReady()
                && controller.isInitialized() && controllerConfigured;
    }

    /**
     * Check if it's safe to open the clamp.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY,
            level = Command.ENGINEERING1,
            description = "Check if the onlineClamp can be locked.")
    public void checkConditionsForOpening()  {
        checkConditionsForClosing();
        
        if (!autochanger.isHoldingFilter()) {
            throw new RejectedCommandException(getName()+" can't be OPENED if autochanger is not HOLDING filter.");
        }
    }

    /**
     * Check if it's safe to close the clamp.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY,
            level = Command.ENGINEERING1,
            description = "Check if the onlineClamp can be locked.")
    public void checkConditionsForClosing()  {
        if (!autochanger.isAtOnline()) {
            throw new RejectedCommandException(getName()+" can't be CLOSED if autochanger trucks are not at ONLINE position.");
        } else if (autochanger.isEmpty()) {
            throw new RejectedCommandException(getName()+" can't be CLOSED if there is no filter in autochanger trucks.");
        }
    }

    /**
     * This method is called during INITIALIZATION phase. It configures the
     * controller for the digital inputs and outputs. It changes it to CURRENT
     * mode if it's not in this mode and it writes in the CPU of the controller
     * the values of configuration for CURRENT mode parameters.
     *
     * @return
     * @throws HardwareException
     */
    @Override
    //tested on CPPM testbench in september 2015
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Initialize controller and check parameters.")
    public TreeWalkerDiag checkHardware() throws HardwareException {
        super.checkHardware();

        FCSLOG.debug(getName() + " checking hardware.");

        try {
            controller.initializeAndCheckHardware();
            configureController();
            controller.changeMode(EposMode.CURRENT);
            controller.writeParameters(EposMode.CURRENT);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false, ex);

        } catch (FailedCommandException ex) {
            throw new HardwareException(true, ex);
        }
        this.initialized = true;
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * Read parameters for mode CURRENT on controller CPU.
     * If a parameter has a different value than in configuration, throws an exception.
     * 
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY,
            level = Command.ENGINEERING1,
            description = "Read in the CPU of the controller parameters for mode CURRENT."
            + "If a parameter has a different value than in configuration, "
            + "throws an exception.")
    public void checkControllerBeforeAction()  {
        controller.enable();
        controller.changeMode(EposMode.CURRENT);
        try {
            controller.checkParameters(EposMode.CURRENT);
            controller.checkFault();
        } catch (HardwareException ex) {
            String msg = getName() + " error in parameters ";
            FCSLOG.error(msg + ex);
            throw new FcsHardwareException(msg + ex);
        }
        if (!controller.isParametersOK()) {
            String msg = getName() + " Some parameter values are not"
                    + " the same in CPU and configuration system.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }

    }

    /**
     * Configure ONLINE clamp controller.
     * @throws HardwareException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION,
            level = Command.ENGINEERING1,
            description = "Configure controller.")
    //tested on CPPM testbench in september 2015
    public void configureController() throws HardwareException {
        try {
            controllerConfigured = false;
            controller.activateBrake();
            controller.shutdownEPOS();
            this.configureDigitalInputOfOnlineClamps();
            this.configureDigitalOutputOfOnlineClamps();
            controllerConfigured = true;
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName(),ex);
        }
    }


    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the
     * method.
     *
     * @param hexaValues
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors(String[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;
            
            this.closeSensors.updateValues(hexaValues);
            this.openSensors.updateValues(hexaValues);

            boolean locked = this.closeSensors.isOn();
            boolean unlocked = this.openSensors.isOn();
            boolean someSensorsInError = closeSensors.isInError() || openSensors.isInError();
            boolean inError =  someSensorsInError || (locked && unlocked);

            if (inError) {
                lockStatus = FcsEnumerations.LockStatus.ERROR;
            } else if (locked) {
                lockStatus = FcsEnumerations.LockStatus.CLOSED;
            } else if (unlocked) {
                lockStatus = FcsEnumerations.LockStatus.OPENED;
            } else {
                lockStatus = FcsEnumerations.LockStatus.INTRAVEL;
            }

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }

    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (action == MobileItemAction.CLOSE_ONLINECLAMP) {
            return this.lockStatus == LockStatus.CLOSED;
            
        } else if (action == MobileItemAction.OPEN_ONLINECLAMP) {
            return this.lockStatus == LockStatus.OPENED;
            
        } else throw new IllegalArgumentException(getName() + " invalid action for ONLINE clamp:"+action);
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        autochanger.updateStateWithSensors();
    }
    
    /**
     * Close the ONLINE clamp.
     */
    /*Tested in december 2016*/
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Close the ONLINE clamp.")
    public void close() {
        if (this.isClosed()) {
            throw new RejectedCommandException(getName() + " is already CLOSED.");
        }
        //uncommented for tests in december 2016
//        checkConditionsForClosing();
        this.executeAction(MobileItemAction.CLOSE_ONLINECLAMP, timeoutForClosing);
    }
    
    /**
     * Open the ONLINE clamp.
     * 
     */
    /*Tested in december 2016*/
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Unlock the ONLINE clamp.")
    public void open() {
        if (this.isOpened()) {
            throw new RejectedCommandException(getName() + " is already OPENED.");
        }     
        //uncommented for tests in december 2016
//        checkConditionsForOpening();
        this.executeAction(MobileItemAction.OPEN_ONLINECLAMP, timeoutForOpening);
    }
    


    /**
     * Start action of locking or unlocking.
     * @param action
     * @throws FcsHardwareException 
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        if (action == MobileItemAction.CLOSE_ONLINECLAMP) {
            sendCurrentToController(this.currentToClose);
            
        } else if (action == MobileItemAction.UNLOCK) {
            sendCurrentToController(this.currentToClamp);
            /* write a ramp of newCurrentValue from  currentToClamp to 0 in 4 steps with a period of CURRENT_RAMP_PERIOD*/
            this.executeCurrentRamp(this.currentToClamp, 0, 4, CURRENT_RAMP_PERIOD);
            controller.writeCurrent(this.currentToOpen);
            
        } else if (action == MobileItemAction.OPEN_ONLINECLAMP) {
            sendCurrentToController(this.currentToOpen);
            
        } else throw new IllegalArgumentException(getName() + " invalid action for ONLINE clamp:"+action);
    }
    /**
     * enables controller and sends current to controller
     * first applies a force on the clamp, and then releaseBrake
     * don't do it in the reverse order !!!!
     * @param cur to be sent to the controller
     */
    public void sendCurrentToController(int cur) {
        checkControllerBeforeAction();
        controller.writeCurrent(cur);
        controller.releaseBrake();
    }
    
    
    

    /**
     * What to do to abort an action.
     * @param action
     * @param delay
     * @throws FcsHardwareException 
     */
    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action,long delay)  {
        controller.quickStop();
        controller.activateBrake();
//        controller.disable();
    }

    /**
     * What to be done after the action is completed.
     * @param action
     * @throws FcsHardwareException 
     */
    @Override
    public void postAction(FcsEnumerations.MobileItemAction action)  {
        controller.activateBrake();
        controller.writeCurrent(0);
    }
    


    
    

    /**
     * This method is used to configure the controllers of the autochanger
     * online clamps. * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Digital
     * Inputs activated for OnlineClamps controllers are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ QuickStop Input=01
     * Mask=Enabled Polarity=HighActive HE_Sensor General_A Input=02
     * Mask=Enabled Polarity=HighActive HE_Sensor\ General_B Input=03
     * Mask=Enabled Polarity=HighActive Check Brake Status
     * ******************************************** 1 - Configure the Digital
     * Input Functionnality index 0x2070 sindex input_ID Functionnality tab
     * Value = 15 General Purpose A Value = 14 General Purpose B Value = 13
     * General Purpose C Value = 12 General Purpose D Value = 11 General Purpose
     * E Value = 10 General Purpose F Value = 9 General Purpose G Value = 8
     * General Purpose H Value = 7 General Purpose I Value = 6 General Purpose J
     * Value = 5 QuickStop Value = 4 Drive enable Value = 3 Position marker
     * Value = 2 Home switch Value = 1 Positive Limit switch Value = 0 Negative
     * Limit switch msg = 'Set DigIn_1 Quick Stop,wsdo,%s,2070,01,02,0005\n' %
     * (str(cobID)) self.com_hal.executeOperation(client,msg) msg = 'Set DigIn_2
     * General_A,wsdo,%s,2070,02,02,8000\n' % (str(cobID))
     * self.com_hal.executeOperation(client,msg) msg = 'Set DigIn_3
     * Genral_B,wsdo,%s,2070,02,02,4000\n' % (str(cobID))
     * self.com_hal.executeOperation(client,msg)
     * *************************************************************** 2 -
     * Configure the Digital Input Polarity index 0x2071 subindex 0x03 Value = 0
     * High Active Value = 1 Low Active
     * **************************************************************** 3 -
     * Configure the Digital Input Mask index 0x2071 subindex 0x02 Value = 0
     * Functionnality state will not be displayed Value = 1 Functionnality state
     * will be displayed * Digital Input Functionnality tab value Bit_15 General
     * Purpose A 1 Bit_14 General Purpose B 1 Bit_13 General Purpose C 0 Bit_12
     * General Purpose D 0 Bit_11 General Purpose E 0 Bit_10 General Purpose F 0
     * Bit_9 General Purpose G 0 Bit_8 General Purpose H 0 Bit_7 General Purpose
     * I 0 Bit_6 General Purpose J 0 Bit_5 QuickStop 1 Bit_4 Drive enable 0
     * Bit_3 Position marker 0 Bit_2 Home switch 0 Bit_1 Positive Limit switch 0
     * Bit_0 Negative Limit switch 0
     * ********************************************************************* 4 -
     * Configure the Digital Input Execution Mask index 0x2071 subindex 0x04
     * Digital Input Excecution Mask Value Bit_15 to Bit_6 reserved 0 Bit_5
     * QuickStop 1 Bit_4 Drive enable 1 Bit_3 Position marker 0 Bit_2 Home
     * switch 0 Bit_1 Positive Limit switch 0 Bit_0 Negative Limit switch 0
     * **********************************************************************
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the online clamps controllers.")
    public void configureDigitalInputOfOnlineClamps()  {
        //1-Configure Digital Input Fonctionnality
        //QuickStop
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalInput1, "0005");
        //General_A
        //writeParameterInHexa(CanOpenEPOS.Parameter.ConfigurationOfDigitalInput2, "8000");
        //General_B
        //writeParameterInHexa(CanOpenEPOS.Parameter.ConfigurationOfDigitalInput3, "4000");

        //2-Configure the Digital Input Polarity
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityPolarity, "0");

        //3-Configure the Digital Input Mask
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityMask, "C20");

        //4 - Configure the Digital Input Execution Mask
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityExecutionMask, "30");

    }

    /**
     * This methods is used to configure the autochanger onlineClamps EPOS
     * controllers. * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Digital
     * Outputs activated for OnlineClamps controller are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Ready/Fault Output=01
     * Mask=Enabled Polarity=HighActive General_A Output=04 Mask=Enabled
     * Polarity=HighActive ******************************************** 1 -
     * Configure the Digital Output Functionnality index 0x2079 subindex
     * input_ID Functionnality tab Value = 15 General Purpose Out_A Value = 14
     * General Purpose Out_B Value = 13 General Purpose Out_C Value = 12 General
     * Purpose Out_D Value = 11 General Purpose Out_E Value = 10..8 not used
     * Value = 7..3 reserved Value = 2 Holding Brake Value = 1 Position compare
     * Value = 0 Ready/Fault
     * ********************************************************************************
     * 2 - Configure the Digital Output Functionnality Mask index 0x2078
     * subindex 0x02 Value = 0 functionnality not activatd Value = 1
     * functionnality activated
     * ********************************************************************************
     * Digital Output Functionnality Mask Value Bit_15 General Purpose Out_A 1
     * Bit_14 General Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12
     * General Purpose Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General
     * Purpose Out_F 0 Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H
     * 0 Bit_7 reserved 0 Bit_6 reserved 0 Bit_5 reserved 0 Bit_4 reserved 0
     * Bit_3 reserved 0 Bit_2 Holding Brake 0
     * ********************************************************************************
     * 3 - Configure the Digital Output Functionnality Polarity index 0x2078
     * subindex 0x03 Value = 0 associated output not change => HighActive Value
     * = 1 associated output inverted => LowActive
     * ********************************************************************************
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the online clamps controllers.")
    public void configureDigitalOutputOfOnlineClamps()  {
        //1 - Configure the Digital Output Functionnality
        //Ready/Fault
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput1, "0");
        //General_A
        //gives an error : CAN OPEN DEVICE ERROR CODE=6090031 index=2079 subindex=04 error name=Value too high Error
        //writeParameterInHexa(Parameter.ConfigurationOfDigitalOutput4, "8000"); //pas bon

        //2 - Configure the Digital Output Functionnality Mask
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityMask, "8001");

        //3 - Configure the Digital Output Functionnality Polarity
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityPolarity, "8001");
    }

    /**
     * What to do when the Modules we observe send there new values. This class
     * observes its controller to publish data when its controller is in fault
     * or after a faultReset. Needed to update the GUI.
     *
     * @param source
     * @param v
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        FCSLOG.debug(getName() + ":processUpdate from source=" + source.toString()
                + " ValueUpdate=" + v.getName());
        if (!(source instanceof EPOSController)) {
            return;
        }
        if (v.getValue() instanceof EmergencyMessage) {
            EmergencyMessage emcyMsg = (EmergencyMessage) v.getValue();
            FCSLOG.debug(getName() + ":EmergencyMessage received from CanOpenProxy="
                    + emcyMsg.toString());
            processEmergencyMessage(emcyMsg);


        } else if (v.getValue() instanceof String) {
            EPOSController ctrl = (EPOSController) source;
            String msgFromController = (String) v.getValue();
            if (ctrl.getName().equals(controller.getName()) &&  ("faultReset".equals(msgFromController))) {               
                    this.controllerInFault = false;
                    this.publishData();
            }
        }
    }

    
    /**
     * Creates and returns the object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerOnlineClamp createStatusDataPublishedByOnlineClamp() {
        StatusDataPublishedByAutochangerOnlineClamp status = new StatusDataPublishedByAutochangerOnlineClamp();
        status.setName(getName());
        status.setLockSensorValue(closeSensors.isOn());
        status.setUnlockSensorValue(openSensors.isOn());
        status.setLockStatus(lockStatus);
        status.setLockSensorInError(closeSensors.isInError());
        status.setUnlockSensorInError(openSensors.isInError());
        status.setInError(lockStatus == FcsEnumerations.LockStatus.ERROR);
        status.setControllerInFault(controllerInFault);
        return status;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(
                new KeyValueData(getName(), createStatusDataPublishedByOnlineClamp()));
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay)  {
        controller.quickStop();
    }
    
    /************************************************************
     * Methods to write wurrent to the controller with a ramp
    /************************************************************
    */

    private void cancelCurrentRamp() {
        lock.lock();
        try {
            FCSLOG.debug(" => stop writing current");
            currentRampEnded.signalAll();

        } finally {
            lock.unlock();
        }
        this.currentRampHandle.cancel(true);
        FCSLOG.debug(" => current ramp ended");
    }
    
    
     /**
     * This method waits until the newCurrentValue ramp is completed. 
     */
    private void waitForEndOfCurrentRamp() {
        while (hasToWaitForEndOfCurrentRamp) {
            try {
                FCSLOG.debug(" waiting for end of current Ramp" );
                currentRampEnded.await();
            } catch (InterruptedException ex) {
                FCSLOG.debug(" InterruptedException received=" + ex.toString());
                break;
            }
        }
        FCSLOG.debug(" STOP WAITING FOR END OF CURRENT RAMP");
    }
   
    /**
     * Write newCurrentValue to controller with a newCurrentValue ramp.
     * Uses a scheduler to schedule the task.
     * @param initialValue
     * @param finalValue
     * @param nbStep
     * @param period 
     */
    private void writeCurrentRamp(final int initialValue, final int finalValue, final int nbStep, final long period) {
        FCSLOG.debug("############################");
        FCSLOG.debug("initialValue="+initialValue);
        FCSLOG.debug("finalValue="+finalValue);
        FCSLOG.debug("nbStep="+nbStep);
        FCSLOG.debug("period="+period);
        FCSLOG.debug("############################");
        final int stepHeight = (finalValue - initialValue) / nbStep;
        FCSLOG.debug("stepHeight="+stepHeight);

        final Runnable currentRamp = new Runnable() {
            private int newCurrentValue = initialValue + stepHeight;
            public boolean finalValueReached() {
                boolean goingUp = stepHeight > 0 && newCurrentValue > finalValue;
                boolean goingDown = stepHeight < 0 && newCurrentValue < finalValue;
                return goingUp || goingDown;
            }
            
            @Override
            public void run() {
                if (finalValueReached()) {
                    hasToWaitForEndOfCurrentRamp = false;
                    cancelCurrentRamp();
                } else {
                    FCSLOG.debug("current to write to controller ="+newCurrentValue);
                    controller.writeCurrent(newCurrentValue);
                    newCurrentValue = newCurrentValue + stepHeight;
                }
            }
        };       
        this.currentRampHandle = scheduler.scheduleAtFixedRate(currentRamp, 0, period, TimeUnit.MILLISECONDS);
    }
    
    /**
     * Writes newCurrentValue to controller with a newCurrentValue ramp : from initialValue to finalValue in nbStep.
     * 
     * @param initialValue
     * @param finalValue
     * @param nbStep
     * @param period of time to wait between 2 writes to the controller.
     */
    public void executeCurrentRamp(int initialValue,int finalValue, int nbStep, long period) {
        lock.lock();
        try {
            hasToWaitForEndOfCurrentRamp = true;
            writeCurrentRamp(initialValue, finalValue, nbStep, period);
            waitForEndOfCurrentRamp();
        } finally {
            hasToWaitForEndOfCurrentRamp = false;
            currentRampEnded.signalAll();
            lock.unlock();
        }
    }
    
    /************************************************************
     * end of Methods to write wurrent to the controller with a ramp
    /************************************************************
    */
}
