
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This class is used with the final hardware in which we have a clamp online to
 * hold the filter at online position. It's not used for Single Filter Test but
 * is used for testbenchCPPM.
 *
 * @author virieux
 */
public class AutoChangerModule extends Module implements HardwareController, FilterHolder, AlertRaiser {

    @ConfigurationParameter
    private String plutoGatewayName;
        
    private BridgeToHardware bridge;
    

    private PlutoGatewayInterface plutoGateway;
    private FilterManager filterManager;
    private FilterIdentificator filterIdentificator;

    private final NumericSensor loaderConnectedSensor0;
    private final NumericSensor loaderConnectedSensor1;
    private final ComplementarySensors loaderConnectedSensors;

    private final AutoChangerTwoTrucksModule autochangerTrucks;
    private final AutochangerTwoLatches latches;
    private final AutochangerThreeOnlineClamps onlineClamps;
    
    private int filterOnTrucksID = 0;
    
    private FilterHolder carousel;

    private FilterHolder loader;

    private boolean loaderConnectedSensorsInError = false;
    private boolean loaderConnected = false;

    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

    /**
     * Build  new AutoChangerModule with following arguments :
     * @param plutoGatewayName
     * @param loaderConnectedSensor0
     * @param loaderConnectedSensor1
     * @param trucks
     * @param latches
     * @param onlineClamps 
     */
    public AutoChangerModule(
            String plutoGatewayName,
            NumericSensor loaderConnectedSensor0,
            NumericSensor loaderConnectedSensor1,
            AutoChangerTwoTrucksModule trucks,
            AutochangerTwoLatches latches,
            AutochangerThreeOnlineClamps onlineClamps) {
        this.plutoGatewayName = plutoGatewayName;
        this.loaderConnectedSensor0 = loaderConnectedSensor0;
        this.loaderConnectedSensor1 = loaderConnectedSensor1;
        this.autochangerTrucks = trucks;
        this.latches = latches;
        this.onlineClamps = onlineClamps;
        this.loaderConnectedSensors = new ComplementarySensors(loaderConnectedSensor0, loaderConnectedSensor1);
    }


    /**
     * Return false if the 2 redondant loader sensors are equal.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant loader sensors are equal.")
    public boolean isLoaderConnectedSensorsInError() {
        return loaderConnectedSensorsInError;
    }
    
    /**
     * Return true if an error has been detected on sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
        description = "Return true if an error has been detected on sensors.")
    public boolean isSensorsInError() {
        return latches.isInError() || onlineClamps.isInError() || autochangerTrucks.isPositionSensorsInError();
    }

    /*for the simulator - used in python scripts */
    public int getHandoffPosition() {
        return this.autochangerTrucks.getHandoffPosition();
    }

    /*for the simulator - used in python scripts */
    public int getOnlinePosition() {
        return this.autochangerTrucks.getOnlinePosition();
    }

    /*for the simulator - used in python scripts */
    public int getStandbyPosition() {
        return this.autochangerTrucks.getStandbyPosition();
    }

    /**
     * If autochanger holds a filter, return filterID, else return 0.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "If autochanger holds a filter, return filterID, else return 0.")    
    public int getFilterOnTrucksID() {
        return filterOnTrucksID;
    }

    /**
     * Used for simulator only.
     * @param filterOnTrucksID 
     */
    public void setFilterOnTrucksID(int filterOnTrucksID) {
        this.filterOnTrucksID = filterOnTrucksID;
    }
    
    
    
    /**
     * If autochanger holds a filter, return filter name else return null.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "If autochanger holds a filter, return filter name else return null.")
    public String getFilterOnTrucksName() {
        if (this.isHoldingFilter()) {
            return filterManager.getFilterNameByID(filterOnTrucksID);
        } else {
            return null;
        }
    }
    
    /**
     * If autochanger holds a filter at ONLINE, return filter name else return null.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "If autochanger holds a filter at ONLINE, return filter name else return null.")    
    public String getFilterOnlineName() {
        if (this.isAtOnline() && !this.isEmpty()) {
            return filterManager.getFilterNameByID(filterOnTrucksID);
        } else {
            return null;
        }
    }

    /**
     * Returns the boolean field loaderConnected. If the loaderConnected boolean
     * is being updated and waits for a response from a sensor, this methods
     * waits until loaderConnected is updated. If the field loaderConnected is
     * not being updated, it returns immediatly the field loaderConnected.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the loader is connected to the camera. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderConnected() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(getName() + ": has been interrupted while waiting for end of update.");
                }

            }
            return loaderConnected && !loaderConnectedSensorsInError;

        } finally {
            lock.unlock();
        }
    }

    /**
     * TODO when autochanger is in standalone mode this method has to answer the value of the
     * fake carousel sensor.
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carousel is holding the filter at STANDBY position.")
    public boolean isCarouselHoldingFilterAtStandby()   {
            return this.carousel.isAtStandby() && this.carousel.isHoldingFilter();
    }
    
    /**
     * Return true if loader is connected and holding a filter.
     * This command doesn't read again the sensors.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if loader is connected and holding a filter. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderHoldingFilterAtHandoff()  {
        if (loaderConnected) {
            return this.loader.isHoldingFilter();
        } else {
            return false;
        }
    }
    
    /**
     * Return true if the autochanger is running an action.
     * @return 
     */
    public boolean isMoving() {
        //TODO or latches are closing or opening or ONLINE clamps are closing or opening.
        return autochangerTrucks.isMoving();
    }
   
    @Override
    public void initModule() {
        FCSLOG.info(getName() + ": init MODULE");
        ComponentLookup lookup = getSubsystem().getComponentLookup();
        this.plutoGateway = (PlutoGatewayInterface) lookup.getComponentByName(plutoGatewayName);
        this.bridge = (BridgeToHardware) lookup.getComponentByName("tcpProxy");
        this.carousel = (FilterHolder) lookup.getComponentByName("carousel");
        this.loader = (FilterHolder) lookup.getComponentByName("loader");
        this.filterManager = (FilterManager) lookup.getComponentByName("filterManager");
        this.filterIdentificator = (FilterIdentificator) lookup.getComponentByName("filterIdentificator");
    }

    /**
     * @return true if all autochanger hardware is initialized and bridge is hardwareReady.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all autochanger CANopen devices are booted, identified and initialized.")
    public boolean isCANDevicesReady() {
        return bridge.isCANDevicesReady();
    }
    
    /**
     * @return if all autochanger CANopen devices are booted, identified and nitialized and homing of the 
     * controllers is done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all autochanger CANopen devices are booted, identified and "
                    + "initialized and homing of the controllers is done.")
    public boolean isHardwareReady() {
        return bridge.isCANDevicesReady() && isInitialized();
    }    

    /**
     * Return true if plutoGateway, trucks, latches and onlineClamps are initialized.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all autochanger hardware is initialized.")
    public boolean isInitialized() {
        return plutoGateway.isInitialized() && autochangerTrucks.isInitialized() 
                && latches.isInitialized() && onlineClamps.isInitialized();
    }

    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        FCSLOG.debug(getName() + " checking hardware.");
        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }
        /* checkHardware has to be executed on my children which are HardwareController 
        (latches, onlineClamps, trucks)*/
        return TreeWalkerDiag.GO;
    }

    @Override
    public void checkStarted() throws HardwareException {
        FCSLOG.info(getName() + " BEGIN checkStarted");

        //check that all hardware is booted and identified.
        bridge.checkHardware();

        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
            autochangerTrucks.checkHardware();

        } catch (FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }

    }

    /**
     * This method is executed during a shutdown of the subystem to prevent a
     * shutdown when some hardware is moving. Nothing to do here because all the
     * moving hardware of the autochanger is represented by the class
     * MobilItemModule where the method checkStopped is implemented.
     *
     * @throws HardwareException
     */
    @Override
    public void checkStopped() throws HardwareException {
    }

    /**
     * Check if latches can be opened. 
     * If latches can't be openened because the conditions for opening are not filled, 
     * this command throws a RejectedCommandException.
     * If latches are in error, it throws a FcsHardwareException.
     * 
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the latches can be opened.")
    public void checkConditionsForOpeningLatches()  {
        FCSLOG.info(getName() + " checking pre-conditions for opening latches");

        updateStateWithSensors();

        checkLatchesInitialized();
            
        if (latches.isInError()) {
            throw new FcsHardwareException(getName()  + ": latches are in ERROR state - can't unlockClamps latches.");
            
        } 
        //commented out in november 2016 : trucks don't work.
        //TODO check if first condition is ok to be tested ?
        else if (isEmpty()) {
            throw new RejectedCommandException(getName() + ": no filter in autochanger - can't open latches.");
            
        } else if (!autochangerTrucks.isAtStandbyPosition() && !autochangerTrucks.isAtHandoffPosition()) {
            throw new RejectedCommandException(getName() + ": autochanger is loaded with a filter but is not "
                    + "at handoff position neither at standby - can't open latches.");
            
        } else if (autochangerTrucks.isAtStandbyPosition()) {
            checkConditionsForOpeningLatchesAtStandby();
            
        } else if (autochangerTrucks.isAtHandoffPosition()) { 
            checkConditionsForOpeningLatchesAtHandoff();
        }
    }
    
    private void checkLatchesInitialized() {
        if (!latches.isInitialized()) {
            throw new RejectedCommandException(getName()  + ": latches are not intialized. "
                    + "Please initialize hardware first.");
        }
    }
    
    private void checkConditionsForOpeningLatchesAtStandby() {
        /* autochanger holds a filter at STANDBY position*/
        if (isCarouselHoldingFilterAtStandby()) {
            FCSLOG.info(getName() + " carousel is holding filter at STANDBY => latches can be open safely.");
            
        } else {
            throw new RejectedCommandException(getName() + ": autochanger is loaded with a filter and is  "
                    + "at STANDBY position but carousel doesn't hold the filter "
                    + "- can't open latches.");
        }
    }
    
    private void checkConditionsForOpeningLatchesAtHandoff() {
        /* autochanger holds a filter at HANDOFF position*/
        if (isLoaderHoldingFilterAtHandoff()) {
            FCSLOG.info(getName() + " loader is holding filter at HANDOFF => latches can be open safely.");
            
        } else {
            throw new RejectedCommandException(getName() + ": autochanger is loaded with a filter and is  "
                    + "at HANDOFF position but loader doesn't hold the filter "
                    + "- can't open latches.");
        }
    }

    /**
     * Check if Autochanger latches can be closed.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if Autochanger latches can be closed.")
    public void checkConditionsForClosingLatches()  {
        
        FCSLOG.info(getName()+" checking conditions for closing latches.");

        updateStateWithSensors();
        String message;
        checkLatchesInitialized();

        if (latches.isInError()) {
            throw new RejectedCommandException(getName() + ": latches are in ERROR state - can't close latches.");
        
        } 
        //commented out in november 2016 : trucks sensors don't work
        //TODO delete isEmpty condition.
        else if (isEmpty()) {
            message = getName() + ": no filter in autochanger - can't close latches.";
            throw new RejectedCommandException(message);
            
        } else if (!autochangerTrucks.isAtStandbyPosition() && !autochangerTrucks.isAtHandoffPosition()) {
            throw new RejectedCommandException(getName() + ": autochanger is not "
                    + "at handoff position neither at standby - can't close latches.");
        } 
    }
    
    /**
     * This methods checks that filter can be moved by trucks.
     * Returns immediately if autochanger trucks are empty.
     * Throws RejectedCommandException if a filter is in trucks AND held both by autochanger
     * and (loader or carousel).
     * 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Check if its safe for filters to move autochanger trucks.")
    public void checkFilterSafetyBeforeMotion()  {
        
        String msg = getName() + " can't move autochanger trucks because ";
        this.updateStateWithSensors();
        
        if (this.isSensorsInError()) {
            handleSensorsError();
            
        } else if (isEmpty()) {
            FCSLOG.info(getName() + " trucks are empty - can move");
            
        } else if (isAtStandby()) {
            checkFilterSafetyAtStandby(msg);
            
        } else if (isAtHandoff()) {
            FCSLOG.info(getName() + " COUCOU : trucks are empty "+ isEmpty());
            checkFilterSafetyAtHandoff(msg);
            
        } else if (isAtOnline()) {
            checkFilterSafetyAtOnline(msg);
            
        } else if (!isHoldingFilter()) {
            throw new RejectedCommandException(msg + 
                "latches are not LOCKED");
        }
    }
    
    /**
     * Check if trucks can be moved when loaded with a filter at STANDBY.
     * @param message
     * @throws FcsHardwareException 
     */
    private void checkFilterSafetyAtStandby(String message)  {
        if (isHoldingFilter() && isCarouselHoldingFilterAtStandby())  {
            throw new RejectedCommandException(message + 
                    "both carousel and autochanger are holding filter at STANDBY");
            
        } else if (!isHoldingFilter() && !isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message + 
                    "neither carousel nor autochanger are holding filter at STANDBY");
        } else {
            FCSLOG.info(getName() + " filter safe at STANDBY - can move");
        }
    }
    
    /**
     * Check if trucks can be moved when loaded with a filter at HANDOFF.
     * @param message
     * @throws FcsHardwareException 
     */
    private void checkFilterSafetyAtHandoff(String message)  {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message
                + "autochanger is holding filter at HANDOFF but another filter is at STANDBY "
                    + "- can't move trucks");
        }
        if (isHoldingFilter() && isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(message
                + "both loader and autochanger are holding filter at HANDOFF");
        } else if (!isHoldingFilter() && !isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(message + 
                    "neither loader nor autochanger are holding filter at HANDOFF");
        } else {
            FCSLOG.info(getName() + " filter safe at HANDOFF - can move");
        }
    }
    
    /**
     * Check if trucks can be moved when loaded with a filter at ONLINE.
     * @param message
     * @throws FcsHardwareException 
     */
    private void checkFilterSafetyAtOnline(String message)  {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message
                + "autochanger is holding filter at ONLINE but another filter is at STANDBY "
                    + "- can't move trucks");
        }
        if (isHoldingFilter() && onlineClamps.isLocked()) {
            throw new RejectedCommandException(message
                + "both latches and onlineClamps are holding filter at ONLINE - can't move trucks.");
            
        } else if (!isHoldingFilter() && !onlineClamps.isLocked()) {
            throw new RejectedCommandException(message + 
                    "neither latches nor onlineClamps are holding filter at HANDOFF");
        } else {
            FCSLOG.info(getName() + " filter safe at ONLINE - can move");
        }
    }
    
    /**
     * Raise ALERT and throw FcsHardwareException with a detailed message when error is detected in sensors.
     * @throws FcsHardwareException 
     */
    private void handleSensorsError()  {
        String msg = getName() + " error detected in sensors:";
        if (this.autochangerTrucks.isPositionSensorsInError()) {
            msg += " trucks position sensors";
        }
        if (this.latches.isInError()) {
            msg += " latches sensors";
        }
        if (this.onlineClamps.isInError()) {
            msg += " onlineClamps sensors";
        }
        this.raiseAlarm(AC_SENSOR_ERROR, msg);
        throw new FcsHardwareException(msg);
    }
    
    
    /**
     * Return true if a filter is in trucks and latches are LOCKED.
     * @return 
     */
    @Override
    public boolean isHoldingFilter() {
        return !this.latches.isEmpty() && this.latches.isLocked();
    }
    
    /**
     * Return true if onlineClamps are LOCKED.
     * @return 
     */
    public boolean isOnlineClampsLocked() {
        return onlineClamps.isLocked();
    }
    
    /**
     * Update truck position
     */
    void updateTrucksPosition() {
        this.autochangerTrucks.updatePosition();
    }
    

    /**
     * This method reads all the sensors and updates the autochangerTrucks, latches and online clamps state. 
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors()  {
        plutoGateway.checkInitialized();

        lock.lock();
        try {
            updatingState = true;
            this.plutoGateway.updateValues();
            String[] readHexaValues = this.plutoGateway.getHexaValues();
            updateStateWithSensors(readHexaValues);

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
            this.autochangerTrucks.publishData();
            this.latches.publishData();
            this.onlineClamps.publishData();
        }
    }
    
    public void updateStateFromPDO() {
        String[] readHexaValues = this.plutoGateway.readHexaValuesFromPDO();
        updateStateWithSensors(readHexaValues);
    }
    
    private void updateStateWithSensors(String[] readHexaValues) {
        //update of loaderConnectedSensors
        this.loaderConnectedSensors.updateValues(readHexaValues);
        this.loaderConnected = loaderConnectedSensors.isOn();
        this.loaderConnectedSensorsInError = loaderConnectedSensors.isInError();

        this.autochangerTrucks.updateStateWithSensors(readHexaValues);
        this.latches.updateStateWithSensors(readHexaValues);
        this.onlineClamps.updateStateWithSensors(readHexaValues);
        this.filterIdentificator.updateSensorsValues(readHexaValues);
    }

    /**
     * Return true if autochanger trucks are at HANDOFF.
     * This command doesn't read again the sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if autochanger trucks are at HANDOFF. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtHandoff() {
        return autochangerTrucks.isAtHandoffPosition();
    }
    
    /**
     * Return true if autochanger trucks are at ONLINE.
     * This command doesn't read again the sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if autochanger trucks are at ONLINE. "
            + "This command doesn't read again the sensors.")
    public boolean isAtOnline() {
        return autochangerTrucks.isAtOnlinePosition();
    }
    
    /**
     * Return true if autochanger trucks are at STANDBY.
     * This command doesn't read again the sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if autochanger trucks are at STANDBY. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtStandby() {
        return autochangerTrucks.isAtStandbyPosition();
    }

    /**
     * Return true if there is no filter in the autochanger.
     * This command doesn't read again the sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if there is no filter in the autochanger. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {
        return latches.isEmpty();
    }


    
    /**
     * Lock ONLINE clamps.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Lock ONLINE clamps.")    
    public void lockOnlineClamps() {
        this.onlineClamps.lockClamps();
        FCSLOG.info(getName()+":"+this.getFilterOnTrucksName()+ " is LOCKED ONLINE.");
    }
    
    /**
     * Unlock ONLINE clamps.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Unlock ONLINE clamps.")
    public void unlockOnlineClamps() {
        this.onlineClamps.unlockClamps();
        FCSLOG.info(getName()+":"+this.getFilterOnTrucksName()+ " is UNLOCKED ONLINE.");
    }
    
    /**
     * Move trucks to HANDOFF position.
     * @throws FailedCommandException
     * @throws FcsHardwareException 
     */
    void goToHandOff()  {
        autochangerTrucks.goToHandOff();
    }
    
    /**
     * Move trucks to STANDBY position.
     * @throws FailedCommandException
     * @throws FcsHardwareException 
     */
    void goToStandby()  {
        this.getSubsystem().updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_STANDBY, 
                FcsEnumerations.FilterReadinessState.NOT_READY);
        autochangerTrucks.goToStandby();
    }
    
    /**
     * Move trucks to ONLINE position.
     * @throws FailedCommandException
     * @throws FcsHardwareException 
     */
    void goToOnline()  {
        this.getSubsystem().updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_ONLINE, 
                FcsEnumerations.FilterReadinessState.NOT_READY);       
        autochangerTrucks.goToOnline();
    }

    void grabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); 
    }
    
    void ungrabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); 
    }

    void moveFilterToHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); 
    }
    

    /**
     * Close latches on filter at STANDBY.
     * Identify filter at STANDBY in reading identifier ID.
     * Update filter location.
     * @throws RejectedCommandException if autochanger trucks are not at STANDBY.
     */
    @Command(type = Command.CommandType.ACTION, alias = "grabFilter",level = Command.ENGINEERING1, 
        description = "Go to STANDBY to grab filter and stay at STANDBY.")
    public void grabFilterAtStandby() {
        if (!this.isAtStandby()) {
            throw new RejectedCommandException(getName() + ": can't grab a filter when trucks are not at STANDBY.");
        } 
        FCSLOG.info(getName() + ": grabbing a filter at standby position.");
        latches.close();
        filterOnTrucksID = identifyFilterAtStandby();
        filterManager.setFilterLocation(filterOnTrucksID,FcsEnumerations.FilterLocation.AUTOCHANGER);
        publishData();
        FCSLOG.info(getName() + ": filter " + filterOnTrucksID + " is grabbed on autochanger");
    }
    
    /**
     * Open the latches at standby position.
     * position.
     *
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(level = Command.ENGINEERING1, description = "Ungrab filter at STANDBY", 
            type = Command.CommandType.ACTION)
    public void ungrabFilterAtStandby()  {
        if (!this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(getName()+": can't ungrab a filter at STANDBY when carousel is not holding filter.");
        }
        FCSLOG.info(getName() + ": ungrabing filter at standby position.");
        latches.open();
        //eject filter from autochanger
        synchronized (this) {
            filterManager.setFilterLocation(filterOnTrucksID,FcsEnumerations.FilterLocation.CAROUSEL);
            filterOnTrucksID = 0;
        }
        publishData();
    }
    
    /**
     * Open both latches
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
        description = "Open both Latches.")
    public void openLatches() {
        latches.open();
    }
    
    /**
     * Read the filterID sensors at STANDBY to check filterID.
     * @return 
     */
    private int identifyFilterAtStandby() {
        return filterIdentificator.readFilterIdentifier(); 
    }

    @Override
    public void tick() {
        //Don't execute here methods which take a lock.
        this.publishData();
    }

    
    /**
     * Creates an Object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByAutoChanger createStatusDataPublishedByAutoChanger() {
        StatusDataPublishedByAutoChanger status = new StatusDataPublishedByAutoChanger();
        status.setName(getName());
        status.setLoaderConnectedSensorValue0(this.loaderConnectedSensor0.getDigitalValue());
        status.setLoaderConnectedSensorValue1(this.loaderConnectedSensor1.getDigitalValue());
        status.setLoaderConnectedSensorsInError(this.isLoaderConnectedSensorsInError());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData("autochangerGeneral", 
                createStatusDataPublishedByAutoChanger()));
    }

}
