
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.MainModule;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * The Main Module for the single-filter-test.
 * 
 * It raises an alert :
 * - FCS006 in start.
 * @author virieux
 */
public class SftMainModule extends MainModule  {

    private final Filter dummyFilter;
    private SftCarouselModule carousel;
    private SftAutoChangerModule autochanger;
    private boolean filterLocated;

    /**
     * Build a SftMainModule with a bridge and a filter.
     * @param bridge
     * @param dummyFilter 
     */
    public SftMainModule(BridgeToHardware bridge,
            Filter dummyFilter) {
        super(bridge);
        this.dummyFilter = dummyFilter;
    }

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** SETTERS AND GETTERS  *****************************************************
     */
    /**
     * ***********************************************************************************************
     */
    /**
     *
     * @return the dummyFilter
     */
    public Filter getDummyFilter() {
        return dummyFilter;
    }

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** END OF SETTERS AND GETTERS  **********************************************
     */
    /**
     * ***********************************************************************************************
     */
   

    @Override
    public void initModule() {

        super.initModule();
        ComponentLookup lookup = getComponentLookup();
        carousel = (SftCarouselModule) lookup.getComponentByName("carousel");
        autochanger = (SftAutoChangerModule) lookup.getComponentByName("autochanger");
        filterLocated = false;
    }

    
    @Override
    public void start() {
        if (this.isCANDevicesReady() && !filterLocated) {
            try {
                this.locateFilter();
                filterLocated = true;
                FCSLOG.info("Filter has been located. Hardware is ready to receive commands.");
                FCSLOG.debug("DEBUGGING");
            } catch (RejectedCommandException | FailedCommandException | FcsHardwareException ex) {
                filterLocated = false;
                FCSLOG.error(ex);
                String errorMsg = ": couldn't locate filters.";
                this.raiseAlarm("FCS005:"+name, errorMsg);
            }
        }
    }

    /**
     * Read the sensors and update location of the filter and state of Carousel and Autochanger.
     * @throws FcsHardwareException 
     */
    @Command(level = Command.NORMAL, 
            description = "Read the sensors and update location of the filter and state of Carousel and Autochanger.", 
            type = Command.CommandType.QUERY)
    public void locateFilter()  {

        FCSLOG.debug("MODE DEBUG");
        carousel.updateClampsStateWithSensors();
        if (carousel.getClampXminus().getClampState().equals(FilterClampState.CLAMPEDONFILTER)
                || carousel.getClampXminus().getClampState().equals(FilterClampState.UNCLAMPEDONFILTER)) {
            carousel.getSocketByName("socket1").putFilterOnSocket(dummyFilter);

        }
        autochanger.updateLatchesStateWithSensors();

        if (!autochanger.isTrucksEmpty()) {
            autochanger.setFilterOnTrucks(dummyFilter);
        }

        autochanger.updateTrucksLocationWithSensors();

    }

    /**
     * Check if the carousel is ready to unlock the clamps in a safe way for the filter.
     * @throws FcsHardwareException 
     */
    @Command(level = Command.ENGINEERING1, 
            description = "Check if the carousel is ready to unlock the clamps in a safe way for the filter.", 
            type = Command.CommandType.QUERY, alias = "checkUnlock")
    public void checkPreConditionsForUnlockClamps()  {
        this.updateStateWithSensors();
        if (carousel.isHoldingFilterAtStandby() && (!autochanger.isHoldingFilterAtStandby())) {
            throw new FailedCommandException("Can't unlock clamps if filter is not held by autochanger.");
        }
    }

    @Command(level = Command.ENGINEERING1, 
            description = "To display at a console the status of the hardware.", 
            type = Command.CommandType.QUERY)
    public String displayData() {
        StringBuilder sb = new StringBuilder("Data published by ");
        sb.append(getSubsystem().getName()).append("\n  ");
        FCSLOG.info("Carousel status=" + carousel.toString());
        sb.append(carousel.toString());
        sb.append("\n  ");
        sb.append("clampXminus status=");
        sb.append(carousel.getClampXminus().toString());
        sb.append("\n  ");
        sb.append("clampXplus status=");
        sb.append(carousel.getClampXplus().toString());
        sb.append("\n  ");
        FCSLOG.info("autochanger status=" + autochanger.toString());
        sb.append(autochanger.toString());
        sb.append("\n  ");
        sb.append("latchXminus status=");
        sb.append(autochanger.getLatchXminus().toString());
        sb.append("\n  ");
        sb.append("latchXplus status=");
        sb.append(autochanger.getLatchXplus().toString());
        sb.append("\n  ");
        return sb.toString();
    }

    /**
     * Initial state for sequence1 : autochanger is in highposition and trucks
     * are empty carousel holds a filter and its clamps are in CLAMPEDONFILTER
     * state.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(level = Command.ENGINEERING1, description = "Move the hardware to the initial state : ", type = Command.CommandType.ACTION)
    public String goToInitialState()  {
        this.updateStateWithSensors();
        String ack = "Hardware is at initial state and ready for a sequence1";
        if (carousel.isHoldingFilterAtStandby()) {
            
            /** Autochanger trucks are at STANDBACK.*/
            if (autochanger.isAtStandback()) {
                /*nothing to do in this case.*/
                return ack;
                
            /** Autochanger trucks are at STANDBY.*/
            } else if (autochanger.isAtStandby()) {
                FCSLOG.info("=======> Autochanger moving back empty to online position.");
                autochanger.ungrabFilterAtStandby();
                return ack;
                
            /** Autochanger trucks are in the middle of the rail.*/
            } else if (autochanger.isTrucksEmpty()) {
                return autochanger.goToStandback();
            } else {
                return "Autochanger trucks are in the midle, but not empty, I don't know what to do.";
            }

        } else if (autochanger.isHoldingFilterAtStandback()) {
            if (!carousel.isReadyToGrabAFilterAtStandby()) {
                carousel.releaseClamps();
            }
            FCSLOG.info("=======> Autochanger trucks moving back the filter to standby position.");
            autochanger.moveFilterToStandby(dummyFilter);

            // carousel checks if the filter is clamped at standby position
            FCSLOG.info("=======> Carousel is grabbing filter at standby position.");
            carousel.grabFilterAtStandby(getDummyFilter());

            //then the autochanger can unclamp the filter and go to swapout position
            FCSLOG.info("=======> Autochanger moving back empty to online position.");
            autochanger.ungrabFilterAtStandby();
            return ack;

        } else if (autochanger.isHoldingFilterAtStandby()) {
                //in this case the carousel doesn't hold the filter
            //autochanger trucks hold the filter and move it to Online Position
            FCSLOG.info("=======> Autochanger trucks about to move loaded with the filter to standback position.");
            autochanger.moveFilterToStandback(dummyFilter);

            FCSLOG.info("=======> Carousel about to release Clamps.");
            carousel.releaseClamps();

                //then we go back to the initial position :
            //autochanger move back the filter in standby position
            FCSLOG.info("=======> Autochanger trucks moving back the filter to standby position.");
            autochanger.moveFilterToStandby(dummyFilter);

            // carousel checks if the filter is clamped at standby position
            FCSLOG.info("=======> Carousel is grabbing filter at standby position.");
            carousel.grabFilterAtStandby(getDummyFilter());

            //then the autochanger can unclamp the filter and go to swapout position
            FCSLOG.info("=======> Autochanger moving back empty to online position.");
            autochanger.ungrabFilterAtStandby();
            return ack;

        } else if (!autochanger.isAtStandback() && !autochanger.isAtStandby()) {
            return "Autochanger trucks are in the midle, I don't know what to do.";
        } else {
            return "Nobody holds the filter : I don't know how to go to initial state.";
        }
    }

    /**
     * This method is the main sequence of commands that we want to test on the
     * Single Filter Test. It executes this sequence of commands once. From the
     * initial position (autochanger trucks empty in the higher position) :
     * <UL> <LI> the autochanger trucks go down along the rails to standby
     * position,
     * <LI> the autochanger grabs the filter at standby position,
     * <LI> the carousel ungrabs the filter at standby position,
     * <LI> the autochanger moves the filter to the higher position,
     * <LI> the carousel releases the clamps to be ready to clamp a filter
     * again,
     * <LI> the autochanger moves back the filter at standby position,
     * <LI> the carousel checks if the filter is clamped at standby position,
     * <LI> the autochanger ReleaseBrakes the filter latches,
     * <LI> the autochanger trucks go empty to the higher position.
     * </UL>
     *
     * @return a message for the operator
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.CanOpenException
     */
    @Command(level = Command.NORMAL, description = "The world famous sequence 1",
            type = Command.CommandType.ACTION, alias = "es1")
    public String executeSequence1()  {
        FCSLOG.info("=======> BEGIN Execution of Sequence1.");

        //autochanger trucks go to standby position to grab the filter held by carousel
        FCSLOG.info("=======> Autochanger trucks about to move empty to standby position.");
        autochanger.grabFilterAtStandby(dummyFilter);

        if (haltRequired.get()) {
            return getName() + " Execution of Sequence1 HALTED at step 1";
        }

        // carousel clamps are opened and go in the READYTOCLAMP state
        // when autochanger moves up the filter
        FCSLOG.info("=======> Carousel is ungrabbing filter at standby position.");
        carousel.ungrabFilterAtStandby(dummyFilter.getName());

        if (haltRequired.get()) {
            return getName() + " Execution of Sequence1 HALTED at step 2";
        }

        //autochanger trucks hold the filter are move it to Online Position
        FCSLOG.info("=======> Autochanger trucks about to move loaded with the filter to standback position.");
        autochanger.moveFilterToStandback(dummyFilter);

        FCSLOG.info("=======> Carousel about to release Clamps.");
        carousel.releaseClamps();

        if (haltRequired.get()) {
            return getName() + " Execution of Sequence1 HALTED at step 3";
        }

        //then we go back to the initial position :
        //autochanger move back the filter in standby position
        FCSLOG.info("=======> Autochanger trucks moving back the filter to standby position.");
        autochanger.moveFilterToStandby(dummyFilter);

        // carousel checks if the filter is clamped at standby position
        FCSLOG.info("=======> Carousel is grabbing filter at standby position.");
        carousel.grabFilterAtStandby(getDummyFilter());

        if (haltRequired.get()) {
            return name + " Execution of Sequence1 HALTED at step 4";
        }

        //then the autochanger can unclamp the filter and go to swapout position
        FCSLOG.info("=======> Autochanger moving back empty to online position.");
        autochanger.ungrabFilterAtStandby();

        FCSLOG.info("=======> Execution of Sequence1 completed.");
        return "Execution of Sequence1 completed";
    }

    /**
     * This method will execute nb times the Sequence1 : the parameter nb is
     * given by the user at the console.
     *
     * @param nb : number of times we want to execute the Sequence1
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.CanOpenException
     */
    @Command(level = Command.NORMAL, description = "The world famous sequence 1 (with number of repetitions)", 
            type = Command.CommandType.ACTION, alias = "es1")
    public String executeSequence1(
            @Argument(name = "nExecutions", description = "Number of executions") int nb) {

        if ((nb < 1) || (nb > 20000)) {
            throw new IllegalArgumentException("nb has to be comprise between 1 and 20000");
        }

        for (int i = 1; i < nb + 1; i++) {

            FCSLOG.info("===========================BEGIN execution Sequence1 nb=" + i + " ================");
            executeSequence1();
            FCSLOG.info("===========================END execution Sequence1 nb=" + i + " ================");
        }
        return getName() + ":" + nb + " executions of Sequence1 completed";
    }





    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("\n").append(carousel.toString());
        sb.append("\n").append(autochanger.toString());
        return sb.toString();
    }

    /**
     * This command updates the state of the carousel and of the autochanger in
     * reading the sensors. It has to be used in Engineering mode at the console
     * when we send commands step by step.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(level = Command.ENGINEERING1, 
            description = "Read the sensors and update the state of the carousel and the autochanger.",
            type = Command.CommandType.QUERY)
    @Override
    public void updateStateWithSensors()  {
        super.updateStateWithSensors();
            carousel.updateStateWithSensors();
            autochanger.updateStateWithSensors();
    }


}
