/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Map;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenDevice;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This class is to simulate an EPOS Controller.
 *
 * @author virieux
 */
public abstract class SimuEPOSControllerModule extends CanOpenDevice implements EPOSController {

    protected boolean turnedOff;
    private boolean enabled;
    private boolean enabledToPublish;
    protected EposMode mode;
    protected int targetPosition;
    protected int position;
    protected int ssiPosition;
    protected int targetCurrent;
    protected int actualCurrent;

    /* Parameters */
    @ConfigurationParameter
    private final Map<String, Integer> paramsForHoming;
    
    @ConfigurationParameter
    private final Map<String, Integer> paramsForProfilePosition;
    
    @ConfigurationParameter
    private final Map<String, Integer> paramsForCurrent;

    /**
     * Build a new SimuEPOSControllerModule
     * @param nodeID
     * @param serialNB
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming 
     */
    public SimuEPOSControllerModule(
            String nodeID, String serialNB,
            Map<String, Integer> paramsForCurrent,
            Map<String, Integer> paramsForProfilePosition,
            Map<String, Integer> paramsForHoming
    ) {
        super(nodeID, serialNB);
        this.paramsForHoming = paramsForHoming;
        this.paramsForProfilePosition = paramsForProfilePosition;
        this.paramsForCurrent = paramsForCurrent;
        this.position = 0;
        this.turnedOff = false;
        this.mode = EposMode.PROFILE_POSITION;
    }
    
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "For simulator only : Update position with a position given as argument.")
    public void setPosition(int actualPosition) {
        this.position = actualPosition;
    }


    @Override
    public boolean isEnabledToPublish() {
        return enabledToPublish;
    }

    @Override
    public boolean isTurnedOff() {
        return turnedOff;
    }

    //For GUI
    @Override
    public EposMode getMode() {
        return mode;
    }
    
    

    //For GUI
    @Override
    public Map<String, Integer> getParamsForHoming() {
        return paramsForHoming;
    }

    //For GUI
    @Override
    public Map<String, Integer> getParamsForProfilePosition() {
        return paramsForProfilePosition;
    }
    
    //For GUI
    @Override
    public Map<String, Integer> getParamsForCurrent() {
        return paramsForCurrent;
    }

    /**
     * We assum that simulated hardware's controller has always correct parameters.
     * @return 
     */
    @Override
    public boolean isParametersOK() {
        return true;
    }

    /**
     * This methods is called during INITIALIZATION phase. We want to publish
     * data in order for the controllers GUIs to be refreshed.
     *
     * @return a message
     */
    @Override
    public String initializeAndCheckHardware()  {
        super.initializeAndCheckHardware();
        publishData();
        return getName() + " is INITIALIZED.";

    }

    /**
     * Changes the mode to the new mode given as a String argument.
     *
     * @param modeInString
     * @return
     * @throws SDORequestException
     */
    public String changeMode(String modeInString)  {
        EposMode newMode = EposMode.valueOf(modeInString.trim().toUpperCase());
        return changeMode(newMode);
    }

    /**
     * 
     * @param newMode
     * @return
     * @throws SDORequestException 
     */
    @Override
    public String changeMode(EposMode newMode)  {
        this.mode = newMode;
        return getName() + " mode changed to=" + newMode.toString();
    }


    @Override
    public void defineAbsolutePosition(int position)  {
        this.position = position;
    }

    @Override
    public void disable()  {
        this.enabled = true;
        this.enabledToPublish = true;
    }

    @Override
    public void writeParameters(EposMode mode) {
    }

    @Override
    public String writeParameterInHexa(EPOSEnumerations.Parameter parameter, String string) {
        return getName() + " " + parameter.toString() + " set to value in HEXA=" + string;
    }
    
    public String writeParameter(EPOSEnumerations.Parameter param, int value) {
        return "OK";
    }
    
    @Override
    public Object saveParameters()  {
        return "OK";
    }

    @Override
    public void writeTargetPosition(int position)  {
        this.position = position;
    }

    @Override
    public void writeParameters(Map<String, Integer> paramMap) {
    }

    @Override
    public void writeCurrent(int aValue)  {
        this.turnedOff = false;
        this.actualCurrent = aValue;
    }


    
    @Override
    public String off()  {
        this.turnedOff = true;
        return getName() + " is OFF";
    }


    @Override
    public boolean isTargetPositionReached(int targetPosition) {
        return this.position == targetPosition;
    }

    @Override
    public int readPosition()  {
        return this.position;
    }

    /**
     * Read the position returned by the absolute encoder (single serial data).
     *
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Override
    public int readSSIPosition()  {
        return this.ssiPosition;
    }

    @Override
    public String enable()  {
        enabled = true;
        return getName() + "IS ENABLED";
    }

    @Override
    public int readCurrent()  {
        return actualCurrent;
    }

    @Override
    public EposMode readMode()  {
        return mode;
    }

    @Override
    public int getMaxCurrent() {
        throw new UnsupportedOperationException("Not supported yet."); 
    }

    @Override
    public int getMinCurrent() {
        throw new UnsupportedOperationException("Not supported yet."); 
    }

    @Override
    public int getMaxSpeed() {
        throw new UnsupportedOperationException("Not supported yet."); 
    }

    @Override
    public int getMaxPosition() {
        throw new UnsupportedOperationException("Not supported yet."); 
    }

    @Override
    public int getMinPosition() {
        throw new UnsupportedOperationException("Not supported yet."); 
    }

    @Override
    public void quickStop()  {
        FCSLOG.finest(getName() + " quickStop done.");
    }

    @Override
    public int readNumberOfErrors()  {
        return 0;
    }
    
    @Override
    public String readErrorRegister() {
        return null;
    }
    
    @Override
    public String[] readErrorHistory() {
        return new String[0];
    }

    @Override
    public String displayErrorHistory()  {
        return "No ERROR";
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the Controller is in fault.")
    @Override
    public void checkFault()  {
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Clear faults on the controller.")
    public void faultReset() {
        writeControlWord("80");
        setChanged();
        this.notifyObservers(new ValueUpdate(getName(), "faultReset"));
        this.inError = false;
        this.errorRegister = "NO ERROR";
        this.errorHistory = new String[0];
        publishData();
    }

    @Override
    public int readProfileVelocity() {
        //TODO return a value of current in random between min and max
        return 0;
    }

    @Override
    public String readParameters(EposMode mode)  {
        return getName() + ":no parameter to read.";
    }

    @Override
    public void checkParameters(String modeInString) throws HardwareException {
    }

    @Override
    public void checkParameters(EposMode aMode) throws HardwareException {

    }

    @Override
    public void writeControlWord(String f)  {
        FCSLOG.debug(getName() + "writeControlWord :" + f);
    }

    @Override
    public String shutdown()  {
        FCSLOG.debug(getName() + " is SHUTDOWN");
        return getName() + " is SHUTDOWN";
    }

    @Override
    public boolean isEnabled()  {
        return enabled;
    }

    @Override
    public boolean checkTargetReached()  {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void releaseBrake()  {
        FCSLOG.debug(getName() + " brake is released.");
    }

    @Override
    public void switchOnEnableOperation()  {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void activateBrake()  {
        FCSLOG.debug(getName() + " brake is activated.");
    }

    /**
     * Return the object to be published on the STATUS bus by a SimuEPOSControllerModule.
     * @return 
     */
    @Override
    public StatusDataPublishedByEPOSController getStatusData() {
        return FcsUtils.createStatusDataPublishedByEPOSController(this);
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            alias = "refreshGUI",
            description = "Publish data for the controller on the status bus.")
    @Override
    public void publishData() {
        StatusDataPublishedByEPOSController status = this.getStatusData();
        KeyValueData kvd = new KeyValueData(getName(), status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }

}
