
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.AutoChangerModule;
import org.lsst.ccs.subsystems.fcs.AutoChangerTwoTrucksModule;
import org.lsst.ccs.subsystems.fcs.AutochangerTruckModule;
import org.lsst.ccs.subsystems.fcs.CanOpenNumericSensor;
import org.lsst.ccs.subsystems.fcs.CarouselModule;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.FilterIdentificator;
import org.lsst.ccs.subsystems.fcs.FilterManager;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;

/**
 * To simulate the plutoGateway on the autochanger.
 * @author virieux
 */
public class SimuAutochangerPlutoGateway extends SimuPlutoGateway implements SimuFilterIDPlutoGatewayInterface {
    
    /* To read filterID hall effect sensors.*/
    private FilterIdentificator filterIdentificator;
    private FilterHolder carousel;

    private AutoChangerModule autochanger;
    //trucks controllers
    private AutoChangerTwoTrucksModule trucks;
    private SimuAutochangerLinearRailController trucksController;
    private SimuAutochangerLinearRailController trucksSlaveController;

    //latches controllers
    private SimuAutochangerLatchController latchXminusController;
    private SimuAutochangerLatchController latchXplusController;

    //clamps controllers
    private SimuAutochangerOnlineClampController onlineClampXminusController;
    private SimuAutochangerOnlineClampController onlineClampXplusController;
    private SimuAutochangerOnlineClampController onlineClampYminusController;

    //trucks sensors
    private AutochangerTruckModule acTruckXminus;
    private AutochangerTruckModule acTruckXplus;
    
    //loader connected sensors 
    private CanOpenNumericSensor loaderConnectedSensor0;
    private CanOpenNumericSensor loaderConnectedSensor1;

    //latches sensors
    private CanOpenNumericSensor lockSensorLatchXminus;
    private CanOpenNumericSensor unlockSensorLatchXminus;
    private CanOpenNumericSensor filterPresenceSensorLatchXminus;
    private CanOpenNumericSensor lockSensorLatchXplus;
    private CanOpenNumericSensor unlockSensorLatchXplus;
    private CanOpenNumericSensor filterPresenceSensorLatchXplus;
    private CanOpenNumericSensor lockSensorBLatchXminus;
    private CanOpenNumericSensor unlockSensorBLatchXminus;
    private CanOpenNumericSensor filterPresenceSensorBLatchXminus;
    private CanOpenNumericSensor lockSensorBLatchXplus;
    private CanOpenNumericSensor unlockSensorBLatchXplus;
    private CanOpenNumericSensor filterPresenceSensorBLatchXplus;    

    //clamps sensors
    private CanOpenNumericSensor onlineClampXminusLockSensor;
    private CanOpenNumericSensor onlineClampXminusCLockSensor;
    private CanOpenNumericSensor onlineClampXminusUnlockSensor;
    private CanOpenNumericSensor onlineClampXminusCUnlockSensor;
    private CanOpenNumericSensor onlineClampXplusLockSensor;
    private CanOpenNumericSensor onlineClampXplusCLockSensor;
    private CanOpenNumericSensor onlineClampXplusUnlockSensor;
    private CanOpenNumericSensor onlineClampXplusCUnlockSensor;
    private CanOpenNumericSensor onlineClampYminusLockSensor;
    private CanOpenNumericSensor onlineClampYminusCLockSensor;
    private CanOpenNumericSensor onlineClampYminusUnlockSensor;
    private CanOpenNumericSensor onlineClampYminusCUnlockSensor;
    

    /**
     * Build a new SimuAutochangerPlutoGateway
     * @param nodeID
     * @param serialNB 
     */
    public SimuAutochangerPlutoGateway(
            String nodeID, String serialNB) {
        super(nodeID, serialNB);
    }
    
    @Override
    public List<CanOpenNumericSensor> getFilterIDSensors() {
        return filterIdentificator.getFilterIDSensors();
    }

    @Override
    public void initModule() {
        super.initModule();
        ComponentLookup lookup = getComponentLookup();
        filterIdentificator = (FilterIdentificator) lookup.getComponentByName("filterIdentificator");
        carousel = (FilterHolder) lookup.getComponentByName("carousel");
        //trucks
        this.acTruckXminus = (AutochangerTruckModule) lookup.getComponentByName("acTruckXminus");
        this.acTruckXplus = (AutochangerTruckModule) lookup.getComponentByName("acTruckXplus");
        this.loaderConnectedSensor0 = (CanOpenNumericSensor) lookup.getComponentByName("loaderConnectedSensor0");
        this.loaderConnectedSensor1 = (CanOpenNumericSensor) lookup.getComponentByName("loaderConnectedSensor1");
        this.trucksController = (SimuAutochangerLinearRailController) lookup.getComponentByName("linearRailMasterController");
        this.trucksSlaveController = (SimuAutochangerLinearRailController) lookup.getComponentByName("linearRailSlaveController");
        this.trucks = (AutoChangerTwoTrucksModule) lookup.getComponentByName("autochangerTrucks");

        //latches
        this.lockSensorLatchXminus = (CanOpenNumericSensor) lookup.getComponentByName("lockSensorLatchXminus");
        this.unlockSensorLatchXminus = (CanOpenNumericSensor) lookup.getComponentByName("unlockSensorLatchXminus");
        this.filterPresenceSensorLatchXminus = (CanOpenNumericSensor) lookup.getComponentByName("filterEngagedSensorLatchXminus");
        this.lockSensorLatchXplus = (CanOpenNumericSensor) lookup.getComponentByName("lockSensorLatchXplus");
        this.unlockSensorLatchXplus = (CanOpenNumericSensor) lookup.getComponentByName("unlockSensorLatchXplus");
        this.filterPresenceSensorLatchXplus = (CanOpenNumericSensor) lookup.getComponentByName("filterEngagedSensorLatchXplus");
        this.lockSensorBLatchXminus = (CanOpenNumericSensor) lookup.getComponentByName("lockSensorBLatchXminus");
        this.unlockSensorBLatchXminus = (CanOpenNumericSensor) lookup.getComponentByName("unlockSensorBLatchXminus");
        this.filterPresenceSensorBLatchXminus = (CanOpenNumericSensor) lookup.getComponentByName("filterEngagedSensorBLatchXminus");
        this.lockSensorBLatchXplus = (CanOpenNumericSensor) lookup.getComponentByName("lockSensorBLatchXplus");
        this.unlockSensorBLatchXplus = (CanOpenNumericSensor) lookup.getComponentByName("unlockSensorBLatchXplus");
        this.filterPresenceSensorBLatchXplus = (CanOpenNumericSensor) lookup.getComponentByName("filterEngagedSensorBLatchXplus");  
        
        this.latchXminusController = (SimuAutochangerLatchController) lookup.getComponentByName("latchXminusController");
        this.latchXplusController = (SimuAutochangerLatchController) lookup.getComponentByName("latchXplusController");
        

        //ONLINE clamps
        this.onlineClampXminusLockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXminusLockSensor");
        this.onlineClampXminusCLockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXminusCLockSensor");
        this.onlineClampXminusUnlockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXminusUnlockSensor");
        this.onlineClampXminusCUnlockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXminusCUnlockSensor");
        this.onlineClampXplusLockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXplusLockSensor");
        this.onlineClampXplusCLockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXplusCLockSensor");
        this.onlineClampXplusUnlockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXplusUnlockSensor");
        this.onlineClampXplusCUnlockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampXplusCUnlockSensor");
        this.onlineClampYminusLockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampYminusLockSensor");
        this.onlineClampYminusCLockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampYminusCLockSensor");
        this.onlineClampYminusUnlockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampYminusUnlockSensor");
        this.onlineClampYminusCUnlockSensor = (CanOpenNumericSensor) lookup.getComponentByName("onlineClampYminusCUnlockSensor");

        simulateAutochangerTrucksIsAtHandoff();
        simulateAutochangerIsEmpty();
        simulateAutochangerLatchesAreUnlocked();
        simulateAutochangerOnlineClampsAreUnlocked();
    }
    
    
    private void simulateACTruckIsAtHandoff(AutochangerTruckModule truck) {
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensor(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensorB(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensorB(), "1");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksIsAtHandoff() {
        FCSLOG.info("simulateAutochangerTrucksIsAtHandoff");
        simulateACTruckIsAtHandoff(acTruckXminus);
        simulateACTruckIsAtHandoff(acTruckXplus);
        trucksController.setPosition(trucks.getHandoffPosition());
        trucksController.setSsiPosition(trucks.getEncoderRibbonMinValue()
                + trucks.getHandoffPosition());
        trucksSlaveController.setPosition(trucks.getHandoffPosition());
        trucksSlaveController.setSsiPosition(trucks.getEncoderRibbonMinValue()
                + trucks.getHandoffPosition()); 
        setChanged();
        this.notifyObservers(new ValueUpdate(name,trucks.getHandoffPosition()));
    }
    
    private void simulateACTruckIsOnline(AutochangerTruckModule truck) {
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensor(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensorB(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensorB(), "1");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksIsOnline() {
        simulateACTruckIsOnline(acTruckXminus);
        simulateACTruckIsOnline(acTruckXplus);
        trucksController.setPosition(trucks.getOnlinePosition());
        trucksController.setSsiPosition(trucks.getEncoderRibbonMinValue()
                + trucks.getOnlinePosition());
        trucksSlaveController.setPosition(trucks.getOnlinePosition());
        trucksSlaveController.setSsiPosition(trucks.getEncoderRibbonMinValue()
                + trucks.getOnlinePosition());   
        setChanged();
        this.notifyObservers(new ValueUpdate(name,trucks.getOnlinePosition()));
    }
    
    private void simulateACTruckIsAtStandby(AutochangerTruckModule truck) {
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensor(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensorB(), "0");        
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksIsAtStandby() {
        simulateACTruckIsAtStandby(acTruckXminus);
        simulateACTruckIsAtStandby(acTruckXplus);
        trucksController.setSsiPosition(trucks.getEncoderRibbonMinValue());
        trucksController.setPosition(trucks.getStandbyPosition());
        trucksSlaveController.setSsiPosition(trucks.getEncoderRibbonMinValue());
        trucksSlaveController.setPosition(trucks.getStandbyPosition());        
        setChanged();
        this.notifyObservers(new ValueUpdate(name,trucks.getStandbyPosition()));
        simulateFilterIDAtStandby();
    }
    
    private void simulateACTruckIsInTravel(AutochangerTruckModule truck) {
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getHandoffPositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getOnlinePositionSensorB(), "1");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensor(), "0");
        replaceDigitalValues((CanOpenNumericSensor) truck.getStandbyPositionSensorB(), "1"); 
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksIsInTravel() {
        simulateACTruckIsInTravel(acTruckXminus);
        simulateACTruckIsInTravel(acTruckXplus);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsConnected() {
        replaceDigitalValues(loaderConnectedSensor0, "1");
        replaceDigitalValues(loaderConnectedSensor1, "1");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsDisconnected() {
        replaceDigitalValues(loaderConnectedSensor0, "0");
        replaceDigitalValues(loaderConnectedSensor1, "0");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsOnAutochanger() {
        replaceDigitalValues(filterPresenceSensorLatchXminus, "1");
        replaceDigitalValues(filterPresenceSensorLatchXplus, "1");
        replaceDigitalValues(filterPresenceSensorBLatchXminus, "0");
        replaceDigitalValues(filterPresenceSensorBLatchXplus, "0");        
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerIsEmpty() {
        replaceDigitalValues(filterPresenceSensorLatchXminus, "0");
        replaceDigitalValues(filterPresenceSensorLatchXplus, "0");
        replaceDigitalValues(filterPresenceSensorBLatchXminus, "1");
        replaceDigitalValues(filterPresenceSensorBLatchXplus, "1");        
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchesAreLocked() {
        replaceDigitalValues(lockSensorLatchXminus, "1");
        replaceDigitalValues(unlockSensorLatchXminus, "0");
        replaceDigitalValues(lockSensorLatchXplus, "1");
        replaceDigitalValues(unlockSensorLatchXplus, "0");
        replaceDigitalValues(lockSensorBLatchXminus, "0");
        replaceDigitalValues(unlockSensorBLatchXminus, "1");
        replaceDigitalValues(lockSensorBLatchXplus, "0");
        replaceDigitalValues(unlockSensorBLatchXplus, "1");        
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchesAreUnlocked() {
        replaceDigitalValues(lockSensorLatchXminus, "0");
        replaceDigitalValues(unlockSensorLatchXminus, "1");
        replaceDigitalValues(lockSensorLatchXplus, "0");
        replaceDigitalValues(unlockSensorLatchXplus, "1");
        replaceDigitalValues(lockSensorBLatchXminus, "1");
        replaceDigitalValues(unlockSensorBLatchXminus, "0");
        replaceDigitalValues(lockSensorBLatchXplus, "1");
        replaceDigitalValues(unlockSensorBLatchXplus, "0");        
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchIsLocked(String latchName) {
        if (latchName.contains("Xminus")) {
            replaceDigitalValues(lockSensorLatchXminus, "1");
            replaceDigitalValues(unlockSensorLatchXminus, "0");
            replaceDigitalValues(lockSensorBLatchXminus, "0");
            replaceDigitalValues(unlockSensorBLatchXminus, "1");            
        } else if (latchName.contains("Xplus")) {
            replaceDigitalValues(lockSensorLatchXplus, "1");
            replaceDigitalValues(unlockSensorLatchXplus, "0");
            replaceDigitalValues(lockSensorBLatchXplus, "0");
            replaceDigitalValues(unlockSensorBLatchXplus, "1");            
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchIsUnlocked(String latchName) {
        if (latchName.contains("Xminus")) {
            replaceDigitalValues(lockSensorLatchXminus, "0");
            replaceDigitalValues(unlockSensorLatchXminus, "1");
            replaceDigitalValues(lockSensorBLatchXminus, "1");
            replaceDigitalValues(unlockSensorBLatchXminus, "0");
        } else if (latchName.contains("Xplus")) {
            replaceDigitalValues(lockSensorLatchXplus, "0");
            replaceDigitalValues(unlockSensorLatchXplus, "1");
            replaceDigitalValues(lockSensorBLatchXplus, "1");
            replaceDigitalValues(unlockSensorBLatchXplus, "0");            
        }
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLockSensorsInError(String latchName) {
        if (latchName.contains("Xminus")) {
            replaceDigitalValues(lockSensorLatchXminus, "1");
            replaceDigitalValues(lockSensorBLatchXminus, "1");
        } else if (latchName.contains("Xplus")) {
            replaceDigitalValues(lockSensorLatchXplus, "1");
            replaceDigitalValues(lockSensorBLatchXplus, "1");
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampsAreLocked() {
        simulateAutochangerOnlineClampXminusIsLocked();
        simulateAutochangerOnlineClampXplusIsLocked();
        simulateAutochangerOnlineClampYminusIsLocked();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampsAreUnlocked() {
        simulateAutochangerOnlineClampXminusIsUnlocked();
        simulateAutochangerOnlineClampXplusIsUnlocked();
        simulateAutochangerOnlineClampYminusIsUnlocked();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampIsLocked(CanOpenNumericSensor lockSensor,
            CanOpenNumericSensor lockSensorC,
            CanOpenNumericSensor unlockSensor,
            CanOpenNumericSensor unlockSensorC) {
        replaceDigitalValues(lockSensor, "1");
        replaceDigitalValues(lockSensorC, "0");
        replaceDigitalValues(unlockSensor, "0");
        replaceDigitalValues(unlockSensorC, "1");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampIsUnlocked(CanOpenNumericSensor lockSensor,
            CanOpenNumericSensor lockSensorC,
            CanOpenNumericSensor unlockSensor,
            CanOpenNumericSensor unlockSensorC) {
        replaceDigitalValues(lockSensor, "0");
        replaceDigitalValues(lockSensorC, "1");
        replaceDigitalValues(unlockSensor, "1");
        replaceDigitalValues(unlockSensorC, "0");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampXminusIsLocked() {
        simulateAutochangerOnlineClampIsLocked(onlineClampXminusLockSensor,
                onlineClampXminusCLockSensor,
                onlineClampXminusUnlockSensor,
                onlineClampXminusCUnlockSensor);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampXminusIsUnlocked() {
        simulateAutochangerOnlineClampIsUnlocked(onlineClampXminusLockSensor,
                onlineClampXminusCLockSensor,
                onlineClampXminusUnlockSensor,
                onlineClampXminusCUnlockSensor);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampXplusIsLocked() {
        simulateAutochangerOnlineClampIsLocked(onlineClampXplusLockSensor,
                onlineClampXplusCLockSensor,
                onlineClampXplusUnlockSensor,
                onlineClampXplusCUnlockSensor);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampXplusIsUnlocked() {
        simulateAutochangerOnlineClampIsUnlocked(onlineClampXplusLockSensor,
                onlineClampXplusCLockSensor,
                onlineClampXplusUnlockSensor,
                onlineClampXplusCUnlockSensor);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampYminusIsLocked() {
        simulateAutochangerOnlineClampIsLocked(onlineClampYminusLockSensor,
                onlineClampYminusCLockSensor,
                onlineClampYminusUnlockSensor,
                onlineClampYminusCUnlockSensor);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampYminusIsUnlocked() {
        simulateAutochangerOnlineClampIsUnlocked(onlineClampYminusLockSensor,
                onlineClampYminusCLockSensor,
                onlineClampYminusUnlockSensor,
                onlineClampYminusCUnlockSensor);
    }
    
    /**
     * Changes the values of the gateway in order to simulate that the filter read by autochanger 
     * at STANDBY is the filterID of the filter on carousel at STANDBY.
     * This has to be done only for wholeFCS.
     * In autochanger-standalone mode this is not needed.
     */
    private void simulateFilterIDAtStandby() {
        if ((carousel instanceof CarouselModule) && ((CarouselModule)carousel).getFilterAtStandby() != null) {
                simulateFilterID(((CarouselModule)carousel).getFilterAtStandby().getFilterID());
        }
    }
    
    /**
     * Here we want to simulate the autochanger sensors to be consistent with the filter location.
     * If a filter is on autochanger, we suppose that it's at ONLINE position and onlineClamps are LOCKED,
     * and latches are LOCKED, and filter presence sensors detect a filter.
     */
    @Override
    public void start() {
        boolean aFilterIsOnTrucks = false;
        autochanger = (AutoChangerModule) getComponentLookup().getComponentByName("autochanger");
        FilterManager filterManager = (FilterManager) getComponentLookup().getComponentByName("filterManager");
        Map<String, Filter> filtersMap = filterManager.getFiltersMapByName();
        for (Map.Entry<String, Filter> entry : filtersMap.entrySet()) {
            Filter filter = entry.getValue();
            
            if (filter.isOnAutoChanger()) {
                FCSLOG.fine(filter.getName() + " is on autochanger trucks.");
                aFilterIsOnTrucks = true;
                autochanger.setFilterOnTrucksID(filter.getFilterID());
            } 
        }
        if (aFilterIsOnTrucks) {
                simulateAutochangerTrucksIsOnline();
                simulateFilterIsOnAutochanger();
                simulateAutochangerLatchesAreLocked();
                simulateAutochangerOnlineClampsAreLocked();
                
        } else {
                simulateAutochangerTrucksIsAtHandoff();
                simulateAutochangerIsEmpty();
                simulateAutochangerLatchesAreUnlocked();
                simulateAutochangerOnlineClampsAreUnlocked();
        }
    }
}
