
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.subsystems.fcs.common.TcpProxyInterface;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import static org.lsst.ccs.bus.states.AlertState.WARNING;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.CWrapperNotConnected;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;

/**
 * This Module starts a tcpip server, waits for the connection of a client whose
 name is the value of the field clientName. 
 * This client is supposed to be a CANopen C-wrapper. When the client starts it scans the CANopen 
 * network and send to the server some information on the CANopen bootedNodes living on the CAN bus.
 * The main job of the class CanOpenProxy is to check that the CANopen devices booted on the CAN bus
 * are the CANopen devices which are listed in the configuration system.
 * This class launch ALERTS :
 * - FCS001 in sendCanOpenCommand, readDeviceInfo
 * - FCS002 in processEmcyMessage, processErrorRegister
 * - FCS003 in processBootMessage
 * 
 * 
 *
 * @author virieux
 */
public class CanOpenProxy extends FcsTcpProxy implements HardwareController, 
        ClearAlertHandler, TcpProxyInterface {


    /*This is the key word that has to be sent by the client when it connects to the tcp proxy. */
    @ConfigurationParameter(isFinal=true)
    private String clientName;
    
    /**
     * A timeout for the booting process : during initialization, we wait for the boot messages 
     * coming from CANbus field.
     * When this amount of time is over, we don't wait any more.
     * This has to be tuned during the test bench. UNIT = milliseconds
     * 
     */
    @ConfigurationParameter(range="1000..10000",
            description="A timeout for the hardware booting process")
    private long hardwareBootTimeout = 1000;


    /**
     * An array of booted hardware.
     * This represents the list of CANopen nodes which are booted on the CANbus.
     */
    private Map<String,CanOpenNode> bootedNodes = new HashMap<>();

    private boolean hardwareBootProcessEnded = false;
    private boolean hardwareIdentified = false;




    
    /**
     * An array of CANopen hardware controled by this subsystem.
     * This represents the hardware listed in the configuration file which are 
     * children of BridgeToHardware and implement PieceOfHardware.
     * This list of hardware is initialized in the method initModule of the 
     * Module BridgeToCanOpenHardware.
     * protected because used by SimuCanOpenProxy
     */
    protected ArrayList<PieceOfHardware> hardwareList;
    
    /**
     * This map of PieceOfHardware which key is CANopen nodeID of the hardware
     * is used to simplify the process of hardwareList.
     * It's created when the hardwareList is initialized.
     * 
     */
    private Map<String,PieceOfHardware> hardwareMapByNodeID;

    private boolean canOpenNodeNumbersOK = false;
    private StringBuilder errorMessageSB;
    private boolean hardwareIDError;

    private PDOStorage pdoStorage;
    
    private ScheduledFuture<?> checkDevicesHandle;
    private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(1);
    //Used because we have to wait until all the pieces of hardware are booted.
    protected final Condition bootingCompleted = lock.newCondition();

    
    

    /**
     * Creates a CanOpenProxy with a tcpip port number to start the tcp server on,
     * a client name, and a hardware booting process timeout.
     * @param portNumber
     * @param fieldBusTimeout
     * @param clientName
     * @param hardwareBootTimeout 
     */
    public CanOpenProxy(
            int portNumber,
            int fieldBusTimeout,
            String clientName,
            long hardwareBootTimeout) {
        super(portNumber, fieldBusTimeout);
        this.clientName = clientName;
        this.hardwareBootTimeout = hardwareBootTimeout;
    }

    


    /**
     * Set hardwareList and initialize hardwareMapByNodeID. 
     * @param devices
     */
    protected void setHardwareList(ArrayList<PieceOfHardware> devices) {
        this.hardwareList = (ArrayList<PieceOfHardware>) devices.clone();
        this.hardwareMapByNodeID = new HashMap<>();
        hardwareList.stream().forEach((hardware) -> {
            this.hardwareMapByNodeID.put(hardware.getNodeID(), hardware);
        });               
    }

    /**
     * @return the clientName
     */
    public String getMyClientName() {
        return clientName;
    }

    public PDOStorage getPdoStorage() {
        return pdoStorage;
    }

    public boolean isHardwareIdentified() {
        return hardwareIdentified;
    }

    /**
     * For the simulator
     * @param hardwareIdentified 
     */
    public void setHardwareIdentified(boolean hardwareIdentified) {
        this.hardwareIdentified = hardwareIdentified;
    }

    /**
     * For the simulator
     * @param canOpenNodeNumbersOK 
     */
    public void setCanOpenNodeNumbersOK(boolean canOpenNodeNumbersOK) {
        this.canOpenNodeNumbersOK = canOpenNodeNumbersOK;
    }

    public void setHardwareBootProcessEnded(boolean hardwareBootProcessEnded) {
        this.hardwareBootProcessEnded = hardwareBootProcessEnded;
    }
    
    

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** END OF SETTERS AND GETTERS  **********************************************
     */
    /**
     * ***********************************************************************************************
     */
    /**
     * This methods stops the CWrapper client, and the TCP server.
     */
    @Override
    public void disconnectHardware() {
        //TODO we have to test if all the hardware is stopped before doing stopServer
        this.stopServer();
        initialize();
    }

    @Override
    public void connectHardware() throws HardwareException {
        this.startServer();
        this.startThreadReader();
    }

    @Override
    public void initModule() {
        super.initModule();
        initialize();
        /* To be able to clear alerts. */
        getSubsystem().addClearAlertHandler(this);
    }
    
    /**
     * This method initializes the fields of the tcpProxy.
     * It's used when we start the Module and when we disconnect hardware.
     */
    public void initialize() {
        bootedNodes = new HashMap<>();
        hardwareBootProcessEnded = false;
        hardwareIdentified = false;
        canOpenNodeNumbersOK = false;
        this.hardwareIDError = false;
        this.pdoStorage = new PDOStorage();
        errorMessageSB = new StringBuilder(String.valueOf(getName()));
    }


    
    
    /**
     * This method has to be tested on test benches. It should replace checkHardware.
     * 
     * @return
     * @throws HardwareException 
     */
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        hardwareBootProcessEnded = false;

        FCSLOG.debug(getName() + ": BEGIN CHECKHARDWARE");
        this.getSubsystem().updateAgentState(FilterState.CAN_DEVICES_BOOTING, 
                FilterReadinessState.NOT_READY);
        lock.lock();
        try {
            readDevicesInfo(System.currentTimeMillis(), this.hardwareBootTimeout);
            waitForEndOfBooting();

        } finally {
            lock.unlock();
        }
        
        FCSLOG.debug(getName() + ": END OF BOOT PROCESS");
        FCSLOG.info(getName() + "=>" + listBootedNodes());

        /* Here the task of reading devices info is completed because the timeout is reached or
        /* because the number of booted nodes is the number of expected hardware.
        */
        checkBootedCanOpenNodes();
        FCSLOG.debug(getName() + ": END CHECKHARDWARE");
        //for the GUI
        this.publishData();
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }
    
    /**
     * This method is called in the checkHardware method during the initialization process. 
     * It check devices at fixed rate with a scheduler until all devices are booted 
     * or booting timeout is exceeded.
     * @param beginTime
     * @param timeout 
     */
    public void readDevicesInfo(
            final long beginTime, final long timeout)  {

        final Runnable checkDevices = () -> {
            try {                
                // Retrieve some information from the booted CanOpen bootedNodes
                //(info,node_id)
                identifyHardware();
                this.publishData();
                long duration = System.currentTimeMillis() - beginTime;
                if (duration > timeout || bootedNodes.size() == hardwareList.size()) {
                    hardwareBootProcessEnded = true;
                    cancelReadDevicesInfo();
                } 

                
            } catch (HardwareException ex) {
                this.raiseAlarm("FCS001:", getName() + " couln't read devices info because:" + ex);
            }
        };
        checkDevicesHandle = scheduler.scheduleAtFixedRate(checkDevices, 
                500, 500, TimeUnit.MILLISECONDS);
    }
    
    
    /**
     * This stops the reading of the sensors.
     *
     */
    private void cancelReadDevicesInfo() {
        lock.lock();
        try {
            FCSLOG.debug(getName() + " => stop waiting for devices info.");
            bootingCompleted.signalAll();
        } finally {
            lock.unlock();
        }
        this.checkDevicesHandle.cancel(true);
        FCSLOG.debug(getName() + " => readDevicesInfo canceled");
    }
    
    /**
     * This method waits until the booting process is completed. 
     * This methods is called by checkHardwareToBeTested and has already acquired the lock.
     */
    private void waitForEndOfBooting() {
        while (!hardwareBootProcessEnded) {
            try {
                FCSLOG.info(getName() + " waiting until all pieces of hardware are booted.");
                bootingCompleted.await();
            } catch (InterruptedException ex) {
                FCSLOG.info(getName() + ": InterruptedException received=" + ex);
                break;
            }
        }
        FCSLOG.info(getName() + " STOP WAITING FOR END OF BOOTING PROCESS");
    }

    /**
     * checkNewHardware is called by command checkStarted() in higher level
     * Modules and is executed when a command completeInitialization is sent to
     * the subsystem. The command completeInitialization is sent by the end-user
     * to complete the initialization phase when the subsystem was stopped
     * during this phase because of an error (HardwareException).
     *
     * @throws org.lsst.ccs.HardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Retrieve information for booted devices and "
            + "checks if CANopen node ID and serial number match those"
            + "in description file .")
    @Override
    public void checkNewHardware() throws HardwareException {
        FCSLOG.info(getName() + " BEGIN checkStartedToCompleteInitialization");
        
        if (!(isReady(this.clientName))) {
            throw new HardwareException(true, getName() + ": not yet connected with CWrapper.");
        }
        if (!this.isTcpServerStarted()) {
            throw new HardwareException(true, getName()
                    + ": could not start tcp server.");
        }

        identifyHardware();

        checkBootedCanOpenNodes();

    }

    @Override
    public void checkStarted() throws HardwareException {
        FCSLOG.info(getName() + " BEGIN checkStarted");
    }

    @Override
    public void checkStopped() throws HardwareException {
        //TODO find something cleaver to be done here or delete this method.
    }
    
    /**
     * This shutdowns the scheduler. This is executed during CLOSING phase when
     * the CanOpenProxy is stopped. cf checkStopped
     */
    @Override
    public void shutdownNow() {
        super.shutdownNow();
        scheduler.shutdown();
    }

    /**
     * For a piece of hardware that this tcpProxy manages, this methods returns the name of the device
     * when the CANopen node id is given as argument.
     * @param aNodeID
     * @return
     */
    public String getNodeName(String aNodeID) {
        if (this.hardwareMapByNodeID.get(aNodeID) == null) {
            return "Unknown nodeID:"+aNodeID;
        } else return this.hardwareMapByNodeID.get(aNodeID).getName();
    }

    /**
     * Check if the String nodeID already exists in the bootedNodes table.
     * If it is booted it is on the bootedNodes table.    
     * It means that this node ID is on the can open bus.
     * @param aNodeID
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, 
            description = "Return true if the device, with nodeID given as argument, is booted.")
    public boolean isBooted(String aNodeID) {

        if (bootedNodes.isEmpty()) {
            return false;
        } else {
            return bootedNodes.containsKey(aNodeID);
        }        
    }

    /**
     * List the can open bootedNodes which are in the bootedNodes table.
     *
     * @return the list of can open bootedNodes and the information stored in
     * this.bootedNodes.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, 
            description = "Print the list of CANopen nodes which are booted on "
                    + "the CAN bus. (values are in HEXA)")
    public String listBootedNodes() {
        if (bootedNodes.isEmpty()) {
            return getName() + ": no booted CANopen devices.";
        } else {
            return getName() + " booted CANopen devices:" + bootedNodes.toString();
        }

    }

    /**
     * For end users, in engineering mode, this method can be used to send Can Open commands
     * to the Wrapper.
     * Test if the command is valid before sending it to the CAN bus.
     *
     * @param command A Can Open command that the Wrapper should understand.
     * @return the response from the Wrapper
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, 
            description = "Send a CanOpen command to the Can Bus.")
    public String sendCanOpenCommand(String command)  {
        if (!this.tcpServerStarted) {
            throw new FcsHardwareException(getName() + ": is not started, can't send CanOpen commands.");
        }       
        checkCanOpenCommand(command);
        return sendCanOpen(command);
    }
    
    
    /**
     * Tests that a CANOpen command is valid.
     * If the command is not valid it throws a IllegalArgumentException.
     * This is used in the sendCanOpen command method which can be entered by an end user in a console.
     *
     * @param command CANOpen command to be tested
     */
    protected void checkCanOpenCommand(String command) {
        
        if (command == null || command.isEmpty()) throw new IllegalArgumentException("null command");
        
        String[] words = command.split(",");
        String keyWord = words[0];
        
        if ("sync".equals(keyWord) || "scan".equals(keyWord) || "quit".equals(keyWord)) {           
            if (words.length == 1 && !command.endsWith(",")) {
                /*valid command*/
            } else {
                /*invalid command*/
                throw new IllegalArgumentException("Usage: " + keyWord);
            }
            
        } else if ("info".equals(keyWord) || "reset".equals(keyWord) || "srtr".equals(keyWord)) {
            if (words.length == 2 && !command.endsWith(",")) {
                /*valid command*/
                checkNodeID(words[1]);
            } else {
                /*invalid command*/
                throw new IllegalArgumentException("Usage: " + keyWord + ",nodeID");
            }
            
        } else if ("rsdo".equals(keyWord)) {
            if (words.length == 4 && !command.endsWith(",")) {
                /*valid command*/
                checkNodeID(words[1]);    
            } else {
                throw new IllegalArgumentException("Usage: " + "rsdo, nodeID, index, subindex");
            }
            
        } else if ("wsdo".equals(keyWord)) {
            if (words.length == 6 && !command.endsWith(",")) {
                /*valid command*/
                checkNodeID(words[1]);    
            } else {
                throw new IllegalArgumentException("Usage: " + "rsdo, nodeID, index, subindex,size, data");
            }
        
        } else throw new IllegalArgumentException(command + ":invalid CANopen command");
  
    }
    
    /**
     * Check that a nodeID entered by an end user at the console is a valid one:
     * - nodeID should be in the hardwareMapByNodeID.
     * @param nodeID 
     */
    public void checkNodeID(String nodeID) {
        if (!hardwareMapByNodeID.containsKey(nodeID)) throw new IllegalArgumentException(nodeID 
                + " is not in the hardware list for tcpProxy " + getName());
    }
    
    /**
     * Send a command on the CANbus.
     * This method is for internal usage.
     * For end user usage see sendCanOpenCommand.
     * @param command
     * @return
     * @throws FcsHardwareException 
     */
    private String sendCanOpen(String command)  {
        try {
            return (String) call(getMyClientName(), command);

        } catch (CWrapperNotConnected ex) {
            this.raiseAlarm("FCS001:"+getName(), "Can't communicate with hardware because "
                    + "CWrapper is not connected - check ethernet connections.",ex);
            throw new FcsHardwareException("CWrapper not connected.",ex);
            
        //TODO catch this in the code appelant    
        } catch (CanOpenCallTimeoutException ex) {
            String msg = getName() + ": timeout expired while waiting to a response from CANbus "
                    + "to command: " + command + " POWER FAILURE ? ";
            this.raiseWarning("FCS001:"+getName(), msg,ex);
            throw new FcsHardwareException(msg,ex);
        }
    }


    /**
     * Write a SDO request and send it to the can open stack, then analyses the
     * response or throws an exception if the request failed. 
     * The parameters are given in HEXA.
     * For internal usage. 
     *
     * @param nodeID FORMAT=HEXA
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=HEXA
     * @param value FORMAT=HEXA
     * @return "OK" if OK
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public String writeSDO(String nodeID, String index, String subindex, String size, String value) {
        String request = FcsUtils.buildWsdoCommand(nodeID, index, subindex, size, value);
        String sdoResponseLine = sendCanOpen(request);
        /*response of a writeSDO command is : wsdo,nodeID,errorCode*/
        String[] words = sdoResponseLine.split(",");
        
        String errorCode = words[2];
        String msg;
        String deviceName;
        switch (errorCode) {
            case "0":
                return "OK";
            case "-1":
                msg = getName() + ":wrong writeSDO command : " + sdoResponseLine;
                deviceName = getNodeName(nodeID);
                this.raiseAlarm(deviceName, msg);
                throw new SDORequestException(getName() + ":wrong writeSDO command : " + sdoResponseLine);
            default:
                String message = processSDORequestError(request,sdoResponseLine,nodeID,errorCode, ALARM);
                throw new SDORequestException(message);
        }
    }

    /**
     * Command to be used by the end user at the console. 
     * (for internal usage, see writeSDO(String,String,String,String,String);
     * Write a SDO message and send it to the can open stack. 
     * The parameters are given in the following format:
     *
     * @param nodeID FORMAT=hexa
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=decimal
     * @param value FORMAT=decimal
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, 
            description = "Send a CanOpen writeSDO command to the Can Bus. "
                    + " \nValues of the argument are to be given in hexadecimal format for: index and subindex"
                    + "\n and in decimal format for size and value."
                    + "\n size represents the number of bytes on which the value is encoded. See device documentation.")
    public String writeSDO(String nodeID, String index, String subindex, int size, int value) {
        checkNodeID(nodeID);
        if (size < 0 || size > 4) {
            throw new IllegalArgumentException("size must be > 0 and < 4");
        }
        return writeSDO(nodeID, index, subindex, Integer.toHexString(size), Integer.toHexString(value));
    }
    

    /**
     * Read a SDO with the given index and subindex RETURNS the value read in
     * hexa (String) if no error occured or returns the error code if a error
     * was detected.
     *
     * @param nodeID FORMAT=hexa
     * @param index FORMAT=hexa
     * @param subindex FORMAT=hexa
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    public String readSDO(String nodeID, String index, String subindex) 
            throws ShortResponseToSDORequestException, 
            FcsHardwareException {
            String request = FcsUtils.buildRsdoCommand(nodeID, index, subindex);
            
            String sdoLine = sendCanOpen(FcsUtils.buildRsdoCommand(nodeID, index, subindex));
            return processResponseToReadSDO(sdoLine,nodeID,request);
    }
    
    /**
     * Check a response to a readSDO command.
     * Return the data requested if the response is a valid response otherwise throw an Exception
     * of type SDORequestException or ShortResponseToSDORequestException.
     * @param response
     * @param nodeID
     * @param request
     * @return
     * @throws ShortResponseToSDORequestException
     * @throws SDORequestException 
     */
    public String processResponseToReadSDO(String response,String nodeID,String request) 
            throws ShortResponseToSDORequestException, SDORequestException {
            String[] words = response.split(",");
            int responseLength = words.length;
            String msg;
            String deviceName;
            /* response of a readSDO command should be : rsdo,nodeID,errorCode,data */
            String errorCode = words[2];

            if ("0".equals(errorCode) && (responseLength > 3)) {
                return words[3];

            } else if ("0".equals(errorCode) && responseLength == 3) {
                msg = getName() + ":readSDO request received a too short response=" + response;
                deviceName = getNodeName(nodeID);
                this.raiseWarning(deviceName, msg);
                throw new ShortResponseToSDORequestException(msg);

            } else if ("-1".equals(errorCode)) {
                msg = getName() + ":wrong readSDO command : " + response;
                deviceName = getNodeName(nodeID);
                this.raiseAlarm(deviceName, msg);
                throw new SDORequestException(msg);

            } else {
                String message = processSDORequestError(request,response,nodeID,errorCode,ALARM);
                throw new SDORequestException(message);
            }
    }
    
    /**
     * Command to be used by the end user at the console in engineering mode.
     * 
     * Send a CANOpen readSDO command to the Can Bus and returns the response in hexadecimal format.
     * Check CANOpen nodeID.
     * @param nodeID FORMAT=hexa
     * @param index FORMAT=hexa
     * @param subindex FORMAT=hexa
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, 
            description = "Send a CANOpen readSDO command to the Can Bus.")
    public String readSDOCommand(String nodeID, String index, String subindex)  {
        checkNodeID(nodeID);
        return readSDO(nodeID,index,subindex);
    }
    
    /**
     * When we receive a response with an error to a SDO request, this methods retreives the errorName
     * in the CANOpen tables, create an error message and raise an Alert.
     * 
     * @param request
     * @param response
     * @param errorCode
     * @param nodeID 
     * @param alertState 
     * @return msg a message to be added to an exception if needed. 
     */
    public String processSDORequestError(String request, String response, 
            String nodeID, String errorCode, AlertState alertState) {
                        String error = String.format("%08d", Integer.parseInt(errorCode));
        String errorName = CanOpenErrorsTable.getCommErrorNameByCode(error);
        String deviceName = getNodeName(nodeID);
        String msg = getName() + ": SDO request was going wrong for device: " + deviceName ;
        msg = msg + ",request="+request+",received response=" + response;
        msg = msg + ",errorCode="+errorCode + ",errorName=" + errorName;
        if (alertState.equals(WARNING)) {
            this.raiseWarning(deviceName, msg);
        }
        else {
            this.raiseAlarm(deviceName, msg);
        }
        return msg;
    }
    


    /**
     * This method sends a sync command to the can open stack and returns the
     * reply. example of sync reply we can receive :
     * sync,23_1=25a7,23_2=113,23_3=10,23_4=109,23_5=108,23_6=104,23_7=101,23_8=112
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public PDOStorage readPDOs()  {
        try {
            String replyToSync = sendCanOpen("sync");
            FCSLOG.finest(getName() + ": replyToSync=" + replyToSync);
            this.pdoStorage.updatePDOs(replyToSync);
            return this.pdoStorage;

        } catch (FcsHardwareException ex) {
            String msg = getName() + ": error in response to a sync command.";
            this.raiseAlarm("FCS001:", msg, ex);
            throw new FcsHardwareException(msg,ex);
        } 
    }



    /**
     * Identification of the hardware : we want to retrieve the information
     *  stored in the hardware of the booted bootedNodes and update the array of 
     * bootedNodes with this information. 
     * So for each booted node, we send a command :
     * - info,node_ID to the CWrapper and then we update the bootedNodes list 
     * with the information returned in the command result. 
     * The serial number is a main information to retrieve.
     * @throws org.lsst.ccs.HardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Identification of the hardware : we want to retrieve the information \n"
            + "     * stored in the hardware of the booted nodes and update the array of nodes\n"
            + "     * with this information.")
    public void identifyHardware() throws HardwareException {
        FCSLOG.info(getName() + ":Identification of the hardware");
        for (Map.Entry<String,CanOpenNode> entry: bootedNodes.entrySet()) {
            CanOpenNode bootedNode = entry.getValue();
            String bootedNodeID = entry.getKey();
            if ((bootedNode != null) && (!bootedNode.isIdentified())) {
                updateDeviceInfo(bootedNodeID);
            }
        }
        this.hardwareIdentified = true;
        publishData();
    }
    
    /**
     * This method sends a message info,nodeID to the CANbus and update the CANopen device
     * information from data received from the device.
     * @param bootedNodeID
     * @throws HardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update device information in sending to the CANbus a CANopen command : info,nodeID")
    public void updateDeviceInfo(String bootedNodeID) 
            throws HardwareException {
        try {
                FCSLOG.debug(getName() + ":Sending to can open command : info,"
                        + bootedNodeID);
                String result = sendCanOpen("info," + bootedNodeID);
                processInfoMessage(result, bootedNodeID);

            } catch (FcsHardwareException ex) {
                FCSLOG.error(ex);
                throw new HardwareException(false, ex);
            }
    }
    


    /**
     * This method returns true if : all the hardware items are booted and
     * identified and the hardware have the node ID expected within the
     * configuration and the hardware is initialized.
     *
     * @return a boolean
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all CANopen devices are booted and identified."
                    + "Identified means that the serial numbers match the CANopen nodeID which reside in"
                    + "configuration.")
    public boolean isCANDevicesReady() {
        FCSLOG.info(getName() + ":hardwareBootProcessEnded=" + hardwareBootProcessEnded);
        FCSLOG.info(getName() + ":hardwareIdentified=" + hardwareIdentified);
        FCSLOG.info(getName() + ":canOpenNodeNumbersOK=" + this.canOpenNodeNumbersOK);
        FCSLOG.info(getName() + ":hardwareIDError=" + this.hardwareIDError);
        return this.hardwareBootProcessEnded && this.hardwareIdentified
                && this.canOpenNodeNumbersOK && !this.hardwareIDError;
    }


    @Override
    void processBootMessage(String nodeID) {

        /*The node is already in the table of bootedNodes*/
        if (isBooted(nodeID)) {
            if (hardwareBootProcessEnded) {
                /*the boot process is ended: something weird has just happened*/
                String msg = getName() + " An unsynchronous boot message is arrived for node ID = " + nodeID
                        +" Possible power failure?";
                this.raiseWarning("FCS003:"+getName()+":"+nodeID,msg);
            }

            /*the node is not already in the table of bootedNodes : we have to add it*/
        } else {

            //TODO : if the number of bootedNodes we found is greater than the expectedNodesNB, 
            //throw an exception. 
            bootedNodes.put(nodeID, new CanOpenNode(nodeID));
            FCSLOG.info("Node " + nodeID + " added");
            FCSLOG.info("Number of booted devices=" + bootedNodes.size());
            FCSLOG.info("Number of expected devices=" + hardwareList.size());
            if (hardwareBootProcessEnded) {
                /*the boot process is ended: the device has been powered on ?*/
                String msg = getName() + ": boot message is arrived for node ID:"+ nodeID +" when boot process is completed. +"
                        + "Has the device been powered on ?";
                this.raiseWarning("FCS003:"+getName()+":"+nodeID, msg);
            }
        }
    }

    /**
     * Process an emergency message received from a CANopen device on the status bus :
     * - ALARM or WARNING Alert is raised,
     * - a message is logged,
     * - Observers are notified (CANopen devices are CanOpenProxy Observers).
     * @param message 
     */
    @Override
    //TODO change the name of this method to notifyEmcyMessage par exemple. Bof ! 
    public void processEmcyMessage(String message) {
        String[] words = message.split(",");
        /*NodeID without the leading zeros*/
        String nodeID = words[1].replaceFirst("^0+(?!$)", "");
        String deviceErrorCode = words[2];
        String errReg = words[3];
        
        String deviceErrorName = CanOpenErrorsTable.getDeviceErrorNameByCode(deviceErrorCode);
        String errorRegisterName = CanOpenErrorsTable.getErrorRegisterNameByCode(errReg);        
        String deviceName = this.getNodeName(nodeID);
        EmergencyMessage emcyMsg = new EmergencyMessage(getName(), nodeID, deviceName, deviceErrorCode,
                deviceErrorName, errReg, errorRegisterName);
        setChanged();
        this.notifyObservers(new ValueUpdate(getName(), emcyMsg));
        FCSLOG.finest(getName()+" getNObserverThreads()="+this.getNObserverThreads());
        FCSLOG.warning(getName() + " received EMERGENCY message="+message+ " for nodeID="+nodeID);

        if ("00".equals(errReg)) {
            this.raiseWarning("FCS002"+getName()+nodeID, emcyMsg.toString());
        } else {
            this.raiseAlarm("FCS002"+getName()+nodeID, emcyMsg.toString());
        }
    }
    
    /**
     * Process the response to a command info,node_ID.
     * @param message 
     */
    void processInfoMessage(String message, String bootedNodeID) {
        FCSLOG.debug(getName() + ":Received on socket command = " + message);
                String[] words = message.split(",");
                String command = words[0];
                String nodeID = words[1];

                if ("info".equals(command) && bootedNodeID.equals(nodeID)) {
                    FCSLOG.debug(getName() + ":updating Node Info for node " + nodeID);
                    String type = words[2];
                    String vendor = words[3];
                    String productCode = words[4];
                    String revision = words[5];
                    String serialNB = words[6];
                    bootedNodes.get(bootedNodeID).setNodeInfo(type, vendor, productCode, revision, serialNB);
                } else {
                    //this should not happen.
                    FCSLOG.error(getName() + ":ERROR for command = " + command
                            + "NodeID = " + nodeID);
                    //TODO throw an exception
                }
    }
    
    /**
     * Process a message received from the CAN bus when this message is an unknown message.
     * @param message 
     */
    @Override
    void processUnknownCommand(String message) {
        this.raiseWarning("FCS002"+getName(), "Unknown command received by CWrapper:" + message);
    }

    /**
     * Check if the list of the booted CANopen devices on the CANbus is
     * identical to the hardware list we have in the description.
     *
     * @throws HardwareException
     */
    public void checkBootedCanOpenNodes() throws HardwareException {
        
        FCSLOG.info(getName() + ":CHECKING HARDWARE CONFIGURATION ");

        checkNumberOfBootedNodes();
        hardwareIDError = false;
        
        for (PieceOfHardware hardware : this.hardwareList) {
            FCSLOG.info("ABOUT TO CHECK: " + hardware.getName() + " NODE_ID=" + hardware.getNodeID());
            checkHardwareID(hardware);           
        }

        if (hardwareIDError) {
            this.getSubsystem().updateAgentState(FcsEnumerations.FilterState.CAN_DEVICES_BOOTING, 
                FcsEnumerations.FilterReadinessState.NOT_READY);
            throw new HardwareException(true, errorMessageSB.toString());
        } else {
            this.getSubsystem().updateAgentState(FcsEnumerations.FilterState.HOMING_TO_BE_DONE, 
                FcsEnumerations.FilterReadinessState.NOT_READY);
        }
        publishData();
    }
    
    /**
     * This methods checks if the number of booted nodes is correct : it has to be equal 
     * to the number of hardware that this tcpProxy manages.
     * @throws HardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Checks if the number of booted nodes is correct : it has to be equal "
                    + "to the number of CANopen devices that this subsystem manages.")
    public void checkNumberOfBootedNodes() throws HardwareException {
        String msg;
        int expectedDevicesNumber = hardwareList.size();
        if (bootedNodes.isEmpty()) {
            msg = String.format(" (hardwareBootTimeout=%02d, expectedNodesNB=%02d)", 
                    this.hardwareBootTimeout, hardwareList.size());
            FCSLOG.error(getName() + ":NO HARDWARE DETECTED - POWER FAILURE ?" + msg);
            this.raiseAlarm("NO_HARDWARE_DETECTED", ":NO HARDWARE DETECTED - POWER FAILURE ?", getName() + " this could happen "
                    + "because of a POWER FAILURE on CAN bus, or on CANopen devices." + msg);
            throw new HardwareException(true, getName() + msg);
            
        //Added in Februar 2016 - has to be tested on test benches.
        } else if (bootedNodes.size() < expectedDevicesNumber) {
            msg = ":SOME HARDWARE IS MISSING - POWER FAILURE ?";
            FCSLOG.error(getName() + msg);
            this.raiseAlarm("FCS001:"+getName()+":HARDWARE_MISSING",":SOME HARDWARE IS MISSING - POWER FAILURE ?" + msg);
            throw new HardwareException(true, getName() + msg);
            
        } else if (bootedNodes.size() == expectedDevicesNumber) {
            this.canOpenNodeNumbersOK = true;
        }
    }

    /**
     * When the hardware is booted and we have retrieve the information (serial
     * number), we want to check for each piece of hardware that we have the
     * good node ID and the good serial number: so we compare the serial number
     * found on the CANopen bus (in Map bootedNodes) with the serial number in the configuration
     * file.
     *
     * @param pieceOfHardware
     * @throws org.lsst.ccs.HardwareException
     */
    public void checkHardwareID(PieceOfHardware pieceOfHardware)
            throws HardwareException {
        String msg;

        if (bootedNodes.containsKey(pieceOfHardware.getNodeID())) {
            //the node id of this piece of hardware has been found in the booted bootedNodes list.
            CanOpenNode bootedNode = bootedNodes.get(pieceOfHardware.getNodeID());
            
            if (bootedNode.getSerialNB().equals(pieceOfHardware.getSerialNB())) {
                //and it's the same serial ID : every thing is OK  
                pieceOfHardware.setBooted(true);
                FCSLOG.info(pieceOfHardware.getName() + " BOOTED AND SERIAL NUMBER OK:" 
                        + pieceOfHardware.toString());
                this.publishHardwareData(pieceOfHardware);
                
            } else {
                //bad serial number for this pieceOfHardware
                pieceOfHardware.setBooted(false);
                msg = pieceOfHardware.getName() + " has a wrong serial number. Serial number found="
                        + bootedNode.getSerialNB() + ", should be=" + pieceOfHardware.getSerialNB();
                                    FCSLOG.error(msg);
                errorMessageSB.append(msg);
                this.hardwareIDError = true;
                //TODO to be tested on test benches
                this.raiseAlarm("FCS001:"+getName()+":"+pieceOfHardware.getName(), msg);
                throw new HardwareException(false, getName() + msg);
            }

        } else {
            CanOpenNode bootedNode = getBootedNodeBySerialNumber(pieceOfHardware.getSerialNB());
            if ( bootedNode == null) {
                msg = String.format(":HARDWARE NOT DETECTED - "
                            + "Possible power failure for node ID %s ? - "
                            + "(hardware:%s,serial number:%s)",
                            pieceOfHardware.getNodeID(), pieceOfHardware.getName(), 
                            pieceOfHardware.getSerialNB());
                FCSLOG.error(msg);
                errorMessageSB.append(msg);
                this.hardwareIDError = true;
                //TODO to be tested on test benches
                this.raiseAlarm("FCS001:"+getName()+":"+pieceOfHardware.getName()+":HARDWARE_MISSING",
                        ":HARDWARE NOT DETECTED - POWER FAILURE ?"+msg);
                throw new HardwareException(false, getName() + msg);
            } else {
                //the serial number of this piece of hardware has been found in the bootedNodes list
                //with another node ID.
                pieceOfHardware.setBooted(false);
                msg = pieceOfHardware.getName() + ": a device is booted with serial number :"
                        + pieceOfHardware.getSerialNB() 
                        + ", but with CANopen nodeID=" + bootedNode.getNodeID()
                        + " Please change hardware description or hardware CANopen setting.";
                FCSLOG.error(msg);
                errorMessageSB.append(msg);
                this.hardwareIDError = true;
                //TODO to be tested on test benches
                this.raiseAlarm(pieceOfHardware.getName(), msg);
                throw new HardwareException(false, getName() + msg);
            }
        }
    }
    
    /**
     * Find a node by its serial number in the list of booted bootedNodes.
     * Returns null if this serial number is not on the CAN bus.
     * @param sn
     * @return CANopen node which serial number is sn or null.
     */
    public CanOpenNode getBootedNodeBySerialNumber(String sn) {
        for (Map.Entry<String,CanOpenNode> entry: bootedNodes.entrySet()) {
            CanOpenNode bootedNode = entry.getValue();
            if (bootedNode.getSerialNB().equalsIgnoreCase(sn)) {
                return bootedNode;
            }
        }
        return null;
    }
    
    /**
     * Returns a String representation of a CANopen node booted on the CANbus 
     * for the device which serial number is given as a parameter.
     * This is for the end users.
     * @param sn
     * @return a String
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Returns a String representation of a CANopen node booted"
            + " on the CANbus for the device"
            + " which serial number is given as a parameter.")
    public String printBootedNodeBySerialNumber(String sn) {   
        if (sn == null ) {
            throw new IllegalArgumentException(getName() + " Serial Number must be not null.");
        } else {
            return String.valueOf(getBootedNodeBySerialNumber(sn));
        }
    } 


    /**
     * Return a String with the list of hardware expected in this subsystem.
     * For debug purpose.
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware.")
    @Override
    public String printHardwareList() {
        return hardwareList.toString();
    }
    
    /**
     * Return a printed list of hardware with the initialization state.
     * For debug purpose.
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware with the initialization state.")
    @Override
    public String printHardwareState() {
        publishData();
        StringBuilder sb = new StringBuilder(getName() + " CANopen devices : {");
        for (PieceOfHardware pieceOfHardware : this.hardwareList) {
            sb.append(pieceOfHardware.printState());
            sb.append(';');
        }
        sb.append('}');
        return sb.toString();
    }    
    
    /**
     * Return an Array containing hardware names handled by this component.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return an Array containing hardware names handled by this component.")
    @Override
    public List<String> listHardwareNames() {
        List<String> res = new ArrayList<>();
        for (PieceOfHardware pieceOfHardware : this.hardwareList) {
            res.add(pieceOfHardware.getName());
        }
        return res;
    }
    
    /**
     * Overridden method from ClearAlertHandler interface to define conditions when Alerts can be cleared.
     * 
     * @param alert
     * @return 
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert) {
        switch (alert.getAlertId()) {
            case "FCS001":
                if (this.clientContext.socket.isConnected() 
                        && !this.clientContext.socket.isClosed()) {
                    return ClearAlertCode.CLEAR_ALERT;
                } else {
                    return ClearAlertCode.DONT_CLEAR_ALERT;
                }
            
            case "FCS002":
                return ClearAlertCode.CLEAR_ALERT;
                
            default:
                return ClearAlertCode.UNKWNOWN_ALERT;
        }
    }
    
    


    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Publish booting information for all CANopen devices.")
    @Override
    public void publishData() {
        for (PieceOfHardware device : hardwareList) {
            publishHardwareData(device);
        }
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     * @param device
     */
    public void publishHardwareData(PieceOfHardware device) {
        StatusDataPublishedByHardware status
                = FcsUtils.createStatusDataPublishedByHardware(device);
        FCSLOG.debug(getName() + ":publishHardwareData is publishing:" + status);
        KeyValueData kvd = new KeyValueData(getName(), status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }
    
    /**
     * Publish Data on status bus for trending data base and GUIs for a CANopen device.
     * @param nodeID 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Publish booting information for a CANopen device given by its CANopen nodeID.")   
    public void publishHardwareData(String nodeID) {
        //TODO
    }

    /**
     * For tests. Return a printed format of PDOStorage.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return a printed format of PDOStorage.")
    public String printPDOStorage() {
        return this.pdoStorage.toString();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append("/clientName=");
        sb.append(this.clientName);
        sb.append("/hardwareBootTimeout=");
        sb.append(this.hardwareBootTimeout);
        return sb.toString();
    }

}
