/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;

/**
 * A class to represent the CANopen pluto gateway which is a device where we can read sensors.
 * Its raises an alert :
 * - FCS004 in readNewHexaValues
 * 
 * @author virieux
 */
public class CanOpenPlutoGateway extends PlutoGatewayModule {

    //identifie pluto address on gateway
    @ConfigurationParameter
    private final String station_addr;

    /**
     * build a new CanOpenPlutoGateway
     * @param nodeID
     * @param serialNB
     * @param station_addr 
     */
    public CanOpenPlutoGateway(String nodeID, String serialNB, String station_addr) {
        super(nodeID, serialNB);
        this.station_addr = station_addr;
    }

    public String getStation_addr() {
        return station_addr;
    }


    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Initialize the Gateway.")
    public String initializeAndCheckHardware()  {
        if (!isBooted()) {
            throw new FcsHardwareException(name + " is not booted - can't be initialized.");
        }
        try {
            FCSLOG.info(name + " BEGIN initializeHardware");
            this.configurePlutoForThreeBlocs();
            this.initialized = true;
            FCSLOG.info(name + " is INITIALIZED.");
            //publishData for the LoaderHardwareGUI
            this.tcpProxy.publishHardwareData(this);
            return name + " is INITIALIZED.";

        } catch (SDORequestException | FailedCommandException ex) {
            throw new FcsHardwareException(name,ex);
        }
    }

    @Override
    public void initModule() {
        super.initModule();
        Map.Entry<String, Object> entry = getComponentLookup().getParent(getName());
        BridgeToHardware bridge = (BridgeToHardware) entry.getValue();
        tcpProxy = (CanOpenProxy) bridge.getTcpProxy();
    }

    /**
     * Assuming Pluto has 3 fonctions Gateway_to_User_C First one =>
     * Fonction_ID=01 Second one => Fonction_ID=02 Third one => Fonction_ID=03
     *
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Configure this gateway to receive 3 data blocs from Pluto.")
    //TODO cf voir ce qui est fait dans frontBox_Tests.
    public String configurePlutoForThreeBlocs()  {

        //Disable additional Data
        this.tcpProxy.writeSDO(getNodeID(), "2010", "01", "01", "01");

        //Set Additionnal data TPDO1 => set Area_0 Function_ID = 01
        this.tcpProxy.writeSDO(getNodeID(), "2011", "01", "02", station_addr + "01"); //define AREA0 (doc p:96)

        //Set Additionnal data TPDO1 => set Area_1 Function_ID = 02
        this.tcpProxy.writeSDO(getNodeID(), "2011", "02", "02", station_addr + "02"); //define AREA1 (doc p:96)

        //Set Additionnal data TPDO2 => set Aera_2 Function_ID = 03
        this.tcpProxy.writeSDO(getNodeID(), "2012", "01", "02", station_addr + "03"); //define AREA2 (doc p:96)

        //Activate additional data for Area_0, Area_1 and Area_2
        this.tcpProxy.writeSDO(getNodeID(), "2002", "01", "01", "07");

        return name + " is configured.";
    }

    /**
     * Write a transmission type in the CPU of the pluto gateway.
     * @param transmissionType
     * @return
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    public String writeTransmissionType(String transmissionType)  {
        if (transmissionType == null) {
            throw new IllegalArgumentException(getName() + ": Can't write transmission type to device because "
                    + "transmission type is null.");
        }
        String ack = this.tcpProxy.writeSDO(getNodeID(), "2005", "1", "1", transmissionType);
        String goodAck = "OK";
        if (!ack.equals(goodAck)) {
            throw new FailedCommandException(getName() + " :bad returned ack for wsdo, index 1801");
        }

        return String.format("Transmission type of device %s is now %s", this.name, transmissionType);
    }

    /* 
     * @return values in hexa read on the device
     */
    @Override
    public String[] readNewHexaValues()  {

        if (!isBooted()) {
            throw new FcsHardwareException(name
                    + " is not booted - can't read new values.");
        }
        try {
            //FOR THE LOADER TEST BENCH
//            hexaValues[0] = tcpProxy.readSDO(getNodeID(), "6000", "03");
//            FCSLOG.finest(String.format("%s : first byte read in hexaValue= %s", name, hexaValues[0]));
//            hexaValues[1] = tcpProxy.readSDO(getNodeID(), "6000", "04");
//            FCSLOG.finest(String.format("%s : second byte read in hexaValue= %s", name, hexaValues[1]));
            
            /*Index 6000*/
            hexaValues[0] = tcpProxy.readSDO(getNodeID(), "6000", "01");
            FCSLOG.finest(String.format("%s : first byte read in hexaValue= %s", name, hexaValues[0]));
            hexaValues[1] = tcpProxy.readSDO(getNodeID(), "6000", "02");
            FCSLOG.finest(String.format("%s : second byte read in hexaValue= %s", name, hexaValues[1]));            
            hexaValues[2] = tcpProxy.readSDO(getNodeID(), "6000", "03");
            FCSLOG.finest(String.format("%s : third byte read in hexaValue= %s", name, hexaValues[2]));
            hexaValues[3] = tcpProxy.readSDO(getNodeID(), "6000", "04");
            FCSLOG.finest(String.format("%s : 4th byte read in hexaValue= %s", name, hexaValues[3]));
            
            /*Index 6001*/
            hexaValues[4] = tcpProxy.readSDO(getNodeID(), "6001", "01");
            FCSLOG.finest(String.format("%s : 5th byte read in hexaValue= %s", name, hexaValues[4]));
            hexaValues[5] = tcpProxy.readSDO(getNodeID(), "6001", "02");
            FCSLOG.finest(String.format("%s : 6th byte read in hexaValue= %s", name, hexaValues[5]));            
            hexaValues[6] = tcpProxy.readSDO(getNodeID(), "6001", "03");
            FCSLOG.finest(String.format("%s : 7th byte read in hexaValue= %s", name, hexaValues[6]));
            hexaValues[7] = tcpProxy.readSDO(getNodeID(), "6001", "04");
            FCSLOG.finest(String.format("%s : 8th byte read in hexaValue= %s", name, hexaValues[7])); 
            
            /*Index 6002*/
            hexaValues[8] = tcpProxy.readSDO(getNodeID(), "6002", "01");
            FCSLOG.finest(String.format("%s : 9th byte read in hexaValue= %s", name, hexaValues[8]));
            hexaValues[9] = tcpProxy.readSDO(getNodeID(), "6002", "02");
            FCSLOG.finest(String.format("%s : 10th byte read in hexaValue= %s", name, hexaValues[9]));            
            hexaValues[10] = tcpProxy.readSDO(getNodeID(), "6002", "03");
            FCSLOG.finest(String.format("%s : 11th byte read in hexaValue= %s", name, hexaValues[10]));
            hexaValues[11] = tcpProxy.readSDO(getNodeID(), "6002", "04");
            FCSLOG.finest(String.format("%s : 12th byte read in hexaValue= %s", name, hexaValues[11]));  
            
            /*Index 6003*/
            hexaValues[12] = tcpProxy.readSDO(getNodeID(), "6003", "01");
            FCSLOG.finest(String.format("%s : 13th byte read in hexaValue= %s", name, hexaValues[12]));
            hexaValues[13] = tcpProxy.readSDO(getNodeID(), "6003", "02");
            FCSLOG.finest(String.format("%s : 14th byte read in hexaValue= %s", name, hexaValues[13]));            
            hexaValues[14] = tcpProxy.readSDO(getNodeID(), "6003", "03");
            FCSLOG.finest(String.format("%s : 15th byte read in hexaValue= %s", name, hexaValues[14]));
            hexaValues[15] = tcpProxy.readSDO(getNodeID(), "6003", "04");
            FCSLOG.finest(String.format("%s : 16th byte read in hexaValue= %s", name, hexaValues[15]));             
            
            
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING SENSOR:" + ex);

        } catch (SDORequestException ex) {
            String message = name + "=> ERROR IN READING SENSOR:";
            this.raiseAlarm("FCS004:"+name, message, ex);
        }
        return hexaValues;
    }
       

    @Override
    public int readNewAnalogValue()  {
        //read the registers
        String hexaL = tcpProxy.readSDO(getNodeID(), "6001", "01");
        FCSLOG.finest(name + ": new analog value in hexa=" + hexaL);
        String hexaM = tcpProxy.readSDO(getNodeID(), "6001", "02");
        String valueInHexa = hexaM + hexaL;
        FCSLOG.finest(name + ": new analog value in hexa=" + valueInHexa);
        int newAnalogValue = Integer.parseInt(valueInHexa, 16);
        FCSLOG.finest(name + ": new analog value =" + newAnalogValue);
        return newAnalogValue;
    }


}
