/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.math.BigInteger;
import java.util.Map;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * A general model for a controller EPOS.
 *
 * @author virieux 
 */
public class CanOpenEPOS extends CanOpenDevice implements EPOSController {
    
    private boolean turnedOff;

    /**
     * used in checkParameters and initializeAndCheckHardware methods
     */
    private boolean parametersOK;

    /* Mode of Operation */
    protected EposMode mode;

    /* Parameters */
    @ConfigurationParameter
    private Map<String, Integer> paramsForHoming;
    
    @ConfigurationParameter
    private Map<String, Integer> paramsForProfilePosition;
    
    @ConfigurationParameter
    private Map<String, Integer> paramsForCurrent;

    /**
     * Used to publish on the status bus for the GUI*
     */
    private boolean enabledToPublish;


    /**
     * Construction of a CanOpenEPOS.
     * @param nodeID
     * @param serialNB
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming 
     */
    public CanOpenEPOS(
            String nodeID, String serialNB,
            Map<String, Integer> paramsForCurrent,
            Map<String, Integer> paramsForProfilePosition,
            Map<String, Integer> paramsForHoming) {
        super(nodeID, serialNB);
        this.paramsForCurrent = paramsForCurrent;
        this.paramsForProfilePosition = paramsForProfilePosition;
        this.paramsForHoming = paramsForHoming;
        turnedOff = false;
        parametersOK = true;
        enabledToPublish = false;
    }

    @Override
    public boolean isEnabledToPublish() {
        return enabledToPublish;
    }


    @Override
    public boolean isTurnedOff() {
        return turnedOff;
    }
    
    /**
     * 
     * @return true if parameters in the CPU of the controllers
     * have the same values than in configuration system.
     */
    @Override
    public boolean isParametersOK() {
        return parametersOK;
    }

    @Override
    public Map<String, Integer> getParamsForHoming() {
        return paramsForHoming;
    }

    @Override
    public Map<String, Integer> getParamsForProfilePosition() {
        return paramsForProfilePosition;
    }

    @Override
    public Map<String, Integer> getParamsForCurrent() {
        return paramsForCurrent;
    }
    
    @ConfigurationParameterChanger
    public void setParamsForCurrent(Map<String, Integer> newMap) {
        this.paramsForCurrent.putAll(newMap);
        this.writeParameters(newMap);
        this.saveParameters();
    }
    
    @Command(description="Change a parameter for mode CURRENT, and write the new value on the CPU of the controller.",
            type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void changeParameterForCurrent(String clef, int value) {
        if (paramsForCurrent.containsKey(clef)) {
            paramsForCurrent.put(clef, value);
            String response = this.writeParameter(Parameter.valueOf(clef), value);
            if (response.equals("OK")) {
                getComponentConfigurationEnvironment().change("paramsForCurrent",paramsForCurrent);
                
            } else {
                throw new IllegalArgumentException(getName()+" couldn't write parameter: "+clef+"with this value:"+ value);
            }
        } else {
            throw new IllegalArgumentException(getName()+" paramsForCurrent doesn't contain this parameter:"+clef);
        }
    }
    
    @Command(description="Change a parameter for mode ProfilePosition, and write the new value on the CPU of the controller.",
        type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void changeParameterForProfilePosition(String clef, int value) {
        if (paramsForProfilePosition.containsKey(clef)) {
            paramsForProfilePosition.put(clef, value);
            String response = this.writeParameter(Parameter.valueOf(clef), value);
            if (response.equals("OK")) {
                getComponentConfigurationEnvironment().change("paramsForProfilePosition",paramsForProfilePosition);
                
            } else {
                throw new IllegalArgumentException(getName()+" couldn't write parameter: "+clef+"with this value:"+ value);
            }
        } else {
            throw new IllegalArgumentException(getName()+" paramsForProfilePosition doesn't contain this parameter:"+clef);
        }
    }
    
    @Command(description="Change a parameter for mode Homing, and write the new value on the CPU of the controller.",
        type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void changeParameterForHoming(String clef, int value) {
        if (paramsForHoming.containsKey(clef)) {
            paramsForHoming.put(clef, value);
            String response = this.writeParameter(Parameter.valueOf(clef), value);
            if (response.equals("OK")) {
                getComponentConfigurationEnvironment().change("paramsForHoming",paramsForHoming);
                
            } else {
                throw new IllegalArgumentException(getName()+" couldn't write parameter: "+clef+"with this value:"+ value);
            }
        } else {
            throw new IllegalArgumentException(getName()+" paramsForHoming doesn't contain this parameter:"+clef);
        }
    }    

    /**
     * Return Epos mode for GUI.
     * @return 
     */
    @Override
    public EposMode getMode() {
        return mode;
    }

    @Override
    public int getMaxCurrent()  {
        return readParameter(Parameter.OutputCurrentLimit);
    }

    @Override
    public int getMinCurrent()  {
        return readParameter(Parameter.OutputCurrentLimit);
    }

    @Override
    public int getMaxSpeed()  {
        return readParameter(Parameter.MaxProfileVelocity);
    }

    @Override
    public int getMaxPosition()  {
        return readParameter(Parameter.MaxPositionLimit);
    }

    @Override
    public int getMinPosition()  {
        return readParameter(Parameter.MinPositionLimit);
    }








    /**
     * Return the value configured for a motor parameter.
     * Doesn't read the CPU of the controller.
     * @param parameterName
     * @return
     * @throws EPOSConfigurationException 
     */
    public int getMotorParameterValue(String parameterName) throws EPOSConfigurationException {
        if (this.paramsForCurrent.containsKey(parameterName)) {
            return this.paramsForCurrent.get(parameterName);
        } else {
            throw new EPOSConfigurationException(parameterName + " :undefined", getName());
        }
    }

    /**
     * Print the configured parameters for this EPOS for a mode name given in
     * argument. This methods doesnt't read the CPU of the EPOS. So if no
 writeParameters(modeIS) command has been done before, the values in
 the EPOS CPU can be different.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Print the configured parameters for this EPOS for this mode."
            + "This command doesn't read the controlle.")
    public String printParameters()  {
        if (mode == null) {
            mode = readMode();
        }
        return printConfigurationParameters(this.mode.toString());
    }

    /**
     * Print the configured parameters for this EPOS for a mode name given in
     * argument. This methods doesnt't read the CPU of the EPOS. So if no
 writeParameters(modeIS) command has been done before, the values in
 the EPOS CPU can be different.
     *
     * @param modeInString
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Print the configured parameters for this EPOS for a mode name "
            + "given in argument. This command doesn't read the controller.")
    public String printConfigurationParameters(String modeInString) {
        if (modeInString == null) 
            throw new IllegalArgumentException("Mode name can't be null.");
        
        EposMode aMode = EposMode.valueOf(modeInString.trim().toUpperCase());
        Map<String, Integer> paramsMap = null;
        switch (aMode) {
            case HOMING:
                paramsMap = this.paramsForHoming;
                break;
                
            case PROFILE_POSITION:
                paramsMap = this.paramsForProfilePosition;
                break;
                
            case CURRENT:
                paramsMap = this.paramsForCurrent;
                break;
                
            default:
                throw new IllegalArgumentException(modeInString +"  is not a valid EPOS mode");
        }
        if (paramsMap == null) {
            return "NO parameter defined in configuration file for mode:"+ aMode;
        } else {
            return paramsMap.toString();
        }
    }

    /**
     * Reads the EPOS mode in the CPU of the EPOS device, updates the field mode
     * and returns the mode as a String.
     *
     * @return
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads the EPOS mode in the CPU of the EPOS device and "
            + "updates the field mode.")
    @Override
    public EposMode readMode()  {
        String modeInHexa;
        modeInHexa = this.readSDO("6061", "0").trim().toUpperCase();
        EposMode readMode = EposMode.getMode(modeInHexa);
        this.mode = readMode;
        return readMode;
    }


    /**
     * Changes the mode to the new mode given as a String argument.
     *
     * @param modeInString
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, 
            description = "Change EPOS mode .")
    public String changeMode(String modeInString)  {
        EposMode newMode = EposMode.valueOf(modeInString.trim().toUpperCase());
        return changeMode(newMode);
    }

    /**
     * This methods changes the mode to the new mode given as an argument. It
     * writes the new mode in the CPU of the EPOS controller and updates the
     * field mode.
     *
     * @param newMode
     * @return
     * @throws SDORequestException
     */
    @Override
    public String changeMode(EposMode newMode)  {
        //TODO check if the mode is already newMode don't change anything.
        this.writeSDO("6060", "0", "1", newMode.getModeInHexa());
        this.mode = newMode;
        publishData();
        return getName() + " mode has been changed to: " + newMode.toString()
                + ". Please check parameters associated with this mode.";
    }


    /**
     * This methods reads in the CPU of the EPOS the values of the parameters
     * for a given mode.
     *
     * @param modeInString
     * @return
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "This methods reads in the CPU of the EPOS the values "
                    + "of the parameters for a given mode.")
    public String readParameters(String modeInString)  {
        return readParameters(EposMode.valueOf(modeInString));
    }

    @Override
    public String readParameters(EposMode aMode)  {
        String modeIS = aMode.toString();
        StringBuilder sb = new StringBuilder(getName() + " parameters VALUES in decimal format for mode ");
        sb.append(modeIS);
        sb.append("\n");
        Parameter[] params = aMode.getParameters();
        for (Parameter param : params) {
            sb.append(param.toString());
            sb.append("=");
            int valueInt = readParameter(param);
            sb.append(valueInt);
            sb.append("\n");
        }
        FCSLOG.debug(getName() + sb.toString());
        return sb.toString();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the parameters for the actual mode.")
    public String readParameters()  {
        String modeStr = readMode().toString();
        return readParameters(modeStr);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "This methods writes in the CPU of the EPOS device the values "
            + "of the parameters set for the mode.")
    public void writeParameters()  {
        readMode();
        writeParameters(this.mode);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "This methods writes in the CPU of the EPOS device the values "
            + "of the parameters set for the mode given as argument")
    public void writeParameters(String modeInString)  {
        EposMode aMode = EposMode.valueOf(modeInString);
        writeParameters(aMode);
    }

    /**
     * This methods writes in the CPU of the EPOS devis the values of the
     * parameters set for the mode. Usually the values of the parameters are
     * given by the configuration system.
     *
     * @param mode
     * @throws SDORequestException
     */
    @Override
    public void writeParameters(EposMode mode)  {
        String modeIS = mode.toString();
        Map<String, Integer> paramsMap = null;
        switch (mode) {
            case HOMING:
                paramsMap = this.paramsForHoming;
                break;
                
            case PROFILE_POSITION:
                paramsMap = this.paramsForProfilePosition;
                break;

            case CURRENT:
                paramsMap = this.paramsForCurrent;
                break;
            
            default:
                assert false : mode;
        }
        if (paramsMap == null) {
            throw new EPOSConfigurationException(getName()
                    + "parameters for mode :" + modeIS + "are NULL");
        }
        if (paramsMap.isEmpty()) {
            throw new EPOSConfigurationException(getName()
                    + "parameters for mode :" + modeIS + "are not defined.");
        }
        writeParameters(paramsMap);

    }

    /**
     * Write to the CPU of the EPOS a map of parameters.
     *
     * @param paramMap
     * @throws SDORequestException
     */
    @Override
    public void writeParameters(Map<String, Integer> paramMap) {
        for (Map.Entry<String, Integer> entry : paramMap.entrySet()) {
            String paramName = entry.getKey();
            int value = entry.getValue();
            writeParameter(paramName, value);
        }
    }

    /**
     * This method writes a parameter in the CPU of the EPOS.
     *
     * @param parameterName the name of the parameter
     * @param value in decimal
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Set EPOS parameter with the value given as argument.")
    public String writeParameter(String parameterName, int value) {
        Parameter param = Parameter.valueOf(parameterName);
        return this.writeParameter(param, value);
    }

    /**
     * Set the parameter given as argument with the decimal value given as
     * argument and write it in the controller CPU.
     *
     * @param param parameter to be set
     * @param value decimal value
     * @return
     * @throws SDORequestException
     */
    public String writeParameter(Parameter param, int value) {
        writeSDO(param.getIndex(), param.getSubindex(), param.getSize(), 
                Integer.toHexString(value));
        return String.format("%s has been written to the CanEpos CPU, value=%d", 
                param.toString(), value);
    }

    /**
     * This method writes a parameter in the CPU of the EPOS.
     *
     * @param parameterName the name of the parameter
     * @param hexaValue value of the parameter given in hexa
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Set EPOS parameter with the hexadecimal value given as argument.")
    public String writeParameterInHexa(String parameterName, String hexaValue) {
        Parameter param = Parameter.valueOf(parameterName);
        return this.writeParameterInHexa(param, hexaValue);
    }

    /**
     * Set the parameter given as argument with the hexadecimal value given as
     * argument and write it in the controller CPU.
     *
     * @param param
     * @param hexaValue
     * @return
     * @throws SDORequestException
     */
    @Override
    public String writeParameterInHexa(Parameter param, String hexaValue) {
        writeSDO(param.getIndex(), param.getSubindex(), param.getSize(), hexaValue);
        return String.format("%s has been written to the CanEpos CPU, value=%s", 
                param.toString(), hexaValue);
    }

    /**
     * Reads in the EPOS CPU the value of the Parameter which parameter name is
     * given as argument.
     *
     * @param parameterName name of the parameter to read
     * @return value of the parameter in decimal format.
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads in the EPOS CPU the decimal value of the Parameter "
            + "which parameter name is given as argument.")
    public int readParameter(String parameterName)  {
        Parameter param = Parameter.valueOf(parameterName);
        return readParameter(param);
    }
    

    /**
     * Reads in the EPOS CPU the value of the Parameter.
     *
     * @param param
     * @return value of the parameter in decimal format.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public int readParameter(Parameter param)  {
        try {
            String valueInHexa = readSDO(param.getIndex(), param.getSubindex());
            int valueInDecimal = new BigInteger(valueInHexa, 16).intValue();
            FCSLOG.debug(getName() + " readParameter:" + param.name() + "=" + valueInDecimal
                    + " in decimal format  (valueInHexa=" + valueInHexa + ")");
            return valueInDecimal;
        } catch (SDORequestException | ShortResponseToSDORequestException ex) {
            String msg = getName() + " Error in reading Parameter: "+param;
            this.raiseWarning(getName(), msg, ex);
        } 
        return 0;
    }

    /**
     * Reads in the EPOS CPU the hexadecimal value of the Parameter.
     *
     * @param parameterName
     * @return value of the parameter in hexa format.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads in the EPOS CPU the hexadecimal value of the Parameter "
            + "which parameter name is given as argument.")
    public String readParameterInHexa(String parameterName)  {
        Parameter param = Parameter.valueOf(parameterName);
        return readParameterInHexa(param);
    }

    /**
     * Reads in the EPOS CPU the value of the Parameter.
     *
     * @param param
     * @return value of the parameter in hexa format.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public String readParameterInHexa(Parameter param)  {
        return readSDO(param.getIndex(), param.getSubindex());
    }

    /**
     * Return true if after a command writeTarget(position), the position is reached.
     * @param position
     * @return
     * @throws FcsHardwareException 
     */
    @Override
    public boolean isTargetPositionReached(int position)  {
        return this.readPosition() == position;
    }



    /**
     * Check if the parameters in the cPU of the controller have the right values.
     * And if the controller is not in fault.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read controller's parameter in the CPU for each mode and "
            + "warn if a parameter has not the same value as in the configuration."
            + "Then publish data.")
    @Override
    public String initializeAndCheckHardware()  {
        if (!isBooted()) {
            throw new FcsHardwareException(getName()
                    + " is not booted - can't be initialized.");
        }
        this.mode = readMode();
        this.parametersOK = true;
        checkParameters(EposMode.CURRENT);
        checkParameters(EposMode.PROFILE_POSITION);
        checkParameters(EposMode.HOMING);
        checkFault();
        this.initialized = true;
        if (!parametersOK) {
            FCSLOG.warning(getName() + " Some parameter values are not"
                    + " the same in CPU and configuration system.");
        }
        //publishData for the GUI
        this.tcpProxy.publishHardwareData(this);
        this.publishData();
        FCSLOG.info(getName() + ": is INITIALIZED.");
        return getName() + ": is INITIALIZED.";
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the controller is enabled.")
    @Override
    public boolean isEnabled()  {
        try {
            String statusWordInHexa = readStatusWord();
            int statusWord = Integer.parseInt(statusWordInHexa, 16);
            int[] statusWordInBinary = FcsUtils.toReverseBinary(statusWord);
            FCSLOG.debug(getName() + ":statusWordInBinary[0]=" + statusWordInBinary[0]);
            FCSLOG.debug(getName() + ":statusWordInBinary[1]=" + statusWordInBinary[1]);
            FCSLOG.debug(getName() + ":statusWordInBinary[2]=" + statusWordInBinary[2]);
            FCSLOG.debug(getName() + ":statusWordInBinary[3]=" + statusWordInBinary[3]);
            return (statusWordInBinary[0] == 1)
                    && (statusWordInBinary[1] == 1)
                    && (statusWordInBinary[2] == 1)
                    && (statusWordInBinary[3] == 0);

        } catch (ShortResponseToSDORequestException ex) {
            String msg = getName() + " couldn't read status word because ShortResponseToSDORequestException :" + ex.getMessage();
            FCSLOG.error(msg);
            FCSLOG.error(ex);
            throw new FcsHardwareException(msg,ex);

        }
    }



    /**
     * This methods enables the controller : i.e. this makes it able to receive
     * commands.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "This methods enables the controller : i.e. this makes it able to receive commands.")
    @Override
    public String enable()  {
        shutdown();
        switchOnEnableOperation();
        enabledToPublish = true;
        FCSLOG.debug(getName() + ": is ENABLED.");
        return getName() + " ENABLED";
    }

    /**
     * Shutdown the controller.
     *
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Shutdown the controller.")
    @Override
    public String shutdown()  {
        writeControlWord("6");
        //TODO test the return code for the command writeSDO
        FCSLOG.debug(getName() + ": is SHUTDOWN.");
        return getName() + " DISABLED";
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "writeControlWord(\"F\")")
    @Override
    public void switchOnEnableOperation()  {
        writeControlWord("F");
    }

    /**
     * Restore default parameters.
     *
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Restore default parameters. AT YOUR OWN RISK.")
    public Object restoreParameters()  {
        //TODO if is Power Disabled
        return this.writeSDO("1011", "1", "4", "64616F6C");
    }

    /**
     * Write a value in hexa in the control world (index=6040,
     * subindex=0,size=2)
     *
     * @param value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Write a value in hexa in the control world (index=6040, subindex=0,size=2)")
    @Override
    public void writeControlWord(String value)  {
        this.writeSDO("6040", "0", "2", value);
    }

    /**
     * Read the control world (index=6040, subindex=0,size=2)
     *
     * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the control world (index=6040, subindex=0,size=2)")
    public String readControlWord()  {
        return this.readSDO("6040", "0");
    }

    /**
     * Read the status world (index=6041, subindex=0,size=2)
     *
     * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the status world (index=6041, subindex=0,size=2)")
    public String readStatusWord()  {
        return this.readSDO("6041", "0");
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "QuickStop.")
    @Override
    public void quickStop()  {

        FCSLOG.debug(getName() + " running QUICKSTOP command.");
        
        if (this.mode.equals(EposMode.HOMING)||this.mode.equals(EposMode.PROFILE_POSITION)
                ||this.mode.equals(EposMode.VELOCITY))
            writeControlWord("B");
        else if (this.mode.equals(EposMode.CURRENT)){
            writeControlWord("2");
        } else throw new IllegalArgumentException(getName() + " has invalid Epos mode:"+this.mode);
    }



    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Turn OFF controller (in mode CURRENT, stop CURRENT) and shutdown.")
    @Override
    public String off()  {
        switch (this.mode) {
            case HOMING:
                this.writeControlWord("11F");
                break;
            case PROFILE_POSITION:
                this.writeControlWord("10F");
                break;
            case CURRENT:
                stopCurrent();
                break;
            default:
                assert false: mode;
        }
        shutdown();
        this.turnedOff = true;
        FCSLOG.info(getName() + " is TURNED OFF");
        return getName() + " OFF";
    }



    /*Methods available in CURRENT mode*/
    /**
     * In current mode this methods send a current to the motor. This make run
     * the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be
     * sent.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     *
     */
    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Set the current value in the EPOS CPU.")
    public void writeCurrent(int aValue)  {
        //TODO read controller CPU
        if (!mode.equals(EposMode.CURRENT)) {
            throw new RejectedCommandException(getName() + " is not in CURRENT mode.");
        }
        //TODO is it useful to test each time ? Response : YES because can be launched from the console.
        if (!isEnabled()) {
            throw new RejectedCommandException(getName() + " is not ENABLED.");
        }
        //check parameter aValue
        try {
            if (aValue > this.getMotorParameterValue("ContinuousCurrentLimit")) {
                throw new IllegalArgumentException(aValue + " is greater than Continuous Current Limit.");
            }
        } catch (EPOSConfigurationException ex) {
            FCSLOG.error(getName() + " ERROR in command writeCurrent:" + ex);
            throw new FcsHardwareException(getName() + ": ContinuousCurrentLimit is not defined.",ex);
        }
        String currentValueInHexa = Integer.toHexString(aValue);
        writeSDO("2030", "0", "2", currentValueInHexa);
        this.turnedOff = false;
        FCSLOG.debug(getName() + ": sent current to controller=" + aValue);
        //TODO check the return code of writeSDO
    }

    /**
     * In Current Mode this methods returns the current actualy received by the
     * motor.
     *
     * @return the current UNIT=mA / FORMAT=decimal
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias = "readCurrentActualValue",
            description = "In Current Mode this methods returns the current actualy received by the motor.")
    public int readCurrent()  {

        String currentInHexa = readSDO("6078", "0");
        int uns16 = Integer.parseInt(currentInHexa, 16);
        FCSLOG.debug(getName() + ":readCurrent="
                + FcsUtils.convertToInteger16(uns16) + " Unsigned value read=" + uns16);
        return FcsUtils.convertToInteger16(uns16);
    }

    /**
     * In Current Mode this methods returns the average of the current received
     * by the motor.
     *
     * @return the current UNIT=mA / FORMAT=decimal
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias = "readCurrentAverage",
            description = "In Current Mode this methods returns the average of the current  received by the motor.")
    public int readCurrentAverageValue()  {

        String currentInHexa = readSDO("2027", "0");
        int uns16 = Integer.parseInt(currentInHexa, 16);
        FCSLOG.debug(getName() + ":readAverageCurrent=" + FcsUtils.convertToInteger16(uns16) + " Unsigned value read=" + uns16);
        return FcsUtils.convertToInteger16(uns16);
    }

    /**
     * In current mode this methods set to zero the value of the current sent to
     * the motor. This stops the motor.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public void stopCurrent()  {
        if (mode.equals(EposMode.CURRENT)) {
            writeSDO("2030", "00", "2", "0");
        } else {
            throw new RejectedCommandException(getName() + "is not in CURRENT mode");
        }

    }




    /**
     * Reads in the CPU the value of the parameter PositionActualValue and
     * returns it in a decimal format.
     *
     * @return value in decimal format
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            alias = "readPositionActualValue",
            description = "In PROFILE_POSITION Mode this methods returns the actual position.")
    @Override
    public int readPosition()  {
        int positionActualValue = this.readParameter(Parameter.PositionActualValue);
        FCSLOG.debug(getName() + ":readPosition=" + positionActualValue);

        return positionActualValue;
    }

    /**
     * Read the position returned by the absolute encoder (single serial data).
     *
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the position returned by the absolute encoder (single serial data).")
    @Override
    public int readSSIPosition()  {
        return this.readParameter(Parameter.SSIEncoderActualPosition);
    }

    /**
     * In PROFILE_POSITION returns the value of the parameter ProfileVelocity.
     *
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION returns the value of the parameter ProfileVelocity.")
    @Override
    public int readProfileVelocity()  {
        int profileVelocity = this.readParameter(Parameter.ProfileVelocity);
        FCSLOG.debug(getName() + ":readProfileVelocity=" + profileVelocity);
        return profileVelocity;
    }

    /**
     * Writes value 35 in hexa to set the Homing method as Actual (See EPOS
     * documentation)
     *
     * @return
     * @throws RejectedCommandException if not in HOMING mode.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public String setHomingMethodActual()  {
        if (!mode.equals(EposMode.HOMING)) {
            throw new RejectedCommandException(getName() + "is not in HOMING mode");
        }
        return this.writeParameter(Parameter.HomingMethod, 35);

    }

    /**
     * Starts homing : (See EPOS documentation) For engineering mode.
     *
     * @throws SDORequestException
     * @throws RejectedCommandException
     */
    public void startHoming()  {
        if (!mode.equals(EposMode.HOMING)) {
            throw new RejectedCommandException(getName() + "is not in HOMING mode");
        }
        this.writeControlWord("1F");
    }

    /**
     * Set the Home Position with the value given as argument in decimal format.
     *
     * @param position in decimal format
     * @return
     * @throws SDORequestException
     * @throws RejectedCommandException
     */
    public String setHomePosition(int position)  {
        if (!mode.equals(EposMode.HOMING)) {
            throw new RejectedCommandException(getName() + "is not in HOMING mode");
        }

        return this.writeParameter(Parameter.HomePosition, position);

    }

    /**
     * Defines the actual position as the absolute position which value is given
     * as an argument.
     *
     * @throws SDORequestException
     * @throws RejectedCommandException
     */
    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Define the actual position as position given as argument.")
    public void defineAbsolutePosition(int position)  {
        
        FCSLOG.debug(getName() + " Defining Absolute Position:" + position);
        this.changeMode(EposMode.HOMING);
        this.writeParameters(EposMode.HOMING);
        if (!this.isEnabled()) {
            this.enable();
        }
        this.setHomePosition(position);
        this.setHomingMethodActual();
        this.writeControlWord("3F");
    }

    /**
     * In PROFILE_POSITION mode this methods set the target position. This make
     * run the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be
     * sent.
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION mode this methods set the target position."
            + " Target position has to be given in microns.")
    @Override
    public void writeTargetPosition(int aValue)  {
        if (!mode.equals(EposMode.PROFILE_POSITION)) {
            throw new RejectedCommandException(getName() + "is not in PROFILE_POSITION mode");
        }

        if (!isEnabled()) {
            throw new RejectedCommandException(getName() + "is not ENABLED");
        }

        String targetPositionInHexa = Integer.toHexString(aValue);
        writeSDO("607A", "0", "4", targetPositionInHexa);
        //TODO check the return code of writeSDO
    }

    /**
     * In HOMING mode and PROFILE_POSITION mode this indicates that the position
     * is reached.
     *
     * @return true if the position is reached.
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "In HOMING mode and PROFILE_POSITION mode this indicates that the position is reached.")
    @Override
    //TODO test this method on test benches
    public boolean checkTargetReached()  {
        String statusWordInHexa = readStatusWord();
        int statusWord = Integer.parseInt(statusWordInHexa, 16);
        int[] statusWordInBinary = FcsUtils.toReverseBinary(statusWord);
        return statusWordInBinary[10] == 1;
    }

    /**
     * Check if the Controller is in fault.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the Controller is in fault.")
    @Override
    public void checkFault()  {
        if (!this.booted) {
            throw new FcsHardwareException(getName() + " is not booted - can't checkFault.");
        }
        if (readNumberOfErrors() > 0) {
            this.inError = true;
            String errorInHexa = this.readErrorRegister();
            this.errorRegister = CanOpenErrorsTable.getErrorRegisterNameByCode(errorInHexa);
            this.errorHistory = this.readErrorHistoryNames();
            setChanged();
            this.notifyObservers(new ValueUpdate(getName(), "checkFault"));
            publishData();
            String errorHistoryString = displayErrorHistory();
            throw new FcsHardwareException(getName()
                    + " is in fault. " + errorHistoryString);
        } else {
            this.inError = false;
            this.errorRegister = "NO ERROR";
            this.errorHistory = new String[0];
            setChanged();
            this.notifyObservers(new ValueUpdate(getName(), "faultReset"));
            publishData();
        }

    }
    
    /**
     * Clear faults on the controller.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Clear faults on the controller.", alias="resetFault")
    public void faultReset()  {
        writeControlWord("80");
        this.inError = false;
        this.errorRegister = "NO ERROR";
        this.errorHistory = new String[0];
        setChanged();
        this.notifyObservers(new ValueUpdate(getName(), "faultReset"));
        publishData();
    }

    @Override
    public void checkParameters(EposMode aMode)   {
        Map<String, Integer> paramsMap = null;
        switch (aMode) {
            case HOMING:
                paramsMap = this.paramsForHoming;
                break;
                
            case PROFILE_POSITION:
                paramsMap = this.paramsForProfilePosition;
                break;

            case CURRENT:
                paramsMap = this.paramsForCurrent;
                break;
                
            default:
                assert false;
        }
        if (paramsMap == null) {
            throw new FcsHardwareException(getName()
                    + "parameters for mode :" + aMode.toString() + "are NULL in configuration system.");
        }
        if (paramsMap.isEmpty()) {
            throw new FcsHardwareException(getName()
                    + "parameters for mode :" + aMode.toString() + "are not defined in configuration system.");
        }
        checkParameters(paramsMap);
    }

    /**
     * For every parameter to be defined for this mode this method compares the
     * values in configuration with the value stored in controller CPU and
     * throws a HardwareException if these values are not equal. values stored
     * in the controller CPU.
     *
     * @param modeInString
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "For every parameter to be defined for this mode this method "
            + "compares the values in configuration with the value stored in controller CPU"
            + "and throws a HardwareException if these values are not equal.")
    @Override
    public void checkParameters(String modeInString)  {
        EposMode aMode = EposMode.valueOf(modeInString.trim().toUpperCase());
        checkParameters(aMode);
    }

    /**
     * Compare values of parameters in the map with the values of parameters in
     * the controller CPU.
     *
     * @param paramsMap
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    public void checkParameters(Map<String, Integer> paramsMap)  {
        boolean errorFound = false;
        StringBuilder sb = new StringBuilder(getName());
        for (Map.Entry<String, Integer> entry : paramsMap.entrySet()) {
            String paramName = entry.getKey();
            int configValue = entry.getValue();
            int controllerValue = readParameter(paramName);

            if (configValue != controllerValue) {
                String msg = " ERROR in CPU controller for parameter="
                        + paramName + " value found in configuration=" + configValue
                        + " value read in CPU=" + controllerValue + "\n";
                this.raiseWarning(getName(), msg);
                sb.append(msg);
                errorFound = true;
                //TODO test on testbench
                writeParameter(paramName,configValue);
            }

        }
        parametersOK = !errorFound;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Change mode to CURRENT mode and initialize the"
            + "CURRENT mode parameters from configuration values.")
    public void activateCurrentMode()  {
        this.changeMode(EposMode.CURRENT);
        this.writeParameters(this.paramsForCurrent);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Change mode to HOMING mode and initialize the"
            + "HOMING mode parameters from configuration values.")
    public void activateHomingMode()  {
        this.changeMode(EposMode.HOMING);
        this.writeParameters(this.paramsForHoming);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Change mode to PROFILE_POSITION mode and initialize the"
            + "PROFILE_POSITION mode parameters from configuration values.")
    public void activateProfilePositionMode()  {
        this.changeMode(EposMode.PROFILE_POSITION);
        this.writeParameters(this.paramsForProfilePosition);
    }

    /**
     * A holding brake can be associated to a Controller. This brake is
     * activated when the controller is powered off. In CPPM test bench a Brake
     * is associated with the controllers of the trucks. This command activates
     * the brake to prevent the trucks motion.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Put the brake on to prevent the trucks motion.")
    @Override
    public void activateBrake()  {
        String hexaValue = readParameterInHexa(Parameter.DigitalOutputFonctionnalityState);
        writeParameterInHexa(Parameter.DigitalOutputFonctionnalityState, FcsUtils.force2zero(hexaValue, 15));
        FCSLOG.debug(getName() + ": brake activated.");
    }

    /**
     * A holding brake can be associated to a Controller. In CPPM test bench a
     * Brake is associated with the controllers of the trucks. Release the brake
     * in order to be able to move the trucks.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Release the brake in order to be able to move the trucks.")
    @Override
    public void releaseBrake()  {
        //If the controller is not enabled and we release brakes it could lead to a 
        //lost of a filter.
        if (!isEnabled()) {
            enable(); //added in sept 2015 and tested with trucks and onlineClamp
        }
        String hexaValue = readParameterInHexa(Parameter.DigitalOutputFonctionnalityState);
        writeParameterInHexa(Parameter.DigitalOutputFonctionnalityState, FcsUtils.force2one(hexaValue, 15));
        FCSLOG.debug(getName() + ": brake released.");
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Activate the holding brake and shutdown the controller.")
    @Override
    public void disable()  {
        activateBrake();
        shutdown();
        enabledToPublish = false;
    }

    /**
     * Return the status object to be published on the STATUS bus.
     * @return 
     */
    @Override
    public StatusDataPublishedByEPOSController getStatusData() {
        return FcsUtils.createStatusDataPublishedByEPOSController(this);
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            alias = "refreshGUI",
            description = "Publish data for the controller on the status bus.")
    @Override
    public void publishData() {
        KeyValueData kvd = new KeyValueData(getName(), this.getStatusData());
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }
}
