/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import java.util.Map.Entry;
import java.util.Observable;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_ERROR;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCanOpenDevice;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 *
 * @author virieux
 */
public class CanOpenDevice extends Module implements PieceOfHardware, AlertRaiser {

    @ConfigurationParameter(isFinal = true, description="serial number of this CANOpen device")
    protected String serialNB;
    
    @ConfigurationParameter(isFinal = true, description="CANOpen node ID of this CANOpen device")
    protected String nodeID;

    protected CanOpenProxy tcpProxy;
    protected BridgeToHardware bridge;
    
    protected boolean booted;
    protected boolean initialized;
    
    protected boolean inError;
    protected String[] errorHistory;
    protected String errorRegister;

    /**
     * Create a new CanOpenDevice with a node ID and a serial number.
     * @param nodeID
     * @param serialNB 
     */
    public CanOpenDevice(String nodeID, String serialNB) {
        this.nodeID = nodeID;
        this.serialNB = serialNB;
        this.booted = false;
        this.initialized = false;
        inError = false;
        errorRegister = NO_ERROR;
        errorHistory = new String[0];
    }

    @Override
    public void setBooted(boolean booted) {
        this.booted = booted;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns this CANopen device's serial number.")
    @Override
    public String getSerialNB() {
        return this.serialNB;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns this CANopen device's node ID.")
    @Override
    public String getNodeID() {
        return this.nodeID;
    }

    /**
     * Returns true if this CANopen node is booted.
     * @return 
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if this CANopen node is booted.")
    public boolean isBooted() {
        return booted;
    }

    /**
     * Returns true if this CANopen node is initialized.
     * @return 
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if this CANopen node is initialized.")    
    public boolean isInitialized() {
        return this.initialized;
    }

    /**
     * Return tcpProxy
     * @return 
     */
    public CanOpenProxy getTcpProxy() {
        return tcpProxy;
    }
    
    
    
    /**
     * For the GUI
     *
     * @return a list of Errors - can be empty
     */
    public String[] getErrorHistory() {
        if (errorHistory == null) {
            return new String[0];
        } else {
            return errorHistory.clone();
        }
    }

    /**
     * For the GUI
     *
     * @return
     */
    public String getErrorRegister() {
        return errorRegister;
    }

    /**
     * For the GUI
     *
     * @return
     */
    public boolean isInError() {
        return inError;
    }

    @Override
    public void initModule() {
        /*
        doesn't work because the devices for the loader has another tcpProxy
        tcpProxy = (CanOpenProxy) this.getComponentByName("tcpProxy"); 
        */
        Entry<String, Object> entry = getComponentLookup().getParent(getName());
        bridge = (BridgeToHardware) entry.getValue();
        tcpProxy = (CanOpenProxy) bridge.getTcpProxy();
        this.initialized = false;
        
        //listens to tcpProxy to detect the emergency messages.
        this.listens((Observable) tcpProxy);
    }
    
    /**
     * What to do when the Modules we observe send there new values.
     * A CanOpenDevice listens to its tcpProxy to get the emergency message.
     *
     * @param source
     * @param v
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        FCSLOG.debug(getName() + ":processUpdate from source=" + source.toString() + " ValueUpdate=" + v.getName());
        
        if (source instanceof CanOpenProxy && v.getName().equals(this.tcpProxy.getName())) {
            /** observes only CanOpenProxy to get the emergency message which contains device name.*/
            EmergencyMessage emcyMsg = (EmergencyMessage) v.getValue();
            if (getName().equalsIgnoreCase(emcyMsg.getDeviceName())) {
            
                /** the emergency message is coming from this device.*/
                FCSLOG.error(getName() + ":EmergencyMessage received from CanOpenProxy=" + emcyMsg.toString());
                if ("00".equals(emcyMsg.getErrorRegisterCode())) {
                    //for the GUI
                    this.inError = false;
                    this.errorRegister = "NO ERROR";
                    this.errorHistory = new String[0];
                    publishData();

                } else {
                    this.inError = true;
                    try {
                        this.errorRegister = emcyMsg.getErrorRegisterName();
                        this.errorHistory = this.readErrorHistoryNames();
                        publishData();
                    } catch (FcsHardwareException ex) {
                        this.raiseAlarm("FCS001", "Error on FCS hardware : couldn't read EPOS error register for "
                                + name,ex);
                    }
                    
                }
                //Notify observers
                setChanged();
                this.notifyObservers(new ValueUpdate(name, emcyMsg));
            }
        }
    }

    /**
     * This methods has to be overridden if there is some initialization to do
     * for the device.
     *
     * @return a message
     */
    @Override
    public String initializeAndCheckHardware()  {
        if (this.booted) {
            this.initialized = true;
            return getName() + ": no hardware initialization for this device to be done.";
        } else {
            this.initialized = false;
            FCSLOG.error(getName()+" is not booted - can't initialize this device.");
            throw new FcsHardwareException(getName() + " is not booted.");
        }
    }

    /**
     * Sends a writeSDO request to the CANopen device.
     * All the arguments of this method take values in hexadecimal format.
     * @param index
     * @param subindex
     * @param size
     * @param newValue
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, 
            description = "Write on the Can Open Device with a command wsdo. All the values are in hexadecimal format."
                    + " To write new data value in decimal format, please use command writeSDO of tcpProxy.")
    public String writeSDO(String index, String subindex, String size, String newValue) {
        return tcpProxy.writeSDO(this.nodeID, index, subindex, size, newValue);
    }

    /**
     * Sends a readSDO request to the CANopen device. 
     * All the arguments of this method take values in hexadecimal format.
     * @param index
     * @param subindex
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "Reads the Can Open Device with a command rsdo.")
    public String readSDO(String index, String subindex)  {
        return tcpProxy.readSDO(this.nodeID, index, subindex);
    }

    /**
     * ***********************************************************************
     */
    /*Methods to read or display the errors that have occured on the device   */
    /*The errors are stored in 2 places :
     /* error register at index 1001 
     /* error history at index 1003               
     /* The error register and the error history 
     /**************************************************************************/
    /**
     * At index 1001 there deviceErrorFile an error register for the device. The
     * device maps internal errors in this byte.
     *
     * @return error code in Hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "Reads Error Register on the controller - in hexa (index 1001).")
    public String readErrorRegister() throws SDORequestException, 
            ShortResponseToSDORequestException, FcsHardwareException {
        String error = readSDO("1001", "0");
        return String.format("%02x", Integer.parseInt(error));
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "Display Error Register on the controller (index 1001).")
    public String displayErrorRegister() throws SDORequestException, 
            ShortResponseToSDORequestException,
            FcsHardwareException {
        String errorInHexa = readErrorRegister();
        String errorName = CanOpenErrorsTable.getErrorRegisterNameByCode(errorInHexa);
        FCSLOG.debug("error register=" + errorInHexa + " error name=" + errorName);
        return errorName;
    }

    /**
     * The error history holds errors that have occurred on the device and have
     * been signalled via the emergency object. This methods reads the Maxon
     * Motor Error History and returns an array of error code.
     *
     * @return errors : a list of device error code
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    public String[] readErrorHistory() throws SDORequestException, 
            ShortResponseToSDORequestException, FcsHardwareException {
        int numberOfErrors = readNumberOfErrors();
        String[] errors = new String[numberOfErrors];
        for (int i = 0; i < numberOfErrors; i++) {
            String subindex = Integer.toHexString(i + 1);
            errors[i] = readSDO("1003", subindex);
        }
        return errors;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Display Error Register on the controller (index 1001).")
    public String displayErrorHistory() throws SDORequestException, 
            ShortResponseToSDORequestException, FcsHardwareException {
        errorHistory = this.readErrorHistory();
        StringBuilder sb = new StringBuilder("Error history contains " + errorHistory.length + " errors.");
        if (!(errorHistory.length == 0)) {
            sb.append("\nList of errors in history,"
                    + "the newest is the first, the oldest the last :\n");
            for (int ix = 0; ix < errorHistory.length; ix++) {
                sb.append("Error code (in hexa)=");
                sb.append(errorHistory[ix]);
                sb.append("/ error name= ");
                sb.append(CanOpenErrorsTable.getDeviceErrorNameByCode(errorHistory[ix]));
                sb.append("\n");
            }
        }
        return sb.toString();
    }

    /**
     * Extract from the Maxon Motor firmware specification : Contains the number
     * of actual errors that are recorded in the array starting at subindex 1.
     * Writing a “0” deletes the error history (empties the array). Values
     * higher then “0” (zero) are not allowed to write."
     *
     * Read and returns the number of errors registred in the Error History
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read and returns the number of errors registred in the Error History.")
    public int readNumberOfErrors() throws SDORequestException, 
            ShortResponseToSDORequestException,
            FcsHardwareException {
        String errorNBinHEXA = readSDO("1003", "0");
        return Integer.parseInt(errorNBinHEXA, 16);
    }

    /**
     * For the GUI or to display at the console : return a list of String with
     * the error code and the error name.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    public String[] readErrorHistoryNames() throws FcsHardwareException {

        String[] errorHistoryList = this.readErrorHistory();
        String[] errorHistoryNames = new String[errorHistoryList.length];
        if (!(errorHistoryList.length == 0)) {
            for (int i = 0; i < errorHistoryList.length; i++) {

                String errorCode = errorHistoryList[i];
                FCSLOG.debug("errorCode=" + errorCode);
                String errorName = CanOpenErrorsTable.getDeviceErrorNameByCode(errorCode);
                FCSLOG.debug("errorName=" + errorName);
                errorHistoryNames[i] = errorCode + "=" + errorName;
            }
        }
        return errorHistoryNames;

    }

    /**
     * This method saves the parameters in the controller memory.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "This method saves the parameters in the controller memory.")
    public Object saveParameters()  {
        return this.writeSDO("1010", "1", "4", "65766173");
    }
    
    /**
     * Configure a node as a hearbeat producer. Every heartbeatTime
    milliseconds this node will produce a message like: 0x700+nodeID: 05 -
    1340268404.980000
     *
     * @param nodeID node ID in hexa
     * @param heartbeatTime FORMAT=hexa UNIT=milliseconds
     * @return error code returned by writeSDO
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Configure a CANopen device as a heartbeat producer.")
    public String configAsHeartbeatProducer(String nodeID, String heartbeatTime)  {
        return tcpProxy.writeSDO(nodeID, "1017", "0", "2", heartbeatTime);
    }

    /**
     * Command to be used by the end users. Configure a node as a heartbeat
     * producer. Parameters given in decimal format.
     *
     * @param nodeID FORMAT=decimal
     * @param heartbeatTime FORMAT=decimal UNIT=milliseconds
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public String configAsHeartbeatProducer(int nodeID, int heartbeatTime)  {
        if (heartbeatTime < 0 || heartbeatTime > FcsUtils.MAX_VALUE_2BYTES) {
            throw new IllegalArgumentException("heartbeat time is coded on 2 bytes" + " can't be > " 
                    + FcsUtils.MAX_VALUE_2BYTES);
        }
        return configAsHeartbeatProducer(Integer.toHexString(nodeID), Integer.toHexString(heartbeatTime));
    }

    /**
     * Configure a node as a hearbeat consumer. Every heartbeatTime
    milliseconds this node will produce a message like: 0x700+nodeID: 05 -
    1340268404.980000
     *
     * @param nodeID node ID in decimal
     * @param producerNodeID
     * @param heartbeatTime FORMAT=decimal UNIT=milliseconds
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public String configAsHeartbeatConsumer(int nodeID, int producerNodeID, int heartbeatTime)  {
        if (nodeID < 0 || nodeID > FcsUtils.MAX_NODE_ID) {
            throw new IllegalArgumentException("nodeID must be > 0 and <=" + FcsUtils.MAX_NODE_ID);
        }
        if (producerNodeID < 0 || producerNodeID > FcsUtils.MAX_NODE_ID) {
            throw new IllegalArgumentException("producerNodeID must be > 0 and <=" + FcsUtils.MAX_NODE_ID);
        }
        if (heartbeatTime < 0 || heartbeatTime > FcsUtils.MAX_VALUE_2BYTES) {
            throw new IllegalArgumentException("heartbeat time is coded on 2 bytes" + " can't be > " 
                    + FcsUtils.MAX_VALUE_2BYTES);
        }
        String value = String.format("%04x", producerNodeID) + String.format("%04x", heartbeatTime);
        return tcpProxy.writeSDO(Integer.toHexString(nodeID), "1016", "1", "4", value);
    }

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Print CANopen device.")
    public String toString() {
        StringBuilder sb;
        if (getName() == null) {
            sb = new StringBuilder();
        } else {
            sb = new StringBuilder(getName());
        }
        sb.append("/SerialNumber=");
        sb.append(this.serialNB);
        sb.append("/NodeID_in_hexa=");
        sb.append(this.nodeID);
        int x = Integer.parseInt(this.nodeID, 16);
        sb.append("/NodeID_in_decimal=");
        sb.append(x);
        sb.append("/");
        // because the configuration system can call toString before the object is completely built
        // for tracing purpose.
        if (tcpProxy == null) {
            return sb.toString();
        }
        if (isBooted()) {
            sb.append("BOOTED/");

        } else {
            sb.append("NOT YET BOOTED/");
        }
        return sb.toString();
    }
    
    /**
     * Creates an object to be published on the STATUS bus by a CanOpenDevice.
     * @return 
     */
    public StatusDataPublishedByCanOpenDevice createStatusDataPublishedByCanOpenDevice() {
        StatusDataPublishedByCanOpenDevice status = new StatusDataPublishedByCanOpenDevice(getName(),
                booted, initialized);
        status.setInError(inError);
        status.setErrorRegister(errorRegister);
        status.setErrorHistory(errorHistory);
        return status;
    }
    
    public StatusDataPublishedByCanOpenDevice getStatusData() {
        return createStatusDataPublishedByCanOpenDevice();
    }
    
    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            alias = "refreshGUI",
            description = "Publish data for the device on the status bus.")
    public void publishData() {
        KeyValueData kvd = new KeyValueData(getName(), getStatusData());
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }


}
