/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import java.nio.file.FileSystemNotFoundException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.*;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This class gathers the description of the real hardware and the tools
 * (methods) to give access to this hardware. 
 * 
 *
 * @author virieux
 */
public class BridgeToCanOpenHardware extends Module implements BridgeToHardware {


    protected CanOpenProxy tcpProxy;
    
    private final ArrayList<PieceOfHardware> devices = new ArrayList<>();

    protected boolean configurationChecked = false;
    protected boolean hardwareInitialized = false;

    protected final Map<String, PieceOfHardware> hardwareMapByName = new HashMap<>();

    public BridgeToCanOpenHardware(CanOpenProxy tcpProxy) {
        this.tcpProxy = tcpProxy;
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "print my hardware list; for tests")
    public String printHardwareList() {
        return hardwareMapByName.toString();
    }

    @Override
    public CanOpenProxy getTcpProxy() {
        return tcpProxy;
    }

    @Override
    public void startServer() {
        tcpProxy.startServer();
    }

    @Override
    public void startThreadReader() {
        tcpProxy.startThreadReader();
    }

    /**
     * This method returns true if the C-Wrapper is connected to the tcp proxy.
     * The C-Wrapper has to send the tcpProxy clientName as a keyword before any
     * communication can begin. So if the C-Wrapper doesn't send the good
     * keyword or if it doesn't connect to the tcp proxy, this method returns
     * false.
     *
     * @return
     */
    @Override
    public boolean isCWrapperConnected() {
        return tcpProxy.isReady(tcpProxy.getMyClientName());
    }

    /**
     * returns true if the hardware is booted, identified and initialized.
     *
     * @return
     */
    @Override
    public boolean isCANDevicesReady() {
        return tcpProxy.isCANDevicesReady();
    }

    /**
     * This methods has to be overridden in a simulation context.
     * @return 
     */
    @Override
    public boolean isRealHardware() {
        return true;
    }

    @Override
    public void initModule() {
        ComponentLookup lookup = getComponentLookup();
        // Adding PieceOfHardwares
        hardwareMapByName.putAll(lookup.getChildren(getName(), PieceOfHardware.class));
        devices.addAll(hardwareMapByName.values());

        FCSLOG.info(getName() + ": init MODULE BRIDGE TO HARDWARE.");
        tcpProxy.setHardwareList(devices);
        FCSLOG.info(getName() + ":NUMBER OF CAN OPEN DEVICES EXPECTED =" + devices.size());
        FCSLOG.info(this.toString());

        //we have to read the errors table for the maxon motor and can open devices.
        try {
            CanOpenErrorsTable.loadDeviceErrorTable();
            CanOpenErrorsTable.loadErrorRegisterTable();
            CanOpenErrorsTable.loadCommunicationErrorTable();
        } catch (DeploymentException ex) {
            String msg = ex + ": Could not load CanOpen Error Tables";
            FCSLOG.error(msg);
            throw new FileSystemNotFoundException(msg);
        }
    }


    @Override
    public PDOStorage readPDOs()  {
        FCSLOG.debug(getName() + "/readPDOS");
        return this.tcpProxy.readPDOs();

    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(String.valueOf(this.getName()));
        sb.append(",tcpProxy:");
        sb.append(String.valueOf(tcpProxy.toString()));
        sb.append(",devices:");
        sb.append(hardwareMapByName);
        return sb.toString();
    }
}
