
package org.lsst.ccs.subsystems.fcs.common;

import java.util.Observable;
import org.lsst.ccs.UsesSubsystem;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This interface is to be implemented by every class which is a model for a hardware 
 * that is moved by an EPOSController.
 * The goal of this interface is to factorise methods to process emergency messages
 * coming from the CANbus either when a controller is in fault or when a command faultReset 
 * has been sent to the controller.
 * 
 * 
 * @author virieux
 */
public interface MovedByEPOSController extends UsesSubsystem {
    
    /**
     * Return the name of the controller.
     * @return 
     */
    String getControllerName();
    
    /**
     * Return true if the controller is in fault (an EmergencyMessage has been previously received.)
     * @return 
     */
    boolean isControllerInFault();
    
    /**
     * 
     * @param controllerInFault 
     */
    void setControllerInFault(boolean controllerInFault);
    

    
    /**
     * 
     */
    void publishData();
    
    /**
     * What to do when the Modules we observe send there new values. This class
     * observes its controller to publish data when its controller is in fault
     * or after a faultReset. Needed to update the GUI.
     * 
     * The class which implements this interface must listens to its controller.
     * (Done in initModule)
     *
     * @param source
     * @param v
     */
    default void processUpdate(Observable source, Module.ValueUpdate v) {
        FCSLOG.debug(getName() + ":"
                + "=" + source.toString()
                + " ValueUpdate=" + v.getName());
        if ((source instanceof EPOSController) && v.getName().equals(this.getControllerName())) {
            if (v.getValue() instanceof EmergencyMessage) {
                EmergencyMessage emcyMsg = (EmergencyMessage) v.getValue();
                FCSLOG.debug(getName() + ":EmergencyMessage received from CanOpenProxy=" + emcyMsg.toString());
                processEmergencyMessage(emcyMsg);

            } else if (v.getValue() instanceof String) { 
                if ("faultReset".equals((String) v.getValue())) {
                    processFaultReset();
                } else if ("checkFault".equals((String) v.getValue()))
                    this.setControllerInFault(true);
                }
            }
        this.publishData();
    }
    
    /**
     * This method is to be used when receiving an EmergencyMessage from the CANbus.
     * 
     * An emergency message for a controller can be received from the CANbus in 2 cases:
     * - a faultReset has been previously sent to the controller, in this case the error code
     * of the emergency message is "00",
     * - or when a new error is detected by the controller device, in this case the error code 
     * is different from "00".
     * 
     * This method updates the boolean controllerInFault and publishes data.
     * It's useful for the GUI and the monitoring of the hardware.
     * 
     * @param emcyMsg the emergency message to be processed.
     */
    default void processEmergencyMessage(EmergencyMessage emcyMsg) {
        if (getControllerName().equals(emcyMsg.getDeviceName())) {
            
            String errCode = emcyMsg.getDeviceErrorCode();
            switch (errCode) {
                case "00":
                    FCSLOG.debug(getName() + ":faultReset ?=" + emcyMsg.toString());
                    setControllerInFault(false);
                    break;

                default:
                    FCSLOG.debug(getName() + ":EmergencyMessage received for "
                            + "my controller from CanOpenProxy=" + emcyMsg.toString());
                    setControllerInFault(true);
            }
            this.publishData();
        }
    }
    
    /**
     * What to do when a faultReset has been done on the controller.
     */
    default void processFaultReset() {
        this.setControllerInFault(false);
    }
    
}
