/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.common;

import java.util.Map;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;

/**
 *
 * @author virieux
 */
public interface EPOSController extends PieceOfHardware {
    
    /**
     * Return true if the controller is turned OFF;
     * @return 
     */
    boolean isTurnedOff();
    
    /**
     * Turn off controller.
     * @return
     * @throws FcsHardwareException 
     */
    String off();

    /**
     * Return the max value of current which can be sent to this controller.
     * @return
     * @throws FcsHardwareException 
     */
    int getMaxCurrent();

    /**
     * Return the min value of current which can be sent to this controller.
     * @return
     * @throws FcsHardwareException 
     */
    int getMinCurrent();

    /**
     * Return the min value of speed for the motor controlled by this controller.
     * @return
     * @throws FcsHardwareException 
     */
    int getMaxSpeed();

    /**
     * Return the max position for the object moved by motor which is controlled by this controller.
     * (example of objects moved by this controller: loader trucks or autochanger trucks)
     * @return
     * @throws FcsHardwareException 
     */
    int getMaxPosition();

    /**
     * Return the min position for the object moved by motor which is controlled by this controller.
     * (example of objects moved by this controller: loader trucks or autochanger trucks)
     * @return
     * @throws FcsHardwareException 
     */
    int getMinPosition();
    
    /**
     * This methods returns true if the values of parameters stored in CPU are
     * the same than those stored in the configuration system.
     * @return 
     */
    boolean isParametersOK();
    

    /**
     * This methods changes the mode to the new mode given as an argument. It
     * writes the new mode in the CPU of the EPOS controller and updates the
     * field mode.
     *
     * @param newMode
     * @return
     * @throws SDORequestException
     */
    String changeMode(EposMode newMode);


    /**
     * Defines the actual position as the absolute position which value is given
     * as an argument.
     *
     * @param position
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Defines the actual position as the absolute position "
                    + "which value is given as an argument.")
    void defineAbsolutePosition(int position);

    /**
     * Return the number of errors that occured on this controller.
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    int readNumberOfErrors();

    /**
     * Return a printed list of errors that occured on this controller.
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    String displayErrorHistory();

    /**
     * Check if the controller is in fault.
     * Throw an Exception if the controller is in fault.
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    void checkFault();

    /**
     * This methods writes in the CPU of the EPOS devis the values of the
     * parameters set for the mode. Usually the values of the parameters are
     * given by the configuration system.
     *
     * @param mode
     * @throws SDORequestException
     */
    void writeParameters(EposMode mode);

    /**
     * In PROFILE_POSITION mode this methods set the target position. This make
     * run the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be
     * sent.
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION mode this methods set the target position.")
    void writeTargetPosition(int aValue);

    /**
     * Write to the CPU of the EPOS a map of parameters.
     *
     * @param paramMap
     * @throws SDORequestException
     */
    void writeParameters(Map<String, Integer> paramMap);

    /*Methods available in CURRENT mode*/
    /**
     * In current mode this methods send a current to the motor. This make run
     * the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be
     * sent.
     * @throws org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In current mode this methods send a current to the motor.")
    void writeCurrent(int aValue) throws EPOSConfigurationException;

    /**
     * Return true if the target position has been reached.
     * @param position
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if the target position has been reached.")
    boolean isTargetPositionReached(int position);

    /**
     * Read actual position on the controller CPU.
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    int readPosition();

    /**
     * Read the SSS position on the controller CPU.
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    int readSSIPosition();

    /**
     * Read actual current on the controller CPU.
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    int readCurrent();

    /**
     * Read all the value of parameters for a given EPOS mode and return a printed list of this values.
     * @param mode
     * @return
     * @throws FcsHardwareException 
     */
    String readParameters(EposMode mode);

    /**
     * Read ProfileVelocity on the controller CPU.
     * @return
     * @throws FcsHardwareException 
     */
    int readProfileVelocity();

    /**
     * enable the controller. (to be done before each action on the motor.)
     * @return
     * @throws FcsHardwareException 
     */
    String enable();

    /**
     * Return true if the controller is enabled and so ready to receive commands.
     * @return
     * @throws FcsHardwareException 
     */
    boolean isEnabled();

    /**
     * read the EPOS mode on the controller CPU.
     * @return
     * @throws FcsHardwareException 
     */
    EposMode readMode();

    /**
     * Send a quickStop command to the controller.
     * @throws FcsHardwareException 
     */
    void quickStop();

    /**
     * For every parameter to be defined for this mode this method compares the
     * values in configuration with the value stored in controller CPU and
     * throws a HardwareException if these values are not equal. 
     *
     * @param modeInString
     * @throws HardwareException
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    void checkParameters(String modeInString) throws HardwareException;

    /**
     * Check parameters for a given EPOS mode : compare the values stored in the Configuration System and
     * the values stored in controller CPU.
     * Raise a FcsHardwareException if some values are not equals.
     * @param aMode
     * @throws HardwareException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException 
     */
    void checkParameters(EposMode aMode) throws HardwareException;

    /**
     * Write controlWord to the controller.
     * Use to send or start commands. See EPOS documentation.
     * @param f
     * @throws FcsHardwareException 
     */
    void writeControlWord(String f);

    /**
     * Shutdown the controller.
     * @return
     * @throws FcsHardwareException 
     */
    String shutdown();

    /**
     * Throw an Exception if the target was not reached in the previous action.
     * @return
     * @throws FcsHardwareException 
     */
    boolean checkTargetReached();

    /**
     * Activate brake for controller who manages a brake.
     * Use for autochanger ONLINE clamps.
     * @throws FcsHardwareException 
     */
    void activateBrake();

    /**
     * Release brake for controller who manages a brake.
     * Use for autochanger ONLINE clamps.
     * @throws FcsHardwareException 
     */
    void releaseBrake();

    /**
     * SwitchOn and enable controller.
     * @throws FcsHardwareException 
     */
    void switchOnEnableOperation();

    /**
     * Return the list of parameters in Current mode that are in Configuration System for this controller.
     * @return 
     */
    Map<String, Integer> getParamsForCurrent();

    /**
     * Return the list of parameters in Homing mode that are in Configuration System for this controller.
     * 
     * @return 
     */
    Map<String, Integer> getParamsForHoming();

    /**
     * Return the list of parameters in Homing mode that are in Configuration System for this controller.
     * @return 
     */
    Map<String, Integer> getParamsForProfilePosition();
    
    /**
     * Return the EPOS mode stored in field mode. Doesn't read again controller CPU.
     * @return 
     */
    EposMode getMode();

    /**
     * For GUI. Doesn't read again controller CPU.
     * Return true if controller is enabled.
     * @return 
     */
    boolean isEnabledToPublish();

    /**
     * Return true if controller is enabled.
     * @return 
     */
    boolean isInError();

    /**
     * 
     * @return error register in printed format
     */
    String getErrorRegister();

    /**
     * 
     * @return a printed list of errors.
     */
    String[] getErrorHistory();

    /**
     * Write on the CPU of the controller a value given in hexadecimal format for a parameter.
     * @param parameter
     * @param string HEXADECIMAL format of the new value.
     * @return
     * @throws FcsHardwareException 
     */
    String writeParameterInHexa(EPOSEnumerations.Parameter parameter, String string);

    /**
     * Disable controller.
     * @throws FcsHardwareException 
     */
    void disable();
}
