/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This is the model for the loader in the Filter Exchange System. 
 * The loader is used to load a filter into the camera or to unload a filter 
 * from the camera.
 *
 * @author virieux
 */
public class LoaderModule extends Module implements HardwareController, FilterHolder {

    @ConfigurationParameter(isFinal = true)
    private final String plutoGatewayName;
    
    private final LoaderCarrierModule carrier;
    private final LoaderClampModule clamp;
    private BridgeToHardware bridgeToLoader;
    private MainModule main;
    
    private FilterHolder autochanger;

    private final NumericSensor filterPresenceSensor0;
    private final NumericSensor filterPresenceSensor1;
    private final NumericSensor loaderOnCameraSensor0;
    private final NumericSensor loaderOnCameraSensor1;


    private PlutoGatewayInterface plutoGateway;

    
    private boolean connectedOnCamera = false;
    private boolean empty = false;
    private boolean filterPresenceSensorsInError = false;
    private boolean loaderOnCameraSensorsInError = false;

    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

    /**
     * Build a LoaderModule with a carrier, a clamp and sensors.
     * @param carrier
     * @param clamp
     * @param plutoGatewayName
     * @param filterPresenceSensor0
     * @param filterPresenceSensor1
     * @param loaderOnCameraSensor0
     * @param loaderOnCameraSensor1 
     */
    public LoaderModule(
            LoaderCarrierModule carrier,
            LoaderClampModule clamp,
            String plutoGatewayName,
            NumericSensor filterPresenceSensor0,
            NumericSensor filterPresenceSensor1,
            NumericSensor loaderOnCameraSensor0,
            NumericSensor loaderOnCameraSensor1) {
        this.carrier = carrier;
        this.clamp = clamp;
        this.plutoGatewayName = plutoGatewayName;
        this.filterPresenceSensor0 = filterPresenceSensor0;
        this.filterPresenceSensor1 = filterPresenceSensor1;
        this.loaderOnCameraSensor0 = loaderOnCameraSensor0;
        this.loaderOnCameraSensor1 = loaderOnCameraSensor1;
    }

    private Boolean isHaltRequired() {
        return main.getHaltRequired().get();
    }

    private Boolean isStopRequired() {
        return main.getStopRequired().get();
    }

    public NumericSensor getFilterPresenceSensor0() {
        return filterPresenceSensor0;
    }

    public NumericSensor getFilterPresenceSensor1() {
        return filterPresenceSensor1;
    }

    public NumericSensor getLoaderOnCameraSensor0() {
        return loaderOnCameraSensor0;
    }

    public NumericSensor getLoaderOnCameraSensor1() {
        return loaderOnCameraSensor1;
    }


    public boolean isFilterPresenceSensorsInError() {
        return filterPresenceSensorsInError;
    }

    public boolean isLoaderOnCameraSensorsInError() {
        return loaderOnCameraSensorsInError;
    }
    
    

    /**
     * To publish data we don't want to read again the sensors.
     *
     * @return
     */
    public boolean isEmptyToPublish() {
        return empty;
    }

    /**
     * To publish data we don't want to read again the sensors.
     *
     * @return
     */
    public boolean isOnCameraToPublish() {
        return connectedOnCamera;
    }



    //for the Loader GUI
    public EPOSController getHooksController() {
        return (EPOSController) getComponentLookup().getComponentByName("hooksController");
    }

    //for the Loader GUI
    public EPOSController getCarrierController() {
        return (EPOSController) getComponentLookup().getComponentByName("carrierController");
    }

    @Override
    public void initModule() {
        ComponentLookup lookup = getComponentLookup();
        this.plutoGateway = (PlutoGatewayInterface) lookup.getComponentByName(plutoGatewayName);
        this.bridgeToLoader = (BridgeToHardware) lookup.getComponentByName("bridgeToLoader");
        this.main = (MainModule) lookup.getComponentByName("main");
        
        if (lookup.getComponentByName("autochanger") instanceof FilterHolder) {
            autochanger = (FilterHolder) lookup.getComponentByName("autochanger");
        } else {
            final String MSG = getName() + " ==> autochanger doesn't implements FilterHolder -"
                    + " Please fix groovy description file or LoaderModule implementation.";
            FCSLOG.error(MSG);
            throw new IllegalArgumentException(MSG);
        }
    }

    /**
     * Return true if loader clamp and carrier hardware is ready to be controlled.
     * That means that controllers and plutoGateway are booted, identified and initialized
     * and homing of the controllers have been done.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if loader clamp and carrier hardware is ready to be controlled."
                + "That means that controllers and plutoGateway are booted, identified and initialized "
                + "and homing of the controllers have been done.")
    public boolean isHardwareReady() {
        return bridgeToLoader.isCANDevicesReady() && isInitialized();
    }
    
    /**
     * Return true if CAN open devices (controllers and plutoGateway) are booted, identified  and initialized.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description ="Return true if CAN open devices (controllers and plutoGateway) are booted, identified  "
                    + "and initialized.")
    public boolean isCANDevicesReady() {
        return bridgeToLoader.isCANDevicesReady();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if hardware (controllers and plutoGateway) is correctly initialized"
                + "and homing of the controllers is done.")
    public boolean isInitialized() {
        return isCANDevicesInitialized()
                && clamp.isInitialized()
                && carrier.isInitialized();
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if hardware (controllers and plutoGateway) is correctly initialized.")
    public boolean isCANDevicesInitialized() {
        return plutoGateway.isInitialized()
                && ((EPOSController) getHooksController()).isInitialized()
                && ((EPOSController) getCarrierController()).isInitialized();
    }    
    
    

    /**
     * Returns the boolean field empty. If the empty boolean is being updated
     * and waits for a response from a sensor, this methods waits until empty is
     * updated. If the field empty is not being updated, it returns immediatly
     * the field empty.
     *
     * @return empty
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if there is no filter in the loader. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {

        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(getName() + ": has been interrupted while waiting for end of update.",ex);
                }

            }
            return empty;

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * atHandoff is updated. If the field atHandoff is not being updated, it
     * returns immediatly the field atHandoff.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the loader is connected on the camera. This command doesn't read again the sensors.")
    public boolean isConnectedOnCamera() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(getName() + ": has been interrupted while waiting for end of update.",ex);
                }

            }
            return connectedOnCamera;

        } finally {
            lock.unlock();
        }
    }

    /**
     * Return true if the autochanger is holding the filter.
     * This command doesn't read again the sensors.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the autochanger is holding the filter. This command doesn't read again the sensors.")
    public boolean isAutochangerHoldingFilter()  {       
        return autochanger.isHoldingFilter();
    }

    /**
     * Return true if a filter is present and it is held by the loader clamp.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if a filter is present and it is held by the loader clamp.")
    @Override
    public boolean isHoldingFilter()   {
        this.updateStateWithSensors();
        this.clamp.updatePosition();
        return this.clamp.isAtClampedPosition() && !this.isEmpty();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the clamp or the carrier are in action.")
    public boolean isMoving() {
        return this.clamp.isMoving() || this.carrier.isMoving();
    }

    public LoaderClampModule getClamp() {
        return this.clamp;
    }

    public LoaderCarrierModule getCarrier() {
        return carrier;
    }

    public boolean isCarrierAtStoragePosition() {
        return carrier.isAtStoragePosition();
    }

    /**
     * This command has to be executed after the initialization phase and after
     * the checkHardware command.
     * It can't be automaticaly executed during initialization phase because it's not compliant with the
     * CCS requirement which says that the initialization of the subsystem can't make move hardware.
     * @throws HardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Attention ! this commands does the homing of the controllers and might move hardware. "
                    + "Initialize loader hardware.",
            alias="homing")
    public void initializeHardware() throws HardwareException {
        checkHardware();
        try {
            clamp.initializeHardware();
        } catch (FailedCommandException | FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }
        /* If the carrier is empty and the clamp is LOCKED, we can't start the loader subsystem.*/
        /* The test clamp.isLocked can be done only after the clamp has been initialized.*/
        if (carrier.isEmpty() && clamp.isLocked()) {
            String msg = getName() + ": carrier is empty and clamp is LOCKED - can't start.";
            FCSLOG.error(msg);
            throw new HardwareException(true, msg);
        }
        this.updateFCSState();
    }

    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {

        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }

        clamp.checkHardware();
        carrier.checkHardware();
        /*NOT TO BE DONE here : initialization of the hardware because we don't want 
        to move hardware during initialization phase.*/
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * CheckStarted is executed by completeInitialization when a first start has
     * failed. Because end user could have changed many things, we have to check
     * again that all is correct.
     *
     * @throws org.lsst.ccs.HardwareException
     *
     */
    @Override
    public void checkStarted() throws HardwareException {

        FCSLOG.info(getName() + " BEGIN checkStarted");

        //check that all hardware is booted and identified.
        bridgeToLoader.getTcpProxy().checkNewHardware();

        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }
        clamp.checkHardware();
        carrier.checkHardware();
        
        FCSLOG.info(getName() + " END checkStarted");
    }

    /**
     * This method is executed during a shutdown of the subystem to prevent a
     * shutdown when some hardware is moving. Nothing to do here because all the
     * moving hardware of the loader is represented by the class MobilItemModule
     * where the method checkStopped is implemented.
     *
     * @throws org.lsst.ccs.HardwareException
     *
     */
    @Override
    public void checkStopped() throws HardwareException {
        //done already in my children which are MobilItemModule and my other children don't need that.
    }

    @Override
    public void tick() {
        //Don't execute here methods which take a lock.
        this.publishData();
        this.carrier.publishData();
        this.clamp.publishData();
    }

    /**
     * Check if the loader carrier can move.
     * @return
     * @throws FcsHardwareException
     * @throws FailedCommandException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the carrier can move.")
    public boolean checkConditionsForCarrierMotion()  {
        FCSLOG.info(getName() + " checking pre-conditions for carrier motion");

        updateStateWithSensors();

        if (!clamp.isInitialized()) {
            String msg = getName() + ": clamp is not intialized. PLease initialize hardware first.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }

        if (clamp.isInError()) {
            throw new RejectedCommandException(getName()
                    + " carrier can't move because clamp is in ERROR.");
        }

        if (carrier.isEmpty()) {
            return true;
        }

        //here a filter is in the loader       
        if (!this.isConnectedOnCamera()) {
            return true;
        }

        if (this.isHoldingFilter() && isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + " carrier can't move because a filter is in the loader"
                    + " and it's held by loader AND autochanger.");

        } else if (!this.isHoldingFilter() && !isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + " carrier can't move because the filter in the loader is not held"
                    + " neither by loader neither by autochanger. "
                    + "Close loader clamp or autochanger latches");

        } else {
            return true;
        }

    }

    /**
     * Check if the clamp can be opened.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the clamp can be opened.")
    public boolean checkConditionsForOpeningHooks()  {
        
        FCSLOG.info(getName() + " checking pre-conditions for opening hooks");

        updateStateWithSensors();

        if (!isConnectedOnCamera()) {
            throw new RejectedCommandException(getName()
                    + ": Loader not connected - can't open clamp.");
        }

        if (!clamp.isInitialized()) {
            String msg = getName() + ": clamp is not intialized. PLease initialize hardware first.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }

        if (clamp.isInError()) {
            String msg = getName() + ": clamp is in ERROR state - can't open clamp.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }

        if (carrier.isEmpty()) {
            return true;
        }

        if (!carrier.isAtHandoffPosition()) {
            String msg = getName() + ": carrier is loaded with a filter but is not "
                    + "at handoff position - can't open clamp.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }

        if (!isAutochangerHoldingFilter()) {
            String msg = getName() + ": A filter is in the loader but autochanger and "
                    + "not holding it - can't open clamp.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        } else {
            return true;
        }

    }

    /**
     * Check if the clamp can be closed.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the clamp can be closed.")
    public boolean checkPreConditionsForClosingHooks()  {
        FCSLOG.info(getName() + " checking pre-conditions for closing hooks");
        updateStateWithSensors();
        this.clamp.updatePosition();

        if (!isConnectedOnCamera()) {
            throw new RejectedCommandException(getName()
                    + ": Loader not connected - can't execute close command.");
        }

        if (!clamp.isInitialized()) {
            String msg = getName() + ": clamp is not intialized. PLease initialize hardware first.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }

        if (clamp.isInError()) {
            String msg = getName() + ": clamp is in ERROR state - can't execute command open.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }

        if (carrier.isEmpty()) {
            String msg = getName() + ": no filter in carrier - can't execute close command for Module "
                    + clamp.getName();
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);

        }
        //at this point carrier is NOT empty
        if (this.clamp.isAtClampedPosition()) {
            String msg = getName() + ": a filter is already clamped - can't execute close or clamp command for Module "
                    + clamp.getName();
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);

        }

        return true;
    }

    /**
     * This method closes the clamps. When this action is completed the filter
     * is clamped softly on the loader and can't fall.
     *
     * @return a message for the end user.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Close hooks.")
    public String closeHooks()  {
        return clamp.close();
    }

    /**
     * This method closes the clamps strongly. When this action is completed,
     * the filter is clamped softly on the loader and is tight fermely on the
     * carrier.
     *
     * @return a message for the end user.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Clamp hooks.")
    public String clampHooks()  {
        return clamp.clamp();
    }

    /**
     * This command opens the clamp.
     *
     * @return a message for the end user.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(level = Command.ENGINEERING1, type = Command.CommandType.ACTION,
            description = "Open the hooks")
    public String openHooks()  {
        return clamp.open();
    }

    /**
     * Moves the loader carrier to the handoff position.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the loader carrier to HANDOFF position")
    public String goToHandoff()  {
        //TODO check the preconditions
        return carrier.goToHandOff();
    }

    /**
     * Moves the loader carrier to the storage position.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the loader carrier to STORAGE position")
    public String goToStorage()  {
        //TODO check the preconditions
        return carrier.goToStorage();
    }

    /**
     * This methods updates the carrier and clamp state in reading all the
     * sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors()  {
        
        this.bridgeToLoader.getTcpProxy().publishData();

        if (!plutoGateway.isBooted()) {
            throw new FcsHardwareException(getName() + ": plutoGateway not booted - can't read sensors");
        }

        if (!plutoGateway.isInitialized()) {
            throw new FcsHardwareException(getName() + ": plutoGateway not initialized - can't read sensors");
        }

        lock.lock();
        try {
            updatingState = true;
            this.plutoGateway.updateValues();
            String[] readHexaValues = this.plutoGateway.getHexaValues();
            FCSLOG.finest("readHexaValues[0]=" + readHexaValues[0]);
            FCSLOG.finest("readHexaValues[1]=" + readHexaValues[1]);
            this.filterPresenceSensor0.updateValue(readHexaValues);
            this.filterPresenceSensor1.updateValue(readHexaValues);
            this.loaderOnCameraSensor0.updateValue(readHexaValues);
            this.loaderOnCameraSensor1.updateValue(readHexaValues);
            this.clamp.updateStateWithSensors(readHexaValues);
            this.carrier.updateStateWithSensors(readHexaValues);

            this.empty = filterPresenceSensor0.digitalValue == 0
                    && filterPresenceSensor1.digitalValue == 0;
            this.connectedOnCamera = loaderOnCameraSensor0.getDigitalValue() == 1
                    && loaderOnCameraSensor1.getDigitalValue() == 1;

            this.filterPresenceSensorsInError = !(filterPresenceSensor0.getDigitalValue()
                    == filterPresenceSensor1.getDigitalValue());
            this.loaderOnCameraSensorsInError = !(loaderOnCameraSensor0.getDigitalValue()
                    == loaderOnCameraSensor1.getDigitalValue());

        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
            this.clamp.publishData();
            this.carrier.publishData();
        }
    }
    
    /**
     * Check that Loader hardware is ready to be operated and moved.
     * This means that :
     * - all CAN open devices are booted, identified and initialized,
     * - homing has been done on the controllers.
     * This updates the FCS state and FCS readyness state and publishes on the status bus.
     * 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update FCS state and FCS readyness state and publishes on the status bus.")
    public void updateFCSState() {
        if (clamp.isInitialized() && carrier.isInitialized()) {
            main.updateFCSState();

        } 
    }

    /**
     * This command can be launched when a filter is in the loader and we want
     * to put in inside the camera.
     *
     * @return
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Load a filter from the loader into the camera.")
    public String loadFilterInCamera()  {
        
        if (!isConnectedOnCamera()) {
            throw new RejectedCommandException(getName()
                    + ":loader has to be connected to camera first.");
        }
        if (!isHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + ":loader is not holding a filter : can't load a filter into camera.");
        }
        if (!isCarrierAtStoragePosition()) //TODO : what do we do in this case ?
        {
            throw new RejectedCommandException(getName()
                    + ":carrier loader is not at storage position : can't load a filter into camera.");
        }

        //go to Handoff position with the filter
        carrier.goToHandOff();
        if (isHaltRequired() || isStopRequired()) {
            return getName() + ": received HALT or STOP command.";
        }
        //wait until the autochanger hold the filter
        while (!isAutochangerHoldingFilter()) {
            FCSLOG.debug(getName() + " waiting until autochanger holds the filter...");
            try {
                Thread.sleep(300);
            } catch (InterruptedException ex) {
                throw new FcsHardwareException("loadFilterInCamera interrupted while waiting for the autochanger"
                        + " to take the filter",ex);
            }
            updateStateWithSensors();
        }

        if (isAutochangerHoldingFilter()) {
            openHooks();
        }
        if (isHaltRequired() || isStopRequired()) {
            return getName() + ": received HALT or STOP command.";
        }

        // empty carrier go to Storage
        return goToStorage();
    }

    /**
     * This command can be launched when the loader is empty and we want to take
     * a filter from the camera. A filter must be at handoff position and held
     * by autochanger.
     *
     * @return
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     * @throws SDORequestException
     * @throws FailedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Unload a filter from the camera into the loader.")
    public String unloadFilterFromCamera()  {

        updateStateWithSensors();
        if (!isConnectedOnCamera()) {
            throw new RejectedCommandException(getName() + ":loader has to be connected to "
                    + "camera first.");
        }
        if (isHoldingFilter()) {
            throw new RejectedCommandException(getName() + ":loader is holding a filter : "
                    + "can't unload a filter from camera.");
        }

        //wait for the autochanger to put a filter at handoff position
        //in loader test bench it's just a switch to put on.
        while (!isAutochangerHoldingFilter()) {
            FCSLOG.debug(getName() + " waiting until autochanger holds the filter...");
            try {
                Thread.sleep(300);
            } catch (InterruptedException ex) {
                throw new FcsHardwareException("unloadFilterFromCamera interrupted while waiting until autochanger holds the filter",ex);
            }
            updateStateWithSensors();
        }

        //go tho Handoff position - carrier empty
        goToHandoff();
        if (isHaltRequired() || isStopRequired()) {
            return getName() + ": received HALT or STOP command.";
        }

        //at handoff position a filter should be here.
        updateStateWithSensors();
        if (isEmpty()) {
            throw new FailedCommandException(getName()
                    + ": loader presence filter sensor doesn't detect a filter"
                    + " - can't go on.");
        }
        closeHooks();

        //wait for the autochanger to unlock the filter at handoff position
        //in loader test bench it's just a switch to put off.(switch A22)
        while (isAutochangerHoldingFilter()) {
            FCSLOG.debug(getName() + " waiting until autochanger releases the filter...");
            try {
                Thread.sleep(300);
            } catch (InterruptedException ex) {
                throw new FcsHardwareException("unloadFilterFromCamera interrupted while waiting until "
                        + "autochanger releases the filter",ex);
            }
            updateStateWithSensors();
        }

        //close more fermely the hooks to hold the filter.
        clampHooks();
        if (isHaltRequired() || isStopRequired()) {
            return getName() + ": received HALT or STOP command.";
        }

        //carrier holding filter goes to storage position.
        return goToStorage();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "List and display info on the hooks.")
    public String listHooks() {
        return this.clamp.listHooks();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "List and display hooks sensors values.")
    public String listSensorsValues() {
        return this.clamp.listSensorsValues();
    }

    public StatusDataPublishedByLoader getStatusData() {
        return createStatusDataPublishedByLoader();
    }
    
    /**
     * Creates an Object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByLoader createStatusDataPublishedByLoader() {
        StatusDataPublishedByLoader status = new StatusDataPublishedByLoader();
        status.setName(getName());
        status.setFilterPresenceSensorValue0(filterPresenceSensor0.getDigitalValue());
        status.setFilterPresenceSensorValue1(filterPresenceSensor1.getDigitalValue());
        status.setFilterPresenceSensorsInError(filterPresenceSensorsInError);
        status.setLoaderOnCameraSensorValue0(loaderOnCameraSensor0.getDigitalValue());
        status.setLoaderOnCameraSensorValue1(loaderOnCameraSensor1.getDigitalValue());
        status.setLoaderOnCameraSensorsInError(loaderOnCameraSensorsInError);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        StatusDataPublishedByLoader status = this.getStatusData();
        KeyValueData kvd = new KeyValueData("loaderGeneral", status);
        
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }

    @Override
    public boolean isAtHandoff() {
        return carrier.isAtHandoffPosition();
    }

    @Override
    public boolean isAtStandby() {
        return false;
    }

    /**
     * Print list of hardware with initialization information.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, 
            level = Command.ENGINEERING1, 
            description = "Print list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(carrier.printHardwareState());
        sb.append("\n");
        sb.append(clamp.printHardwareState());
        return sb.toString();
    }
}
