
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

/**
 * This is the Main Module for the Loader control software when the loader is in standalone. 
 *
 * @author virieux
 */
public class LoaderMainModule extends MainModule {

    private LoaderModule loader;
    private FilterHolder autochanger;

    /**
     * Build a LoaderMainModule with a bridge.
     * @param bridge 
     */
    public LoaderMainModule(
            BridgeToHardware bridge) {
        super(bridge);
    }

    public LoaderModule getLoader() {
        return loader;
    }

    @Override
    public void initModule() {
        super.initModule();
        ComponentLookup lookup = getComponentLookup();
        this.loader = (LoaderModule) lookup.getComponentByName("loader");
        this.autochanger = (FilterHolder) lookup.getComponentByName("autochanger");
    }
    

    /**
     * This command can be launched when a filter is in the loader and we want
     * to put in inside the camera.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Load a filter from the loader into the camera.")
    public String loadFilterInCamera()   {
        return loader.loadFilterInCamera();
        ////Moved to LoaderModule 29 april 2015 
    }

    /**
     * This command can be launched when the loader is empty and we want to take
     * a filter from the camera. A filter must be at handoff position and held
     * by autochanger.
     *
     * @return
     * @throws FcsHardwareException
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Unload a filter from the camera into the loader.")
    public String unloadFilterFromCamera()  {
        
        return loader.unloadFilterFromCamera();
        ////moved to LoaderModule 29 april 2015
    }

    /**
     * Open the loader clamp.
     * @return 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Open the loader clamp.")
    public String openHooks()  {
        return loader.openHooks();
    }

    /**
     * Close the loader clamp.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Close the loader clamp.")
    public String closeHooks()  {
        return loader.closeHooks();
    }

    /**
     * Clamp (close strongly) the loader clamp.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Clamp the loader clamp.")
    public String clampHooks()  {
        return loader.clampHooks();
    }

    /**
     * Update loader state in reading the sensors.
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Update loader state in reading the sensors.")
    @Override
    public void updateStateWithSensors()  {
        super.updateStateWithSensors();
        autochanger.updateStateWithSensors();
        loader.updateStateWithSensors();
    }
    
    /**
     * Check that Loader hardware is ready to be operated and moved.
     * This means that :
     * - all CAN open devices are booted, identified and initialized,
     * - homing has been done on the controllers.
     * This updates the FCS state and FCS readyness state and publishes on the status bus.
     * 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Update FCS state and FCS readyness state and publishes on the status bus.")    
    @Override
    public void updateFCSState() {
        if (loader.isInitialized()) {
            /* The initialization has been done, so now the hardware is ready */
            this.getSubsystem().updateAgentState(FcsEnumerations.FilterState.HARDWARE_READY, 
                FcsEnumerations.FilterReadinessState.READY);
        }
    }
 
    /**
     * Print list of hardware with initialization information.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Print the list of hardware with initialization information.")    
    @Override
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(super.printHardwareState());
        sb.append("\n ");
        sb.append(loader.printHardwareState());
        return sb.toString();
    }

    /**
     * Move carrier to HANDOFF position.
     * @return 
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Carrier go to Handoff.")
    public String goToHandoff()  {
        return loader.goToHandoff();
    }

    /**
     * Move carrier to STORAGE position.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Carrier go to Storage.")
    public String goToStorage()  {
        return loader.goToStorage();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "List and display hooks sensors.")
    public String listHooksSensors() {
        return this.loader.getClamp().listHooks();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "List and display the hooks sensors values.")
    public String listSensorsValues() {
        return this.loader.getClamp().listSensorsValues();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("\n").append(loader.toString());
        return sb.toString();
    }

}
