/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.Map;
import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This is model for the clamps mechanism in the loader. 
 * 4 hooks are used for the clamping of the filter in the loader. 
 * This hooks are moved all together by the hooksController.
 * 
 * Raises alert :
 * - FCS006 in processUpdate
 *
 * @author virieux
 */
public class LoaderClampModule extends MobileItemModule implements MovedByEPOSController {

    private final LoaderHookModule hook1;
    private final LoaderHookModule hook2;
    private final LoaderHookModule hook3;
    private final LoaderHookModule hook4;
    private EPOSController hooksController;
    private LoaderModule loader;

    private FcsEnumerations.LockStatus lockStatus;

    @ConfigurationParameter(description="timeout in milliseconds : if closing the clamp last more than "
            + "this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForClosingHooks = 60000;
    
    @ConfigurationParameter(description="timeout in milliseconds : if closing strongly the clamp last "
    + "more than this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForClampingHooks = 60000;
    
    @ConfigurationParameter(description="timeout in milliseconds : if opening the clamp last more than"
    + " this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForOpeningHooks = 60000;
    
    @ConfigurationParameter(description="timeout in milliseconds : if the action of hoing to home position "
            + "last more than this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForGoingToHomePosition = 60000;
    
    @ConfigurationParameter(description="target encoder absolute value in qc to open")
    private int targetPositionToOpen = 5000;
    
    @ConfigurationParameter(description="target encoder absolute value in qc when hooks are LOCKED")
    private int targetPositionToClose = 492000;
    
    @ConfigurationParameter(description="target encoder absolute value in qc when hooks are LOCKED STRONGLY")
    private int targetPositionToClamp = 515000;
    
    @ConfigurationParameter(description="current to close in initialisation phase, in mA")
    private int currentToClamp = 280;
    
    @ConfigurationParameter(description="current to go to home position in initialisation phase, in mA")
    private int currentToGoHome = -200;
    
    /* for GUI */
    @ConfigurationParameter(description="min position value used by the GUI")
    private int minPosition = 0;
    
    /* for GUI */
    @ConfigurationParameter(description="max position value used by the GUI")
    private int maxPosition = 515000; 
    
    /* for GUI */
    @ConfigurationParameter(description="min current value used by the GUI")
    private final int minCurrent = -300;
            
    /* for GUI */
    @ConfigurationParameter(description="max current value used by the GUI")
    private final int maxCurrent = 400;
    
    @ConfigurationParameter
    private Map<String, Integer> paramsForCurrentToGoHome;
    
    @ConfigurationParameter
    private Map<String, Integer> paramsForCurrentToClamp;
            
    private volatile boolean initialized = false;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    private int position;
    private int readCurrent;
    
    /**
     * This is the value returned by the force sensor.
     */
    private int force;
    

    /**
     * FOR THE GUI
     * true if the controller is in fault
     * false if the user does a faultReset
     */
    private boolean controllerInFault;



    /**
     * Build a new LoaderClampModule with 4 hooks and the parameters to configure the EPOS controller
     * in mode CURRENT and in mode HOMING.
     * @param hook1
     * @param hook2
     * @param hook3
     * @param hook4
     * @param paramsForCurrentToClamp
     * @param paramsForCurrentToGoHomePosition 
     */
    public LoaderClampModule(
            LoaderHookModule hook1,
            LoaderHookModule hook2,
            LoaderHookModule hook3,
            LoaderHookModule hook4,
            Map<String, Integer> paramsForCurrentToClamp,
            Map<String, Integer> paramsForCurrentToGoHomePosition) {
        this.hook1 = hook1;
        this.hook2 = hook2;
        this.hook3 = hook3;
        this.hook4 = hook4;
        this.paramsForCurrentToClamp = paramsForCurrentToClamp;
        this.paramsForCurrentToGoHome = paramsForCurrentToGoHomePosition;
    }
    
    /**
     * Returns controller name.
     * @return 
     */
    @Override
    public String getControllerName() {
        return hooksController.getName();
    }

    @Override
    public boolean isControllerInFault() {
        return this.controllerInFault; 
    }

    @Override
    public void setControllerInFault(boolean controllerInFault) {
        this.controllerInFault = controllerInFault;
    }

    //for publication on the status bus for the GUI
    public int getFilterPresenceSensorValue() {
        return this.loader.getFilterPresenceSensor0().getDigitalValue();
    }

    public LoaderHookModule getHook1() {
        return hook1;
    }

    public LoaderHookModule getHook2() {
        return hook2;
    }

    public LoaderHookModule getHook3() {
        return hook3;
    }

    public LoaderHookModule getHook4() {
        return hook4;
    }

    public int getPosition() {
        return position;
    }

    //for the GUI
    public int getMinPosition() {
        return minPosition;
    }

    //for the GUI
    public int getMaxPosition() {
        return maxPosition;
    }

    public int getMinCurrent() {
        return minCurrent;
    }

    public int getMaxCurrent() {
        return maxCurrent;
    }

    //for the simulator
    public int getTargetPositionToOpen() {
        return targetPositionToOpen;
    }

    //for the simulator
    public int getTargetPositionToClose() {
        return targetPositionToClose;
    }

    //for the simulator
    public int getTargetPositionToClamp() {
        return targetPositionToClamp;
    }

    //for the simulator
    public int getCurrentToClamp() {
        return currentToClamp;
    }

    //for the simulator
    public int getCurrentToGoHome() {
        return currentToGoHome;
    }

    public int getReadCurrent() {
        return readCurrent;
    }





    /**
     * Returns true if loader clamp is initialized.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if loader clamp is initialized.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * Set min position.
     * @param minPosition 
     */
    public void setMinPosition(int minPosition) {
        if (minPosition > maxPosition) {
            throw new IllegalArgumentException(getName()
                    + "minPosition can't be > maxPosition; maxPosition=" + maxPosition);
        }
        this.minPosition = minPosition;
    }

    /**
     * Set max position.
     * @param maxPosition 
     */
    public void setMaxPosition(int maxPosition) {
        if (minPosition > maxPosition) {
            throw new IllegalArgumentException(getName()
                    + "maxPosition can't be < minPosition; minPosition=" + minPosition);
        }
        this.maxPosition = maxPosition;
    }






    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(getName() + ": interrupted in getLockStatus.",ex);
                }

            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns true if LockStatus=LOCKED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=LOCKED")
    public boolean isLocked() {
        return this.getLockStatus() == LockStatus.LOCKED;
    }

    /**
     * Returns true if LockStatus=UNLOCKED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=UNLOCKED")
    public boolean isUnlocked() {
        return this.getLockStatus() == LockStatus.UNLOCKED;
    }

    /**
     * Returns true if LockStatus=UNLOCKED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=UNLOCKED")
    public boolean isInError() {
        return this.getLockStatus() == LockStatus.ERROR;
    }

    /**
     * Returns true if loader is empty.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if loader is empty.")
    public boolean isEmpty() {
        return loader.isEmpty();
    }

    /**
     * Returns true if loader CANopen devices are booted, identified and initialized.
     * @return 
     */
    @Override
    public boolean isCANDevicesReady() {
        return loader.isCANDevicesReady();
    }

    @Override
    public void initModule() {
        ComponentLookup lookup = getComponentLookup();
        this.hooksController = (EPOSController) lookup.getComponentByName("hooksController");
        this.loader = (LoaderModule) lookup.getComponentByName("loader");
        this.lockStatus = LockStatus.UNKNOWN;

        //listens to hooksController to detect the controller's faultReset
        //or the emergency messages coming from the controller.
        if (hooksController instanceof Observable) {
            this.listens((Observable) hooksController);
        }
    }

    /**
     * Returns true if hook is open and at home position
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if hook is open and at home position")
    public boolean isAtHomePosition()  {
        return position == 0;
    }

    /**
     * Returns true if hook is closed and at clamped position
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if hook is closed and at clamped position")
    public boolean isAtClampedPosition() {
        return (this.targetPositionToClamp - 1000 <= this.position)
                && (this.targetPositionToClamp <= this.position + 1000);
    }

    /**
     * Returns true if hook is at closed position.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if hook is at closed position.")
    public boolean isAtClosedPosition()  {
        return position == this.targetPositionToClose;
    }

    /**
     * Returns true if hook is at open position.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if hook is at open position.")
    public boolean isAtOpenPosition() {
        return position == this.targetPositionToOpen;
    }

    /**
     * If loader is empty, go to home position otherwise go to clamped position.
     * ATTENTION : this command moves the clamp.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Attention : this commands moves the clamp to do the homing of the controller. "
                    + "If loader is empty, go to home position otherwise go to clamped position.",
            alias = "homing")
    public void initializeHardware()  {

        this.updateStateWithSensors();
        publishData();

        if (this.isInError()) {
            String msg = getName() + " in ERROR state. Can't initialize hardware.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }

        if (!hooksController.isInitialized()) {
            throw new FcsHardwareException(getName()
                    + ": hooksController has to be initialized first.");
        }

        if (this.isEmpty()) {
            this.goToHomePosition();

        } else {
            this.goToClampedPosition();

        }
        this.initialized = true;
        loader.updateFCSState();
    }

    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        try {
            hooksController.initializeAndCheckHardware();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + ":" + ex);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false, ex);
        }
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    @Override
    public void tick() {
        //Don't execute here a method which takes a lock.
        publishData();
    }

    /**
     * In the initialisation phase, this close the clamp to be able to define
     * the absolute position of the encoder.
     *
     * @return
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Go to clamped position to intialize hardware.")
    public String goToClampedPosition()  {

        if (!loader.isConnectedOnCamera()) {
            throw new RejectedCommandException(getName()
                    + " Loader not connected - can't execute command goToClampedPosition.");
        }

        updateStateWithSensors();

        if (this.isInError()) {
            String msg = getName() + " in ERROR state - can't execute command goToClampedPosition.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }

        if (isEmpty()) {
            throw new RejectedCommandException(getName()
                    + "No filter in carrier -  can't execute command goToClampedPosition.");
        }

        updatePosition();
        if (initialized && isAtClampedPosition()) {
            String msg = getName() + " is already at clamped position - nothing to do.";
            FCSLOG.warning(msg);
            return msg;
        }

        return this.executeAction(FcsEnumerations.MobileItemAction.GOTOCLAMPEDPOSITION,
                timeoutForClampingHooks);
    }

    /**
     * In the initialisation phase, this open the clamp to be able to define the
     * absolute position of the encoder. September 2014 : this command is also
     * used to open the clamp.
     *
     * @return
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Go to home position to initialize hardware.")
    public String goToHomePosition()  {

        updateStateWithSensors();

        if (this.isInError()) {
            String msg = getName() + " in ERROR state - can't execute command goToHomePosition.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }
        if (!loader.isConnectedOnCamera()) {
            throw new RejectedCommandException(getName()
                    + " Loader hardware not connected on camera - "
                    + "can't execute command goToHomePosition.");
        }

        if (!initialized && !isEmpty()) {
            throw new RejectedCommandException(getName()
                    + " There is a filter in carrier - can't execute command goToHomePosition.");
        }

        if (initialized && !isEmpty() && !loader.isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + " There is a filter in carrier and Autochanger is not holding filter - "
                    + "can't execute command goToHomePosition. ");
        }

        updatePosition();
        if (initialized && isAtHomePosition()) {
            String msg = getName() + " is already at home position - nothing to do.";
            FCSLOG.warning(msg);
            return msg;
        }

        return this.executeAction(FcsEnumerations.MobileItemAction.GOTOHOMEPOSITION,
                timeoutForGoingToHomePosition);
    }

    /**
     * Close clamp in order to hold the filter.
     * When the clamp is closed, the filter is held by loader so autochanger can open its latches
     * and move trucks back without the filter.
     * @return
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Close clamp.")
    public String close()  {

        loader.checkPreConditionsForClosingHooks();
        if (this.isAtClosedPosition()) {
            String msg = getName() + ": is already closed.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
        return this.executeAction(FcsEnumerations.MobileItemAction.CLOSELOADERHOOKS,
                timeoutForClosingHooks);

    }

    /**
     * Clamp clamp or strongly close.
     * When the clamp is clamped, the filter is held strongly and so the carrier can go to STORAGE position safely.
     * @return
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Clamp clamp.")
    public String clamp()  {

        loader.checkPreConditionsForClosingHooks();
        if (!isLocked()) {
            throw new RejectedCommandException(getName()
                    + " has to be LOCKED first. Close hooks prior clamp.");
        }
        if (loader.isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + " Autochanger is holding filter. Open autochanger latches before "
                    + "clamping loader hooks.");
        }
        return this.executeAction(FcsEnumerations.MobileItemAction.CLAMPLOADERHOOKS,
                timeoutForClampingHooks);
    }

    /**
     * Open clamp in order to release filter.
     * @return
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open clamp.")
    public String open()  {

        loader.checkConditionsForOpeningHooks();
        return this.executeAction(FcsEnumerations.MobileItemAction.OPENLOADERHOOKS,
                timeoutForOpeningHooks);
    }

    /**
     * Check if the action is completed. This method is called after
     * updateStateWithSensorsToCheckIfActionIsCompleted where the different new
     * values of current or position or sensors values have been updated.
     *
     * @param action
     * @return
     */
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        /* A loader clamp motion is completed when the position is in a range of */
        /* 10 microns around the target position. */
        int DELTA = 5;
        switch (action) {
            case GOTOCLAMPEDPOSITION:
                //TODO read the force sensor when it will be installed
                FCSLOG.debug("readCurrent=" + readCurrent + " currentToClamp=" + this.currentToClamp);
                return (this.currentToClamp - DELTA <= this.readCurrent)
                        && (this.readCurrent <= this.currentToClamp + DELTA);

            case GOTOHOMEPOSITION:
                FCSLOG.debug("readCurrent=" + readCurrent + " currentToGoHome=" + this.currentToGoHome);
                return (this.currentToGoHome - DELTA <= this.readCurrent)
                        && (this.readCurrent <= this.currentToGoHome + DELTA);

            case CLOSELOADERHOOKS:
                return this.position == this.targetPositionToClose;

            case CLAMPLOADERHOOKS:
                //TODO read the force sensor
                return (this.currentToClamp - DELTA <= this.readCurrent)
                        && (this.readCurrent <= this.currentToClamp + DELTA);

            case OPENLOADERHOOKS:
                return this.position == this.targetPositionToOpen;
                
            default:
                assert false: action;
        }
        return false;

    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        this.updateStateWithSensors();
        if ((currentAction.equals(MobileItemAction.CLOSELOADERHOOKS))
                || (currentAction.equals(MobileItemAction.OPENLOADERHOOKS))) {
            position = this.hooksController.readPosition();
        } else if ((currentAction.equals(MobileItemAction.GOTOHOMEPOSITION))
                || (currentAction.equals(MobileItemAction.GOTOCLAMPEDPOSITION))
                || (currentAction.equals(MobileItemAction.CLAMPLOADERHOOKS))) {
            this.readCurrent = this.hooksController.readCurrent();
            //just to display on the GUI the position of the clamp
            position = this.hooksController.readPosition();
            force = this.readForceSensor();
        }
        //TODO read the force sensor for the actions GOTOCLAMPEDPOSITION and CLAMP
    }

    @Override
    public void startAction(MobileItemAction action)  {

        switch (action) {
            case GOTOCLAMPEDPOSITION:
                hooksController.enable();
                hooksController.changeMode(EposMode.CURRENT);
                hooksController.writeParameters(paramsForCurrentToClamp);
                hooksController.writeCurrent(this.currentToClamp);
                break;

            case GOTOHOMEPOSITION:
                hooksController.enable();
                hooksController.changeMode(EposMode.CURRENT);
                hooksController.writeParameters(paramsForCurrentToGoHome);
                hooksController.writeCurrent(this.currentToGoHome);
                break;

            case CLOSELOADERHOOKS:
                hooksController.enable();
                hooksController.changeMode(EposMode.PROFILE_POSITION);
                hooksController.writeParameters(EposMode.PROFILE_POSITION);
                hooksController.writeTargetPosition(this.targetPositionToClose);
                //hooksController.on();
                hooksController.writeControlWord("3F");
                break;

            case CLAMPLOADERHOOKS:
                if (!(this.lockStatus.equals(LockStatus.LOCKED))) {
                    throw new RejectedCommandException(getName() + 
                            ": Hooks have to be LOCKED before being LOCKED STRONGLY");
                }
                hooksController.enable();
                hooksController.changeMode(EposMode.CURRENT);
                hooksController.writeParameters(paramsForCurrentToClamp);
                hooksController.writeCurrent(this.currentToClamp);
                break;

            case OPENLOADERHOOKS:
                hooksController.enable();
                hooksController.changeMode(EposMode.PROFILE_POSITION);
                hooksController.writeParameters(EposMode.PROFILE_POSITION);
                hooksController.writeTargetPosition(this.targetPositionToOpen);
                hooksController.writeControlWord("3F");
                break;
                
            default:
                assert false;
        }
    }

    @Override
    public void abortAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(getName() + " is ABORTING action " + action.toString()
                + " within delay " + delay);
      
        FCSLOG.info("Current Command: "+getSubsystem().getCurrentAction()+" "+getSubsystem().getState());

        this.hooksController.off();

    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(getName() + " is STOPPING action " + action.toString()
                + " within delay " + delay);
       
        FCSLOG.info("Current Command: "+getSubsystem().getCurrentAction()+" "+getSubsystem().getState());

        this.hooksController.quickStop();

    }

    @Override
    public void postAction(MobileItemAction action)  {

        //because we don't want to let the controller on power
        this.hooksController.off();
        FCSLOG.info(getName() + ":" + action.toString() + " completed - doing postAction.");

        switch (action) {
            case GOTOCLAMPEDPOSITION:
                if (!this.isLocked()) {
                    throw new FailedCommandException(getName()
                            + ": check with sensors: clamp should be LOCKED.");
                }
                this.hooksController.defineAbsolutePosition(this.targetPositionToClamp);
                this.hooksController.off();
                updatePosition(); // for the GUI
                this.lockStatus = LockStatus.CLAMPED; //for the GUI
                this.initialized = true;
                this.publishData();
                loader.updateFCSState();
                break;

            case GOTOHOMEPOSITION:
                if (!this.isUnlocked()) {
                    throw new FailedCommandException(getName()
                            + ": check with sensors: clamp should be UNLOCKED.");
                }
                this.hooksController.defineAbsolutePosition(0);
                this.hooksController.off();
                updatePosition(); // for the GUI
                this.initialized = true;
                this.publishData();
                loader.updateFCSState();
                break;

            case CLOSELOADERHOOKS:
                if (!this.isLocked()) {
                    throw new FailedCommandException(getName()
                            + ": check with sensors: clamp should be LOCKED.");
                }
                break;

            case CLAMPLOADERHOOKS:
                if (this.isLocked() && this.isAtClampedPosition()) {
                    this.lockStatus = FcsEnumerations.LockStatus.CLAMPED;
                    this.publishData();
                }
                break;

            case OPENLOADERHOOKS:
                if (!this.isUnlocked()) {
                    throw new FailedCommandException(getName()
                            + ": check with sensors: clamp should be UNLOCKED.");
                }
                break;
                
            default:
                assert false: action;
        }
    }

    /**
     * This methods updates the lockStatus in reading all the sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update clamp state in reading sensors.")
    public void updateStateWithSensors()  {
        loader.updateStateWithSensors();
    }

    /**
     * This methods updates the lockStatus from an array of hexaValues.
     *
     * @param readHexaValues
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    void updateStateWithSensors(String[] readHexaValues)  {
        lock.lock();
        try {
            updatingState = true;
            this.hook1.updateStateWithSensors(readHexaValues);
            this.hook2.updateStateWithSensors(readHexaValues);
            this.hook3.updateStateWithSensors(readHexaValues);
            this.hook4.updateStateWithSensors(readHexaValues);

            boolean inError = this.hook1.getLockStatus() == LockStatus.ERROR
                    || this.hook2.getLockStatus() == LockStatus.ERROR
                    || this.hook3.getLockStatus() == LockStatus.ERROR
                    || this.hook4.getLockStatus() == LockStatus.ERROR;

            boolean locked = this.hook1.getLockStatus() == LockStatus.LOCKED
                    && this.hook2.getLockStatus() == LockStatus.LOCKED
                    && this.hook3.getLockStatus() == LockStatus.LOCKED
                    && this.hook4.getLockStatus() == LockStatus.LOCKED;

            boolean unlocked = this.hook1.getLockStatus() == LockStatus.UNLOCKED
                    && this.hook2.getLockStatus() == LockStatus.UNLOCKED
                    && this.hook3.getLockStatus() == LockStatus.UNLOCKED
                    && this.hook4.getLockStatus() == LockStatus.UNLOCKED;

            boolean inTravel = this.hook1.getLockStatus() == LockStatus.INTRAVEL
                    && this.hook2.getLockStatus() == LockStatus.INTRAVEL
                    && this.hook3.getLockStatus() == LockStatus.INTRAVEL
                    && this.hook4.getLockStatus() == LockStatus.INTRAVEL;

            if (inError) {
                this.lockStatus = LockStatus.ERROR;
            } else if (locked) {
                this.lockStatus = LockStatus.LOCKED;
            } else if (unlocked) {
                this.lockStatus = LockStatus.UNLOCKED;
            } else if (inTravel) {
                this.lockStatus = LockStatus.INTRAVEL;
            } else {
                this.lockStatus = LockStatus.UNKNOWN;
            }

            if (initialized && locked && isAtClampedPosition()) {
                this.lockStatus = LockStatus.CLAMPED;
            }
            

        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
        }

    }
    
    //TODO put here the right code to read force sensor.
    private int readForceSensor() {
        return 0;
    }   

    /**
     * Updates the field readCurrent of the clamp in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update clamp current in reading controller.")
    public void updateCurrent() {
        try {
            this.readCurrent = hooksController.readCurrent();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> ERROR IN READING CONTROLLER:",ex);
        }
        this.publishData();
    }

    /**
     * Updates the position of the clamp in reading the CPU of the controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Update clamp position in reading controller.")
    public void updatePosition()  {
        try {
            this.position = hooksController.readPosition();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> ERROR IN READING CONTROLLER:",ex);

        }
        this.publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "List and display clamp info.")
    public String listHooks() {
        StringBuilder sb = new StringBuilder("Hooks status:");
        sb.append("\n");
        sb.append(this.hook1.toString());
        sb.append("/");
        sb.append(this.hook1.getLockStatus());
        sb.append("\n");
        sb.append(this.hook2.toString());
        sb.append("/");
        sb.append(this.hook2.getLockStatus());
        sb.append("\n");
        sb.append(this.hook3.toString());
        sb.append("/");
        sb.append(this.hook3.getLockStatus());
        sb.append("\n");
        sb.append(this.hook4.toString());
        sb.append("/");
        sb.append(this.hook4.getLockStatus());
        sb.append("\n");
        return sb.toString();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "List and display clamp sensors values.")
    public String listSensorsValues() {
        StringBuilder sb = new StringBuilder(getName() + "Sensors values=");
        sb.append("/");
        sb.append(this.hook1.listSensorsValues());
        sb.append("/");
        sb.append(this.hook2.listSensorsValues());
        sb.append("/");
        sb.append(this.hook3.listSensorsValues());
        sb.append("/");
        sb.append(this.hook4.listSensorsValues());
        sb.append("/");
        return sb.toString();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "List and display clamp sensors values.")
    public String listStatus() {
        StringBuilder sb = new StringBuilder("Clamp status:");
        sb.append(this.lockStatus);
        sb.append("\n");
        sb.append(this.hook1.getName());
        sb.append("/");
        sb.append(this.hook1.getLockStatus());
        sb.append("\n");
        sb.append(this.hook2.getName());
        sb.append("/");
        sb.append(this.hook2.getLockStatus());
        sb.append("\n");
        sb.append(this.hook3.getName());
        sb.append("/");
        sb.append(this.hook3.getLockStatus());
        sb.append("\n");
        sb.append(this.hook4.getName());
        sb.append("/");
        sb.append(this.hook4.getLockStatus());
        sb.append("\n");
        return sb.toString();
    }
    
    /**
     * Return a printed list of hardware with initialization information.
     * For debug purpose.
     * @return 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware with initialization information.")    
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(getName());
        if (this.isInitialized()) {
            sb.append(" is INITIALIZED.");
        } else {
            sb.append(" is NOT INITIALIZED.");
        }
        return sb.toString();
    }    

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "List and display clamp info.")
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("/timeoutForClosingHooks=");
        sb.append(this.timeoutForClosingHooks);
        sb.append("/timeoutForClosingHooksStrongly=");
        sb.append(this.timeoutForClampingHooks);
        sb.append("/timeoutForOpeningHooks=");
        sb.append(this.timeoutForOpeningHooks);
        return sb.toString();
    }

    /**
     * Returns the object to be published on the STATUS bus by the LoaderClampModule.
     * @return 
     */
    public StatusDataPublishedByLoaderClamp getStatusData() {
        return createStatusDataPublishedByLoaderClamp();
    }
    
    /**
     * Creates and returns the object to be published on the STATUS bus by the LoaderClampModule.
     * @return 
     */
    public StatusDataPublishedByLoaderClamp createStatusDataPublishedByLoaderClamp() {
        StatusDataPublishedByLoaderClamp status = new StatusDataPublishedByLoaderClamp();
        status.setName(getName());
        status.setPosition(position);
        status.setCurrent(readCurrent);
        status.setClampState(lockStatus);
        status.setForce(force);
        status.setHomingDone(initialized);
        status.setStatusPublishedByHook1(hook1.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook2(hook2.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook3(hook3.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook4(hook4.createStatusDataPublishedByLoaderHook());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        StatusDataPublishedByLoaderClamp status = this.getStatusData();
        KeyValueData kvd = new KeyValueData("loaderClamp", status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }




}
