
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This is a model for the carrier in the loader.
 *
 * @author virieux
 */
public class LoaderCarrierModule extends MobileItemModule implements MovedByEPOSController {

    protected NumericSensor storagePositionSensor0;
    protected NumericSensor handoffPositionSensor0;
    protected NumericSensor storagePositionSensor1;
    protected NumericSensor handoffPositionSensor1;

    private int position = 0;
    private int ssiPosition = 0;
    private int absoluteTargetPosition = 0;
    
    @ConfigurationParameter(description="Loader Handoff position in micron")
    private int handoffPosition = 1037000;
    
    @ConfigurationParameter(description="Loader Storage position in micron")
    private int storagePosition = 0; 
    
    @ConfigurationParameter(description=
            "For the Loader GUI : Maximum current to be sent to the Loader Carrier controller (in mA).")
    private int maxCurrent = 1000;
        
    @ConfigurationParameter(description="For the Loader GUI : Loader Carrier Maximum speed (in rpm/mn).")
    private int maxSpeed = 227;

    @ConfigurationParameter(description="timeout in milliseconds to go from storage to handoff on loader")
    private long timeoutForGoingToHandOff = 120000;
    
    @ConfigurationParameter(description="timeout in milliseconds to go from handoff to storage on loader")
    private long timeoutForGoingToStorage = 120000;
    
    private int speed;
    private int current;

    private EPOSController carrierController;
    private LoaderModule loader;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    /**
     * true if the 2 position sensors value at Storage is 1 *
     */
    private boolean atStorage = false;

    /**
     * true if the 2 position sensors value at Handoff is 1 *
     */
    private boolean atHandoff = false;

    /**
     * true if the 2 position sensors value at Storage differ *
     */
    private boolean storageSensorsInError = false;

    /**
     * true if the 2 position sensors value at Handoff differ *
     */
    private boolean handoffSensorsInError = false;


    /**
     * true if the controller is in fault - false if the user does a faultReset*
     */
    private boolean controllerInFault = false;

    /**
     * error message received in the emergency message sent by carrierController*
     */
    private boolean limitSwitchDownInError = false;
    private boolean limitSwitchUpInError = false;
    private volatile boolean initialized = false;

    /**
     * Creation of a LoaderCarrierModule with 4 sensors.
     * @param handoffPositionSensor
     * @param handoffPositionSensor1
     * @param storagePositionSensor
     * @param storagePositionSensor1 
     */
    public LoaderCarrierModule(
            NumericSensor handoffPositionSensor,
            NumericSensor handoffPositionSensor1,
            NumericSensor storagePositionSensor,
            NumericSensor storagePositionSensor1) {
        this.storagePositionSensor0 = storagePositionSensor;
        this.handoffPositionSensor0 = handoffPositionSensor;
        this.storagePositionSensor1 = storagePositionSensor1;
        this.handoffPositionSensor1 = handoffPositionSensor1;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns carrier position.")    
    public int getPosition() {
        return position;
    }

    public int getSsiPosition() {
        return ssiPosition;
    }

    public int getSpeed() {
        return speed;
    }

    public int getCurrent() {
        return current;
    }

    /**
     * Returns true if carrierController is in Fault.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if carrierController is in Fault.")
    @Override
    public boolean isControllerInFault() {
        return controllerInFault;
    }
    
    
    @Override
    public String getControllerName() {
        return this.carrierController.getName();
    }

    @Override
    public void setControllerInFault(boolean controllerInFault) {
        this.controllerInFault = controllerInFault;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if carrier is initialized and ready to receive commands.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     *
     * @return max speed in rpm/mn (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the max speed in rpm/mn (format decimal).")
    public int getMaxSpeed() {
        return maxSpeed;
    }

    /**
     *
     * @return max current in mA (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the max current in mA (format decimal).")
    public int getMaxCurrent() {
        return maxCurrent;
    }

    /**
     *
     * @return handoff position in microns (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the handoff position in microns (format decimal).")
    public int getHandoffPosition() {
        return handoffPosition;
    }

    /**
     *
     * @return storage position in microns (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the storage position in microns (format decimal).")
    public int getStoragePosition() {
        return storagePosition;
    }

    public NumericSensor getStoragePositionSensor0() {
        return storagePositionSensor0;
    }

    public NumericSensor getHandoffPositionSensor0() {
        return handoffPositionSensor0;
    }

    public NumericSensor getStoragePositionSensor1() {
        return storagePositionSensor1;
    }

    public NumericSensor getHandoffPositionSensor1() {
        return handoffPositionSensor1;
    }

    /**
     *
     * @return timeout for going to Handoff action in millis (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the timeout for going to Handoff in millis (format decimal).")
    public long getTimeoutForGoingToHandOff() {
        return timeoutForGoingToHandOff;
    }

    /**
     *
     * @return timeout for going to Storage action in millis (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the timeout for going to Storage in millis (format decimal).")
    public long getTimeoutForGoingToStorage() {
        return timeoutForGoingToStorage;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Storage are equal.")
    public boolean isStorageSensorsInError() {
        return storageSensorsInError;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at HandOff are equal.")
    public boolean isHandoffSensorsInError() {
        return handoffSensorsInError;
    }

    public boolean isLimitSwitchDownInError() {
        return limitSwitchDownInError;
    }

    public boolean isLimitSwitchUpInError() {
        return limitSwitchUpInError;
    }
    



    @Override
    public void initModule() {
        ComponentLookup lookup = getComponentLookup();
        this.loader = (LoaderModule) lookup.getComponentByName("loader");
        this.carrierController = (EPOSController) lookup.getComponentByName("carrierController");

        //listens to carrierController to detect the controller's faultReset
        //or the emergency messages coming from the controller.
        this.listens((Observable) carrierController);
    }

    @Override
    public void tick() {
        this.publishData();
    }

    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        super.checkHardware();
        try {
            carrierController.initializeAndCheckHardware();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + ":",ex);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false, ex);
        }
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * Returns true if loader CANopen devices are booted, identified and initialized.
     * @return 
     */
    @Override
    public boolean isCANDevicesReady() {
        return loader.isCANDevicesReady();
    }

    /**
     * Updates the boolean field empty and returns it.
     *
     * @return empty
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Read the sensors and return true if a filter is in the loader.")
    public boolean checkFilterPresence()  {
        this.updateStateWithSensors();
        return !this.isEmpty();
    }

    /**
     * Returns the boolean field empty. If the empty boolean is being updated
     * and waits for a response from a sensor, this methods waits until empty is
     * updated. If the field empty is not being updated, it returns immediatly
     * the field empty.
     *
     * @return empty
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if there is no filter in the loader. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {
        return loader.isEmpty();
    }

    /**
     * Returns the boolean field atStorage. If the atStorage boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * atStorage is updated. If the field atStorage is not being updated, it
     * returns immediatly the field atStorage.
     *
     * @return atStorage
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at storage position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtStoragePosition() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(getName() + ": has been interrupted while waiting for end of update.",ex);
                }

            }
            return atStorage;

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * atHandoff is updated. If the field atHandoff is not being updated, it
     * returns immediatly the field atHandoff.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at HANDOFF position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtHandoffPosition() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(getName() + ": has been interrupted while waiting for end of update.",ex);
                }

            }
            return atHandoff;

        } finally {
            lock.unlock();
        }
    }

    /**
     *
     * @return true if loader is connected on camera
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the loader is connected on the camera. "
            + "This command doesn't read again the sensors.")
    public boolean isConnectedOnCamera() {
        return loader.isConnectedOnCamera();
    }

    /**
     * Return true if the autochanger is holding the filter at HANDOFF.
     * This command doesn't read again the sensors.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the autochanger is holding the filter. "
            + "This command doesn't read again the sensors.")
    public boolean isAutochangerHoldingFilter()  {
        return loader.isAutochangerHoldingFilter();
    }

    /**
     * Move the carrier to Handoff position.
     *
     * @return
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            alias = "goToHandoff",
            description = "Move the carrier to Handoff position.")
    public String goToHandOff()  {
        if (this.atHandoff) {
            throw new RejectedCommandException(getName() + " is already at Handoff position.");
        }
        loader.checkConditionsForCarrierMotion();
        return this.executeAction(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_HANDOFF,
                timeoutForGoingToHandOff);
    }

    /**
     * Move the carrier to storage position.
     *
     * @return
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the carrier to STORAGE position.")
    public String goToStorage()  {
        if (this.atStorage) {
            throw new RejectedCommandException(getName() + " is already at STORAGE position.");
        }
        loader.checkConditionsForCarrierMotion();
        return this.executeAction(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_STORAGE,
                timeoutForGoingToStorage);
    }
    
    /**
     * For end user.
     * Move the carrier to absolute position given as argument (in decimal format).
     * @param absolutePosition
     * @return 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the carrier to absolute position given as argument (in decimal format).")    
    public String goToAbsolutePosition(int absolutePosition) {
        this.updatePosition();
        if (position == absolutePosition) {
            throw new RejectedCommandException(getName() + " is already at position "+absolutePosition);
        }
        loader.checkConditionsForCarrierMotion();
        this.absoluteTargetPosition = absolutePosition;
        return this.executeAction(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION,
                timeoutForGoingToStorage);
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {

            case MOVE_LOADERCARRIER_TO_HANDOFF:
                return this.position == this.handoffPosition;

            case MOVE_LOADERCARRIER_TO_STORAGE:
                return this.position == this.storagePosition;
                
            case MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION:
                return this.position == this.absoluteTargetPosition;
                
            default:
                assert false;
        }
        return false;
    }

    /**
     * Updates the field position of the carrier in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update carrier position in reading controller.")
    public void updatePosition()  {
        this.position = carrierController.readPosition();
        this.publishData();
    }

    /**
     * To display position for end user.
     * Updates carrier position in reading controller and returns it.
     * @return position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "To update and display position for end user."
                    + "Updates carrier position in reading controller and returns it.")
    public int readPosition() {
        updatePosition();
        return this.position;
    }

    /**
     * Updates the field position of the carrier in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update carrier current in reading controller.")
    public void updateCurrent()  {
        this.current = carrierController.readCurrent();
        this.publishData();
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            carrierController.checkFault();
            this.position = this.carrierController.readPosition();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> SDO ERROR IN READING CONTROLLER:",ex);
        }
        //for the GUI : is it the right place ?
        try {
            this.updateStateWithSensors();
            this.current = this.carrierController.readCurrent();
            this.speed = this.carrierController.readProfileVelocity();
        } catch (ShortResponseToSDORequestException | SDORequestException ex) {
            FCSLOG.warning(getName() + "=> SDO ERROR IN READING CONTROLLER:",ex);
        }

    }

    /**
     * This methods updates the carrier state in reading all the carrier
     * sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update clamp state in reading sensors.")
    public void updateStateWithSensors()  {
        loader.updateStateWithSensors();
    }

    /**
     * This methods updates the lockStatus from an array of hexaValues.
     *
     * @param readHexaValues
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    protected void updateStateWithSensors(String[] readHexaValues){
        lock.lock();

        try {
            updatingState = true;

            this.handoffPositionSensor0.updateValue(readHexaValues);
            this.handoffPositionSensor1.updateValue(readHexaValues);
            this.storagePositionSensor0.updateValue(readHexaValues);
            this.storagePositionSensor1.updateValue(readHexaValues);

            //TODO compute a state 
            this.atStorage = storagePositionSensor0.getDigitalValue() == 1
                    && storagePositionSensor1.getDigitalValue() == 1;
            this.atHandoff = handoffPositionSensor0.getDigitalValue() == 1
                    && handoffPositionSensor1.getDigitalValue() == 1;
            this.storageSensorsInError = storagePositionSensor0.getDigitalValue() != storagePositionSensor1.getDigitalValue();
            this.handoffSensorsInError = handoffPositionSensor0.getDigitalValue() != handoffPositionSensor1.getDigitalValue();
            //TODO check if motion is allowed when positions sensors are in error.

        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();

        }

    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        carrierController.checkFault();

        if (action.equals(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_HANDOFF)) {
            //TODO : check that if the filter is in the autochanger,the autochanger must hold the filter
            //and the loader clamp must be opened.
            carrierController.enable();
            carrierController.changeMode(EposMode.PROFILE_POSITION);
            carrierController.writeTargetPosition(this.handoffPosition);
            carrierController.writeControlWord("3F");

        } else if (action.equals(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_STORAGE)) {
            carrierController.enable();
            carrierController.changeMode(EposMode.PROFILE_POSITION);
            carrierController.writeTargetPosition(this.storagePosition);
            carrierController.writeControlWord("3F");

        } else if (action.equals(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION)) {
            carrierController.enable();
            carrierController.changeMode(EposMode.PROFILE_POSITION);
            carrierController.writeTargetPosition(this.absoluteTargetPosition);
            carrierController.writeControlWord("3F");
        }
    }

    @Override
    public void abortAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(getName() + " is ABORTING action " + action.toString()
                + " within delay " + delay);
       
        FCSLOG.info("Current Command: "+getSubsystem().getCurrentAction()+" "+getSubsystem().getState());

        this.carrierController.off();
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(getName() + " is STOPPING action " + action.toString()
                + " within delay " + delay);
        
        FCSLOG.info("Current Command: "+getSubsystem().getCurrentAction()+" "+getSubsystem().getState());

        //doesn't work : we can't do a quickStop when in PROFILE_POSITION
        //this.carrierController.quickStop();
        this.carrierController.off();

    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action)  {
        //because we don't want to let the controller on power
        this.carrierController.off();
        FCSLOG.info(getName() + ":" + action.toString() + " completed - doing postAction.");

        switch (action) {

            case MOVE_LOADERCARRIER_TO_STORAGE:
                if (!this.isAtStoragePosition()) {
                    throw new FailedCommandException(getName()
                            + ": check with sensors: storage sensors don't confirm carrier position.");
                }
                break;

            case MOVE_LOADERCARRIER_TO_HANDOFF:
                if (!this.isAtHandoffPosition()) {
                    throw new FailedCommandException(getName()
                            + ": check with sensors: handoff sensors don't confirm carrier position.");
                }
                break;
                
            case MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION:
                this.absoluteTargetPosition = 0;
                break;    
                
            default:
                assert false;
        }

    }

    /**
     * Command for ENGINEERING mode. Not used during INITIALIZATION.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Check if hardware is ready to be started.")
    public void initializeHardware()  {

        if (!carrierController.isInitialized()) {
            throw new FcsHardwareException(getName()
                    + ": carrierController has to be initialized.");
        }
        updateStateWithSensors();
        try {
            updatePosition();
        } catch (SDORequestException ex) {
            String msg = getName() + ": couldn't update position";
            FCSLOG.error(msg);
            throw new FcsHardwareException(getName(), ex);
        }
        if (this.isAtHandoffPosition()
                && !(this.position == this.handoffPosition || this.position > this.handoffPosition)) {
            throw new FcsHardwareException(getName()
                    + ": handoff sensors don't confirm position read on carrierController.");
        }
        if (this.isAtStoragePosition() && !(this.position == this.storagePosition)) {
            throw new FcsHardwareException(getName()
                    + ": storage sensors don't confirm position read on carrierController.");
        }
        this.initialized = true;
        loader.updateFCSState();
    }
    
    /**
     * Return a printed list of hardware with initialization information.
     * For debug purpose.
     * @return 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware with initialization information.")    
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(getName());
        if (this.isInitialized()) {
            sb.append(" is INITIALIZED.");
        } else {
            sb.append(" is NOT INITIALIZED.");
        }
        return sb.toString();
    }

    /**
     * Returns the object to be published on the STATUS bus by the LoaderCarrierModule.
     * @return 
     */
    public StatusDataPublishedByLoaderCarrier getStatusData() {
        return createStatusDataPublishedByLoaderCarrier();
    }
    
    /**
     * Creates an bject to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByLoaderCarrier createStatusDataPublishedByLoaderCarrier() {
        StatusDataPublishedByLoaderCarrier status
                = new StatusDataPublishedByLoaderCarrier(getName(),
                        position, ssiPosition,
                        speed, current,
                        storagePositionSensor0.getDigitalValue(),
                        storagePositionSensor1.getDigitalValue(),
                        handoffPositionSensor0.getDigitalValue(),
                        handoffPositionSensor1.getDigitalValue(),
                        storageSensorsInError,
                        handoffSensorsInError,
                        controllerInFault);
        status.setLimitSwitchDownInError(limitSwitchDownInError);
        status.setLimitSwitchUpInError(limitSwitchUpInError);
        return status;
    }


   
    
    /**
     * What to do when we received an Emergency message from controller with an errorCode; 
     * @param msg
     */
    @Override
    public void processEmergencyMessage(EmergencyMessage msg) {
        MovedByEPOSController.super.processEmergencyMessage(msg);
        if ("ff06".equalsIgnoreCase(msg.getDeviceErrorCode())) {
            //Negative Limit Switch Error
            this.limitSwitchDownInError = true;
        } else if ("ff07".equals(msg.getDeviceErrorCode())) {
            this.limitSwitchUpInError = true;
        }
        this.publishData();
    }
    
    /**
     * 
     */
    @Override
    public void processFaultReset() {
        MovedByEPOSController.super.processFaultReset();
        this.limitSwitchDownInError = false;
        this.limitSwitchUpInError = false;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        StatusDataPublishedByLoaderCarrier status = this.getStatusData();
        KeyValueData kvd = new KeyValueData("loaderCarrier", status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }


}
