package org.lsst.ccs.subsystems.fcs;

import java.util.Map;
import org.lsst.ccs.UsesSubsystem;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This interface is to factorise code between main Module for carousel in
 * standalone mode and main Module for whole fcs. In both case, during
 * initialization of the subsystem, we want to update the carousel socket with
 * the information on the location of the filters stored in Configuration
 * System. In the configuration system, for each filter on the Carousel, we
 * store the name of the socket this filter is on (socketName).
 *
 * @author virieux
 */
public interface FilterLocator extends UsesSubsystem {

    String FILTER_CONFIG_CATEGORY = "filterLocation";

    /**
     * Return the carousel Module of this filter locater.
     *
     * @return
     */
    CarouselModule getCarousel();

    /**
     * Return the filter manager of this filter locater.
     *
     * @return
     */
    FilterManager getFilterManager();

    /**
     * This method updates the filters location in the Carousel sockets during
     * the initialization phase.
     */
    @Command(type = Command.CommandType.QUERY,
            level = Command.NORMAL,
            description = "Check which filters are on Carousel and on which socket and update the sockets.",
            alias = "locateFilters")
    default void checkFiltersLocation() {
        FCSLOG.info(getName() + " updating filters location on carousel .....");
        Map<String, Filter> filtersMap = getFilterManager().getFiltersMapByName();
        for (Map.Entry<String, Filter> entry : filtersMap.entrySet()) {
            Filter filter = entry.getValue();
            
            filter.initializeFilterLocation();
            if (filter.isOnCarousel()) {
                FCSLOG.info(filter.getName() + " location: "+ filter.getFilterLocation()+ " on "+ filter.getSocketName());
                getCarousel().getSocketByName(filter.getSocketName()).putFilterOnSocket(filter);
            } else {
                filter.publishData();
                FCSLOG.info(filter.getName() + " location: "+ filter.getFilterLocation());
            }
        }

        //TODO updateStateWithSensors and check that when a filter is supposed to be on the Carousel
        //the filter presence sensor confirms the information.
    }

    /**
     * This method save the location of the filters in the filesystem or
     * Configuration System. This command has to be executed before shutting
     * down the subsystem to store filters location information. The goal is to
     * be able to restart next time with the updated information.
     *
     */
    @Command(type = Command.CommandType.QUERY,
            level = Command.NORMAL,
            description = "Save filter location in filesystem (in ~ccs/etc/) or Configuration System.")
    default void saveFilterLocation() {
        try {
            this.getSubsystem().getSubsystemConfigurationEnvironment().saveChangesForCategories(FILTER_CONFIG_CATEGORY);
        } catch (Exception ex) {
            FCSLOG.error("Unable to save filter location.", ex);
        }
    }

}
