
package org.lsst.ccs.subsystems.fcs;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;

/**
 * This is a Model for the system which identifies filters on the autochanger trucks.
 * On the autochanger trucks, we have hall effect sensors to identify filters.
 * 3 sensors are on the truck X- and 3 sensors are on the truck X+.
 * Each sensors gives a digital signal depending if there is a magnet in front of the sensors.
 * Following the magnet combination, we can define 64 ID numbers (from 0 to 63).
 * The location are named ID0, ID1 and ID2 on the X- side, 
 * and they are named ID3, ID4, and ID5 on the X+ side.
 * The ID0 is the Least Significant Bit (LSB)
 * The ID5 is the Most Significant Bit (MSB)
 * 
 * cf document LCA-13492

 * @author virieux
 */
public class FilterIdentificator extends Module {
    
    @ConfigurationParameter(isFinal=true)
    private String plutoGatewayName;
    
    private static final int SENSORS_NUMBER = 6;
    
    /**
     * An array of 6 filter ID numeric sensors.
     */
    private final ArrayList<CanOpenNumericSensor> filterIDSensors = new ArrayList<>();
    
    /**
     * An array of 6 filter ID numeric sensors values.
     */
    private final int[] sensorsValues;
    private final char[] chars;
    
    private PlutoGatewayInterface plutoGateway;
    
    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();

    /**
     * Build a FilterIdentificator with its 6 hall effect sensors.
     * @param gwName
     */
    public FilterIdentificator (String gwName) { 
        this.plutoGatewayName = gwName;
        sensorsValues = new int[SENSORS_NUMBER];
        chars = new char[SENSORS_NUMBER];
        for (int i=0;i<SENSORS_NUMBER;i++) {
            sensorsValues[i] = 0;
            chars[i] = '0';
        }
    }

    /**
     * Return the list of filterID sensors.
     * @return 
     */
    public List<CanOpenNumericSensor> getFilterIDSensors() {
        return (List<CanOpenNumericSensor>) filterIDSensors.clone();
    }

    @Override
    public void initModule() {
        this.plutoGateway = (PlutoGatewayInterface) getComponentLookup().getComponentByName(plutoGatewayName);
        Map<String, CanOpenNumericSensor> cosMap = getSubsystem().
                getComponentLookup().getChildren(getName(), CanOpenNumericSensor.class);
        filterIDSensors.addAll(cosMap.values());
    }
    
    /**
     * Read the 6 hall effect sensors, compute the number given by the magnets and returns it.
     * 
     * @return filterId
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
        description = "Read hall effect sensors, compute filter number and return it.")
    public int readFilterIdentifier() {
        updateSensorsValues();
        return computeFilterID();
    }
    

    
    /**
     * Read filter ID sensors and update values.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
        description = "Read filter ID sensors and update values.")
    public void updateSensorsValues() {
        lock.lock();
        try {
            this.plutoGateway.updateValues();
            updateSensorsValues(this.plutoGateway.getHexaValues());
            
        }   finally {

            stateUpdated.signal();
            lock.unlock();
        }
    }
    
    /**
     * Update sensors values from hexavalues read from gateway.
     * @param readHexaValues 
     */
    public void updateSensorsValues(String[] readHexaValues) {
        FCSLOG.finest(Arrays.toString(readHexaValues));
        for (int i=0; i<SENSORS_NUMBER;i++) {
            filterIDSensors.get(i).updateValue(readHexaValues);
            sensorsValues[i] = filterIDSensors.get(i).getDigitalValue();
        }
        this.update(sensorsValues);
    }
    
    /**
     * 
     * @param values 
     */
    private void update(int[] values) {
        for (int i=0; i<SENSORS_NUMBER; i++) {
            if (values[i] == 1) {
                chars[SENSORS_NUMBER-1 - i] = '1';
            } else {
                chars[SENSORS_NUMBER-1 - i] = '0';
            }
        }
    }
    
    
    private int computeFilterID() {
        return Integer.parseInt(new String(chars),2);
    }
    
    @Override
    public String toString() {
        return String.valueOf(chars);
    }

    
    

}
