package org.lsst.ccs.subsystems.fcs;

import static java.lang.Math.abs;
import java.util.Map;
import java.util.TreeMap;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.TreeWalkerDiag;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FCS_NAME_FOR_MCM;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.ModuleState;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import static org.lsst.ccs.subsystems.fcs.utils.FcsUtils.addAngle;

/**
 * This is a representation of the hardware of the carousel. It receives
 * commands from the FCSMAinModule and send back an acknowledge. It publishes
 * data on the status bus. In engineering mode it can receive commands from the
 * engineering console. 
 * 
 *
 * @author virieux
 *
 */
public class CarouselModule extends MobileItemModule implements HardwareController, FilterHolder {

    private static final long serialVersionUID = -2376279469784152348L;
    
    @ConfigurationParameter
    private int rotationTimeout = 20000;

    private int position;
    private int absoluteTargetPosition;
    private int relativeTargetPosition;
    
    private final int speed;
    private int current;

    /*This actuator opens the clamp Xminus when the carousel is halted at standby filterPosition.*/
    private EPOSController clampXminusController;

    /*This actuator opens the clamp Xplus when the carousel is halted at standby filterPosition.*/
    private EPOSController clampXplusController;

    /*Controls carousel rotation.*/
    private EPOSController carouselController;

    /*CANOpen devices to read the values of the clamps sensors.*/
    private PieceOfHardware hyttc580;

    /*CANOpen devices to read the values of the brakes sensors and temperatures.*/
    protected PieceOfHardware ai814;
    private PieceOfHardware pt100;

    /*To be able to know if the autochanger holds a filter. */
    private FilterHolder autochanger;

    /**
     * A map to store the sockets by their names. The key of this map is the
     * socket name.
     *
     */
    protected Map<String, CarouselSocket> socketsMap;

    private BridgeToHardware bridge;

    private volatile ModuleState state = ModuleState.HALTED;
    private boolean initialized = false;

    /**
     * A Constructor with a variable numbers of sockets.
     * In single-filter-test carousel has only 1 socket.
     * In Filter Exchanger final product and prototype, carousel has 5 sockets.
     */
    public CarouselModule() {
        super(5000);
        this.position = 0;
        this.current = 0;
        this.speed = 0;
    }

    /**
     * Set a new rotationTimeout
     * @param rotationTimeout 
     */
    public void setRotationTimeout(int rotationTimeout) {
        if (rotationTimeout < 0 || rotationTimeout > 500000) {
            throw new IllegalArgumentException("rotationTimeout must be between 0 and 500000" );
        }
        this.rotationTimeout = rotationTimeout;
    }
    
    /**
     * 
     * @return true if CANopen devices are booted and initialized and homing has been done.
     */
    @Command(description="Return true if CANopen devices are booted and initialized and homing has been done.")
    boolean isInitialized() {
        //TODO test also if homing has been done.
        return this.isCANDevicesReady();
    }

    /**
     * Return a CarouselSocket which name is given as parameter.
     *
     * @param socketName
     * @return
     */
    public CarouselSocket getSocketByName(String socketName) {
        if (socketsMap.containsKey(socketName)) {
            return socketsMap.get(socketName);
        } else {
            throw new IllegalArgumentException(getName() + ": no such name for socket:" + socketName);
        }
    }

    /**
     * return carousel position.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return carousel position.", alias="printPosition")
    public int getPosition() {
        return position;
    }

    public synchronized ModuleState getState() {
        return state;
    }

    public synchronized void setState(ModuleState state) {
        this.state = state;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXminusController() {
        return clampXminusController;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXplusController() {
        return clampXplusController;
    }





    /**
     * This method returns the clampX- which is at standby filterPosition. It
     * can returns null if there is no socketAtStandby halted at standby
     * filterPosition.
     *
     * @return
     */
    public CarouselClampModule getClampXminus() {
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXminus();
        }
    }

    /**
     * This method returns the clampX+ which is at standby filterPosition. It
     * can returns null if there is no socketAtStandby halted at standby
     * filterPosition.
     *
     * @return
     */
    public CarouselClampModule getClampXplus() {
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXplus();
        }
    }

    /**
     * Used to publish on the STATUS bus for the GUI. Returns
     *
     * @return true if a socket is HALTED at STANDBY filterPosition, false
     * otherwise.
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if a socket is HALTED at STANDBY position, false otherwise.")    
    public synchronized boolean isAtStandby() {
        if (this.isRotating()) {
            return false;
        } else {
            for (CarouselSocket socket : socketsMap.values()) {
                if (socket.isAtStandby()) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Return the socket HALTED at STANDBY filterPosition if there is one.
     * Otherwise return null.
     *
     * @return
     */
    public synchronized CarouselSocket getSocketAtStandby() {
        if (this.isRotating()) {
            return null;
        } else {
            for (CarouselSocket socket : socketsMap.values()) {
                if (socket.isAtStandby()) {
                    return socket;
                }
            }
        }
        return null;
    }

    /**
     * This methods return the filter which is in the socketAtStandby at standby
     * filterPosition. When there is no filter at standby filterPosition it
     * returns null.
     *
     * @return filter at standby filterPosition or null if there is no filter at
     * standby filterPosition.
     */
    public synchronized Filter getFilterAtStandby() {

        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getFilter();
        }
    }

    /**
     * Returns name of filter which is in the socket at STANDBY position or
     * NO_FILTER if there is no filter at STANDBY.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns name of filter at STANDBY position  or"
                    + " NO_FILTER if there is no filter at STANDBY.")    
    public synchronized String getFilterAtStandbyName() {

        Filter filterAtStandby = getFilterAtStandby();
        if (filterAtStandby == null) {
            return NO_FILTER;

        } else {
            return filterAtStandby.getName();
        }

    }

    
    /**
     * This methods returns true if the autochanger is holding a filter at
     * STANDBY position.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if Autochanger is holding filter at STANDBY.")
    public boolean isAutochangerHoldingFilter()  {
        /**
         * To be done when a switch is connected : read the switch sensor and 
         * update a boolean of the class like : autochangerHoldingFilter.
         */
        autochanger.updateStateWithSensors();
        return autochanger.isHoldingFilter();
    }

    @Override
    public void initModule() {
        super.initModule();
        
        ComponentLookup lookup = getSubsystem().getComponentLookup();
        
        FCSLOG.info("[CarouselModule] Initializing the carousel module ");
        setState(ModuleState.HALTED);
        this.bridge = (BridgeToHardware) lookup.getComponentByName("bridge");
        this.clampXminusController = (EPOSController) lookup.getComponentByName("clampXminusController");
        this.clampXplusController = (EPOSController) lookup.getComponentByName("clampXplusController");
        this.carouselController = (EPOSController) lookup.getComponentByName("carouselController");
        this.hyttc580 = (PieceOfHardware) lookup.getComponentByName("hyttc580");
        this.ai814 = (PieceOfHardware) lookup.getComponentByName("ai814");
        this.pt100 = (PieceOfHardware) lookup.getComponentByName("pt100");

        // Adding sockets
        Map<String, CarouselSocket> sMap = lookup.getChildren(getName(), CarouselSocket.class);
        
        socketsMap = new TreeMap<>();
        for (Map.Entry<String, CarouselSocket> entry : sMap.entrySet()) {
            FCSLOG.info("Add socket to sockets Map:" + entry.getValue().toString());
            socketsMap.put(entry.getKey(), entry.getValue());
        }

        if (lookup.getComponentByName("autochanger") instanceof FilterHolder) {
            autochanger = (FilterHolder) lookup.getComponentByName("autochanger");
        } else {
            final String MSG = getName() + " ==> autochanger doesn't implements FilterHolder -"
                    + " Please fix groovy description file or CarouselModule implementation.";
            FCSLOG.error(MSG);
            throw new IllegalArgumentException(MSG);
        }

    }

    
    /**
     * Executed during Initialization phase by the framework.
     * @return
     * @throws HardwareException 
     */
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        try {
            this.carouselController.initializeAndCheckHardware();
            this.ai814.initializeAndCheckHardware();
            this.hyttc580.initializeAndCheckHardware();
            this.pt100.initializeAndCheckHardware();
            this.initialized = true;
            return TreeWalkerDiag.GO;
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true,ex);
        }
    }
    
    /**
     * CheckStarted is executed by completeInitialization when a first start has
     * failed. Because end user could have changed many things, we have to check
     * again that all is correct.
     *
     * @throws org.lsst.ccs.HardwareException
     *
     */
    @Override
    public void checkStarted() throws HardwareException {

        FCSLOG.info(getName() + " BEGIN checkStarted");

        //check that all hardware is booted and identified.
        bridge.getTcpProxy().checkNewHardware();

        try {
            this.ai814.initializeAndCheckHardware();
            updateClampsStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }
        
        FCSLOG.info(getName() + " END checkStarted");
    }

    /**
     * This method is executed during a shutdown of the subystem to prevent a
     * shutdown when some hardware is moving. Nothing to do here because all the
     * moving hardware of the loader is represented by the class MobilItemModule
     * where the method checkStopped is implemented.
     *
     * @throws org.lsst.ccs.HardwareException
     *
     */
    @Override
    public void checkStopped() throws HardwareException {
        //done already in my children which are MobileItemModule and my other children don't need that.
    }
    

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData("carousel",
                createStatusDataPublishedByCarousel()));
    }

    /**
     * Create an object StatusDataPublishedByCarousel to be published on the
     * STATUS bus.
     *
     * @return status
     */
    //TODO refactorize this method because now FilterManager publishes on the bueses the filters state.
    public StatusDataPublishedByCarousel createStatusDataPublishedByCarousel() {
        StatusDataPublishedByCarousel status = new StatusDataPublishedByCarousel();
        status.setName(getName());
        status.setRotationInDegrees(position);
        status.setLocked(this.isLocked());
        status.setCurrent(current);
        status.setSpeed(speed);

        if (CarouselModule.this.isAtStandby()) {
            status.setAtStandby(true);
            //FCSLOG.debug(getName() + " Filter at STANDBY=" + this.getFilterAtStandbyName());
            status.setFilterAtStandbyName(this.getFilterAtStandbyName());
            status.setSocketAtStandbyName(this.getSocketAtStandby().getName());
        } else {
            status.setAtStandby(false);
        }
        /*List of filters on carousel*/
        Map<String, String> mapSocketFilter = new TreeMap<>();
        for (Map.Entry<String, CarouselSocket> entry : socketsMap.entrySet()) {
            CarouselSocket socket = entry.getValue();
            if (socket.getFilter() == null) {
                mapSocketFilter.put(socket.getName(), NO_FILTER);
            } else {
                mapSocketFilter.put(socket.getName(), socket.getFilter().getName());
            }
        }
        status.setFiltersOnCamera(mapSocketFilter);
        return status;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("\n Numbers of sockets: ");
        sb.append(socketsMap.size());
        for (CarouselSocket socket : socketsMap.values()) {
            sb.append("\n Socket name ");
            sb.append(String.valueOf(socket.getName()));
            sb.append(" / ");
            sb.append(String.valueOf(socket));
        }
        return sb.toString();
    }


    /**
     * This method let us know if the carousel is ready to receive a filter at
     * standby filterPosition : - the carousel must not rotate - an empty
     * socketAtStandby is at standby filterPosition.
     *
     * @return true if the filterPosition of the carousel matches the
     * filterPosition when one of its sockets is at standby filterPosition and
     * this socketAtStandby is empty. false
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(level = Command.ENGINEERING1, type = Command.CommandType.QUERY,
            description = "Returns true if Carousel is stopped and no filter is at STANDBY position")
    public boolean isReadyToGrabAFilterAtStandby()  {

        if (this.isRotating()) {
            return false;
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return false;
        }
        return socketAtStandby.isEmpty() && socketAtStandby.isReadyToClamp();
    }

    /**
     *
     * @return true if a filter is clamped at STANDBY position and
     * carousel is stopped
     */
    @Command(level = Command.ENGINEERING1, type = Command.CommandType.QUERY,
            description = "Returns true if a filter is clamped at STANDBY position")
    public boolean isHoldingFilterAtStandby()  {
        if (this.isRotating()) {
            return false;
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null || socketAtStandby.isEmpty()) {
            return false;
        }
        return socketAtStandby.isClampedOnFilter();
    }

    /**
     * Return 
     * @return 
     */
    @Command(level = Command.ENGINEERING1, type = Command.CommandType.QUERY,
            description = "Returns true if carousel is rotating.")
    public boolean isRotating() {
        return isMoving();
    }

    /**
     *
     * @return true if the brake is released
     */
    @Command(level = Command.ENGINEERING1, type = Command.CommandType.QUERY,
            description = "Returns true if the brake is released")
    public boolean isAbleToMove() {
        return true;
    }

    public boolean isLocked() {
        return true;
    }

    @Command(level = Command.ENGINEERING1, type = Command.CommandType.ACTION,
            description = "Engage the carousel brake to stop the rotation")
    public String engageBrake() {
        return getName() + " locked";

    }

    @Command(level = Command.ENGINEERING1, type = Command.CommandType.ACTION,
            description = "Release the carousel brake")
    public String releaseBrake() {
        return getName() + " unlocked";
    }

    /**
     * Updates the filterPosition of the carousel in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update carousel position in reading controller.")
    public void updatePosition()  {
        try {
            this.position = carouselController.readPosition();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> ERROR IN READING CONTROLLER:",ex);
        }
        this.publishData();
    }

    /**
     * Updates the field current in reading the CPU of the controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update field current in reading controller.")
    public void updateCurrent()  {
        try {
            this.current = carouselController.readCurrent();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> ERROR IN READING CONTROLLER:",ex);
        }
        this.publishData();
    }

    /**
     * Read the clamp sensors one by one with RSDO command
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromSDO()  {

        for (CarouselSocket socket: socketsMap.values()) {
            socket.updateClampsStateWithSensors();
        }
    }

    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one
     * time. 
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromPDO()  {

        PDOStorage pdoStore = this.bridge.readPDOs();
        FCSLOG.finest(getName() + ":pdoStore=" + pdoStore.toString());
        /* perhaps do it only for socket at STANDBY*/
        for (CarouselSocket socket : socketsMap.values()) {
            socket.updateClampsStateWithSensors(pdoStore);
        }
    }

    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one
     * time. Tested with success on single-filter-test in April 2013
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read clamps sensors and update clamps state")
    public void updateClampsStateWithSensors()  {
        
        updateClampsStateWithSensorsFromPDO();
    }

    /**
     * What has to be done for each tick of the timer. We have to publish on the
     * status bus to refresh the GUI.
     */
    @Override
    public void tick() {
        this.publishData();
    }

    /**
     * This method has no action on hardware. This method updates the
     * socketAtStandby at standby filterPosition when a filter is put on the
     * carousel at standby filterPosition. A filter must be detected in the
     * socketAtStandby at standby filterPosition (socketAtStandby not empty).
     *
     * @param filter
     * @throws RejectedCommandException
     */
    private synchronized void putFilterOnCarousel(Filter filter) {

        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new RejectedCommandException(getName() + ": there is no socket at standby "
                    + "position to put a filter in.");
        }

        socketAtStandby.putFilterOnSocket(filter);
        this.publishData();
    }


    //TODO throw an exception if filter is not on carousel.
    private int getStandbyPositionForFilter(Filter filter) {
        CarouselSocket socket = socketsMap.get(filter.getSocketName());
        if (socket == null) {
            throw new IllegalArgumentException(getName() + " no socket name for filter " + filter.getName());
        } else {
            return socket.getStandbyPosition();
        }
    }


        

    /**
     * Grab a filter at STANDBY position.
     * 
     * @param filterName
     * @return
     * @throws FcsHardwareException 
     */
    public String grabFilterAtStandby(String filterName)  {

        FcsMainModule fcsMainModule = (FcsMainModule) getComponentLookup()
                .getComponentByName("main");
        fcsMainModule.checkFilterName(filterName);

        Filter filterToGrabbe = fcsMainModule.getFilterByName(filterName);

        // the entered filter getName() is correct
        FCSLOG.info("Filter to move : " + filterName);
        FCSLOG.info("Filter location : " + filterToGrabbe.getFilterLocation());
        return getName() + grabFilterAtStandby(filterToGrabbe);
    }

    /**
     * This method has to be executed when a filter has just been moved on the
     * carousel at standby filterPosition by the autochanger. 
     * It updates the carousel socket at standby or throw an exception.
     *
     * This method tests if the clamps are Locked on a filter at standby
     * filterPosition The clamps Locks automaticaly, and the filter presence
     * sensor tells us if the filter is here or not. If the clamps are Locked on
     * a filter at standby filterPosition, this method updates the
     * socketAtStandby at standby filterPosition : it puts the filter on the
     * socketAtStandby.
     *
     * @param filter
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public synchronized String grabFilterAtStandby(Filter filter)  {
        if (filter == null) {
            throw new IllegalArgumentException(getName() + ": grabbeFilterAtStandby must receive a non null argument as a filter");
        }
        if (this.isRotating()) {
            throw new RejectedCommandException(getName() + ": grabbeFilterAtStandby can't grabbe a filter while rotating");
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new RejectedCommandException(getName() + ": grabbeFilterAtStandby can't be executed because there is no socket at standby position.");
        }

        socketAtStandby.updateClampsStateWithSensors();

        if (socketAtStandby.isEmpty()) {
            throw new RejectedCommandException(getName() + ": grabbeFilterAtStandby can't be executed because no filter is detected in socket at standby position.");
        }

        if (socketAtStandby.isClampedOnFilter()) {
            FCSLOG.info(getName() + ": grabbing " + filter.getName() + " at standby position.");
            putFilterOnCarousel(filter);

            FCSLOG.info(getName() + ": " + filter.getName() + " is grabbed on carousel.");
            return getName() + ": " + filter.getName() + " is grabbed at standby position";
        } else {
            throw new RejectedCommandException(getName() + ": grabbeFilterAtStandby can't be executed because the clamps are not CLAMPED ON FILTER.");
        }

    }

    /**
     * This unlock clamps at STANDBY.
     * @param filterName
     * @throws FcsHardwareException 
     */
    public void ungrabFilterAtStandby(String filterName)  {

        if (filterName == null) {
            throw new IllegalArgumentException("Filter to ungrabb should not be null");
        }

        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new RejectedCommandException(getName() + ":A socket has to be halted at standby position for this operation.");
        }

        if (!socketAtStandby.isClampedOnFilter()) {
            throw new RejectedCommandException(getName() + ": can't ungrabb filter which is already unclamped.");
        }

        FCSLOG.info(getName() + ": ungrabbing " + filterName + " at standby position.");

        //TODO test if the filter is held by the autochanger trucks.
        // The clamps can be opened.
        this.unlockClamps();
        FCSLOG.info(getName() + ": socket at standby state : " + socketAtStandby.toString());

        FCSLOG.info(getName() + ": " + filterName + " is ungrabbed from carousel.");
    }

    /**
     * Release clamps at standby position to get ready to clamp again. 
     */
    @Command(type = Command.CommandType.ACTION,level = Command.ENGINEERING1, 
            description = "Release clamps at standby position to get ready to clamp again")
    public void releaseClamps()  {

        if (getSocketAtStandby() == null) {
            throw new RejectedCommandException(getName()
                    + ":Can't release clamps when no socket is halted at standby position.");
        } else {
            getSocketAtStandby().releaseClamps();
        }
    }

    /**
     * Unlock the clamps at STANDBY.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws FailedCommandException
     * @throws RejectedCommandException
     */
    @Command(level = Command.ENGINEERING3, description = "Unlock the clamps at STANDBY.",
        type = Command.CommandType.ACTION)
    public void unlockClamps()  {

        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            throw new RejectedCommandException("Can't unlock clamps while a socket is not halted at standby position.");
        } else {
            FCSLOG.info("Unlocking clamps at standby.");
            socketAtStandby.unlockClamps();
        }
    }


    /**
     * Return true if the filter given as argument is at STANDBY position.
     * 
     * @param filter
     * @return 
     */
    public boolean isAtStandby(Filter filter) {
        return this.getPosition() == this.getStandbyPositionForFilter(filter);
    }

    public void releaseClampsContact() {
        //TODO;
    }

    public void engageClampsContact() {
        //TODO ;
    }

    @Override
    public boolean isCANDevicesReady() {
        return this.bridge.isCANDevicesReady() && this.initialized;
    }
    
        
    /*******************************************************************/
    /************** ROTATION COMMANDS **********************************/
    /*******************************************************************/
    
    /**
     * Check if carousel rotation is permitted.
     * 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException (RuntimeException)
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if carousel rotation is permitted.")
    public void checkConditionsForRotation()  {
        String message;
        if (!initialized) throw new FcsHardwareException("Carousel hardware is not initialized. Can't rotate.");
        
        if (isRotating()) {
            message = "Carousel is rotating, stop it before sending a new rotate command.";
            FCSLOG.error(message);
            throw new RejectedCommandException(message);
        }

        if (!this.isAbleToMove()) {
            message = "Carousel is unable to move. Check motor brake.";
            FCSLOG.error(message);
            throw new RejectedCommandException(message);
        }
        
        updateClampsStateWithSensors();
        if (this.isAtStandby() && this.getSocketAtStandby().isUnclampedOnFilter()) {
            message = "Filter at STANDBY position is not held by clamps. Can't rotate carousel.";
            FCSLOG.error(message);
            throw new RejectedCommandException(message);
        }
        
        if (!autochanger.isAtHandoff()) {
            throw new RejectedCommandException(name + " can't rotate if autochnager is not at HANDOFF position.");
        }
    }

    /**
     * This command rotates carousel until the absolute target position is reached.
     * It doesn't compute the shortest way.
     *
     * @param angle
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Rotate carousel to an absolute angle.")
    public void rotateToAbsoluteAngle(@Argument(name = "angle",
            description = "Angle to reach in degrees") int angle)  {
        checkConditionsForRotation();
        
        if (angle < 0 || angle > 359) {
            throw new IllegalArgumentException("angle must be between 0 and 359");
        }
        
        this.absoluteTargetPosition = angle;
        
        //Release The clamps contact before the rotation
        releaseClampsContact();
        
        
        //rotate the carousel
        this.executeAction(MobileItemAction.ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION, rotationTimeout);
        this.publishData();
    }

    /**
     * A command to rotate carousel. 
     * Angle of rotation is given as argument and is relative to actual position.
     * If initial position is 10, rotateToRelativeAngle(20) will rotate carousel until position is 30.
     * If initial position is 10, rotateToRelativeAngle(-20) will rotate carousel until position is 350.
     * 
     * carousel can rotate in both ways (clockwise or not). 
     * This command computes the shortest rotation to be done to reach the final goal.
     *
     * @param angle
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Rotate carousel for an angle.")
    public void rotateToRelativeAngle(@Argument(name = "angle",
            description = "Relative angle to reach in degrees. Can be positive or negative.") int angle) {
        
        checkConditionsForRotation();
        
        if (abs(angle) > 360) {
            throw new IllegalArgumentException(angle + ": illegal value for an angle. Must be between -360 and 360");
        }
        FCSLOG.info(getName() + " is at position: "+position+ "; about to rotate to relative position: " + angle);
        absoluteTargetPosition = addAngle(position,angle);

        relativeTargetPosition = FcsUtils.computeRotationShortestWay(angle);
        
        FCSLOG.fine(getName() + " is at position: "+position+ "; about to rotate - target position: " + absoluteTargetPosition
            + " relative target position: " + relativeTargetPosition);
        
        //Release The clamps contact before the rotation
        releaseClampsContact();
        //rotate the carousel
        this.executeAction(MobileItemAction.ROTATE_CAROUSEL_TO_RELATIVE_POSITION, rotationTimeout);
        this.publishData();
    }
    
    
    /**
     * Rotate carousel to move a socket which name is given as argument to STANDBY position.
     * This methods computes the shortest way to go to STANDBY position.
     * @param socketName 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move a socket which name is given as argument to STANDBY position.",
            alias="moveSocketToStandby")
    public void rotateSocketToStandby(String socketName)  {
        
        FcsUtils.checkSocketName(socketName);

        CarouselSocket socketToMove = this.socketsMap.get(socketName);

        int requiredPosition = socketToMove.getStandbyPosition();
        
        String msgWhenCompleted = getName() + ":" + socketName + " is at STANDBY position on carousel.";

        if (position != requiredPosition) {
            FCSLOG.info(getName() + " is at position: "+position+ "; about to rotate to position: " + requiredPosition);
            int rotationAngle = requiredPosition - position;
            FCSLOG.info(getName() + " rotation angle= "+rotationAngle);
            /* compute shortest way*/
            relativeTargetPosition = FcsUtils.computeRotationShortestWay(rotationAngle);
            rotateToRelativeAngle(relativeTargetPosition);
        }
        FCSLOG.info(msgWhenCompleted);
    }

    /**
     * Move an empty socket to STANDBY position.
     * The socket moved is the first empty one in the socketsMap.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void rotateEmptySocketAtStandby()  {
        for (Map.Entry<String, CarouselSocket> entry : socketsMap.entrySet()) {
            CarouselSocket socket = entry.getValue();
            if (socket.isEmpty()) {
                rotateSocketToStandby(entry.getKey());
                return;
            }
        }
        FCSLOG.error("no empty socket on carousel - nothing to do.");
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION:
                return this.position == this.absoluteTargetPosition;

            case ROTATE_CAROUSEL_TO_RELATIVE_POSITION:
                return this.position == this.absoluteTargetPosition;
                
            default :
                assert false: action;
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        try {
            carouselController.checkFault();
            this.position = this.carouselController.readPosition();
            FCSLOG.debug(getName() + " position=" + this.position);
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> SDO ERROR IN READING CONTROLLER:",ex);
        }
    }

    /**
     * Starts action ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION or ROTATE_CAROUSEL_TO_RELATIVE_POSITION.
     * This :
     * - enable controller,
     * - make controller go to required position previously written in field absoluteTargetPosition
     * or relativeTargetPosition,
     * - writeControlWord "3F" or "7F" on controller depending if we want to go to a relative position 
     * or to an absolute position.
     * @param action 
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        
        this.getSubsystem().updateAgentState(FCS_NAME_FOR_MCM, FcsEnumerations.FilterState.valueOf("ROTATING"), 
                FcsEnumerations.FilterReadinessState.NOT_READY);
        switch (action) {
            case ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION:
                carouselController.enable();
                carouselController.writeTargetPosition(this.absoluteTargetPosition);
                carouselController.writeControlWord("3F");
                break;

            case ROTATE_CAROUSEL_TO_RELATIVE_POSITION:
                carouselController.enable();
                carouselController.writeTargetPosition(this.relativeTargetPosition);
                carouselController.writeControlWord("7F");
                break;
                
            default:
                assert false : action;
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay)  {
        FCSLOG.debug(getName() + " is ABORTING action " + action.toString()+ " within delay " + delay);
        this.carouselController.off();
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay)  {
        throw new UnsupportedOperationException("Not supported yet."); 
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action)  {
        //engage the clamps contact when rotation is completed
        engageClampsContact();
        //lock the carousel to hold it in filterPosition
        engageBrake();
        this.carouselController.disable();
    }
    /*******************************************************************/
    /************** END of ROTATION COMMANDS ***************************/
    /*******************************************************************/

    /*******************************************************************/
    /************** methods which override FilterHolder *** ************/
    /*******************************************************************/  
    /**
     * Return true if carousel is holding a filter at STANDBY position.
     * @return
     * @throws FcsHardwareException 
     */
    @Override
    public boolean isHoldingFilter()  {
        return this.isHoldingFilterAtStandby();
    }

    
    @Override
    public boolean isAtHandoff() {
        return false;
    }

    @Override
    public void updateStateWithSensors()  {
        this.updateClampsStateWithSensors();
    }


    /*******************************************************************/
    /************** END of methods which override FilterHolder *********/
    /*******************************************************************/
}
