
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This is a model for the numeric sensors we have in single-filter-test and the
 * loader, to monitor the latches in the autochanger, the autochanger trucks
 * position in single-filter-test and the loader hooks.
 *
 *
 * This is also used in the loader test bench where the sensors are connected to
 * a Pluto PLC for the camera protection system. We read the sensors through a
 * Gateway (CanOpenPlutoGateway) byte by byte. So we need to know on which byte
 * this sensor is coded.
 *
 * @author virieux
 */
public class CanOpenNumericSensor extends NumericSensor {

    /**
     * This is the name of the CANOpen DIO (digital input output) device in the groovy file. 
     * Used in the toString method, just for tracing purpose when the subsystem starts from groovy
     * configuration.
     *
     */
    @ConfigurationParameter(isFinal=true,description="The name of the Digital Input Output device "
            + "where this sensor is plugged on.")
    private String dioName;

    /**
     * Numero of input chanel on the device where this sensor is plugged.
     */
    @ConfigurationParameter(isFinal=true,range="0..7",description="The numero of input on DIO device "
            + "where this sensor is plugged on.")
    private int inputNumero;

    /**
     * Gateway byte number where the sensor is coded. value = 1 or 2 in january
     * 2014 value = 0,1,2,3,4,5 in december 2014
     */
    @ConfigurationParameter(isFinal=true,range="0..15",description="The numero of the byte "
            + "where the value of this sensor is stored in.")
    protected int byteNumero;

    /**
     * Build a new CanOpenNumericSensor.
     * @param dioName name of the CANopen dio device where this sensor is plugged on.
     * @param inputNumero 
     */
    public CanOpenNumericSensor(String dioName, int inputNumero) {
        this.dioName = dioName;
        this.inputNumero = inputNumero;
    }

    /**
     * Build a new CanOpenNumericSensor which is plugged on a plutoGateway for example.
     * @param dioName
     * @param inputNumero
     * @param byteNumero 
     */
    public CanOpenNumericSensor(String dioName, int inputNumero, int byteNumero) {
        this.dioName = dioName;
        this.inputNumero = inputNumero;
        this.byteNumero = byteNumero;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public int getByteNumero() {
        return byteNumero;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    @Override
    public int getInputNumero() {
        return inputNumero;
    }

    /**
     * This methods updates the digital value for this sensor. It takes as an
     * argument the value in hexa sent by the CompactIOModule. This hexa value
     * represents the 8 values for the 8 channels. To retrieve the value for
     * this sensor, the hexa value is transformed in binary, then reverse and
     * the digital value is the digit at the index= inputNumero. 
     * example if hex = 80, and inputNumero=7, digital value=1 otherwise digital value=0.
     *
     * @param hex
     */
    @Override
    public synchronized void updateValue(String hex) {

        FCSLOG.finest(getName() + " INPUT NO=" + inputNumero);

        String bin = Integer.toBinaryString(Integer.parseInt(hex, 16));
        //TODO appliquer un mask plutôt que passer par un String.

        //to add leading zeros on the left
        String s = String.format("%08d", Integer.parseInt(bin));
        FCSLOG.finest(getName() + " binary value=" + s);

        /*to retrieve the digital value for this sensor
        /*(the digits are given from left to right
        /*1010000 => DIO[7]=1,DIO[6]=0,DIO[5]=1,DIO[4]=0,DIO[3]=0,DIO[2]=0,DIO[1]=0,DIO[0]=0
        */
        int myValue = Character.digit(s.charAt(7 - inputNumero), 16);

        FCSLOG.finest(getName() + " Value updated=" + myValue);

        this.digitalValue = myValue;
    }

    /**
     * Update the field digitalValue from the bytes read on the gateway.
     *
     * @param hexaValues
     */
    @Override
    public synchronized void updateValue(String[] hexaValues) {
        this.updateValue(hexaValues[byteNumero]);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias="printValue")
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("/DIO module=");
        sb.append(this.dioName);
        sb.append("/Input numero=");
        sb.append(String.valueOf(this.inputNumero));
        sb.append("/byteNumero=");
        sb.append(String.valueOf(this.byteNumero));
        sb.append("/digitalValue=");
        sb.append(String.valueOf(this.digitalValue));
        return sb.toString();
    }

}
