
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * The Three Online clamps which holds a filter when it is at ONLINE position
 * can ve viewed as a single object. It's the goal of this class to represent
 * this object. Opening or closing the 3 clamps at ONLINE is not the same action
 * as opening or closing online clamps one by one.
 *
 * @author virieux
 */
public class AutochangerThreeOnlineClamps extends MobileItemModule {

    private AutoChangerModule autochanger;
    private final AutochangerOnlineClampModule onlineClampXminus;
    private final AutochangerOnlineClampModule onlineClampXplus;
    private final AutochangerOnlineClampModule onlineClampYminus;

    private FcsEnumerations.LockStatus lockStatus;


    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamps
     state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    @ConfigurationParameter(description="timeout in milliseconds : if closing the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForLockingClamps = 15000;
    
    @ConfigurationParameter(description="timeout in milliseconds : if unlocking the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.")
    private long  timeoutForUnlockingClamps = 15000;

    /**
     * Create a AutochangerThreeOnlineClamps with 3 AutochangerOnlineClampModule.
     * @param onlineClampXminus
     * @param onlineClampXplus
     * @param onlineClampY
     */
    public AutochangerThreeOnlineClamps(
            AutochangerOnlineClampModule onlineClampXminus,
            AutochangerOnlineClampModule onlineClampXplus,
            AutochangerOnlineClampModule onlineClampY) {
        this.onlineClampXminus = onlineClampXminus;
        this.onlineClampXplus = onlineClampXplus;
        this.onlineClampYminus = onlineClampY;
    }

    /**
     * Return lockStatus
     * If lockStatus is being updated, wait until update completion else return immediately lockStatus.
     * @return 
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(getName() + ": interrupted in getLockStatus.",ex);
                }

            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }
    
    /**
     * Returns true if LockStatus=LOCKED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=LOCKED")
    public boolean isLocked() {
        return this.getLockStatus() == LockStatus.LOCKED;
    }
    
    /**
     * Returns true if LockStatus=UNLOCKED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=UNLOCKED")
    public boolean isUnlocked() {
        return this.getLockStatus() == LockStatus.UNLOCKED;
    }
    
    /**
     * Returns true if LockStatus=ERROR
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=ERROR")
    public boolean isInError() {
        return this.getLockStatus() == LockStatus.ERROR;
    }
    
    

    @Override
    public void initModule() {
        this.autochanger = (AutoChangerModule) getComponentLookup().getComponentByName("autochanger");
        this.lockStatus = FcsEnumerations.LockStatus.UNKNOWN;
    }
    
    /**
     * Return true if the 3 onlineClamps hardware is initialized. 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if the 3 onlineClamps hardware is ready : controllers ready, "
                + "controllers parameters checked and controllers configured.")
    public boolean isInitialized() {
        return onlineClampXminus.isInitialized() && onlineClampXplus.isInitialized() 
                && onlineClampYminus.isInitialized();
    }
    
    /**
     * Lock the online clamps.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Lock the online clamps.")
    public String lockClamps()  {
        if (lockStatus == LockStatus.LOCKED) {
            throw new RejectedCommandException(name + " is already LOCKED");
        }
        return this.executeAction(MobileItemAction.LOCK_ONLINECLAMPS, timeoutForLockingClamps);
    }

    /**
     * Unlock the online clamps.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Unlock the online clamps.")
    public String unlockClamps()  {
        if (lockStatus == LockStatus.UNLOCKED) {
            throw new RejectedCommandException(name + " is already UNLOCKED");
        }        
        return this.executeAction(MobileItemAction.UNLOCK_ONLINECLAMPS, timeoutForUnlockingClamps);
    }

    /**
     * Return true if the 3 onlineClamps hardware is ready. 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if the 3 onlineClamps hardware is ready.")
    @Override
    public boolean isCANDevicesReady() {
        return onlineClampXminus.isCANDevicesReady() && onlineClampXplus.isCANDevicesReady() 
                && onlineClampYminus.isCANDevicesReady();
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (action == MobileItemAction.LOCK_ONLINECLAMPS) {
            return lockStatus == LockStatus.LOCKED; 
            
        } else if (action == MobileItemAction.UNLOCK_ONLINECLAMPS) {
            return lockStatus == LockStatus.UNLOCKED; 
            
        } else {
            throw new IllegalArgumentException(name + " invalid action:"+action);
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        updateStateWithSensors(); 
    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        if (action == MobileItemAction.LOCK_ONLINECLAMPS) {
            onlineClampXminus.getController().enable();
            onlineClampXplus.getController().enable();
            onlineClampYminus.getController().enable();
            
            onlineClampXminus.getController().releaseBrake();
            onlineClampXplus.getController().releaseBrake();
            onlineClampYminus.getController().releaseBrake();
            
            onlineClampXminus.getController().writeCurrent(onlineClampXminus.getCurrentToLock());
            onlineClampXplus.getController().writeCurrent(onlineClampXplus.getCurrentToLock());
            onlineClampYminus.getController().writeCurrent(onlineClampYminus.getCurrentToLock());
            
        } else if (action == MobileItemAction.UNLOCK_ONLINECLAMPS) {
            onlineClampXminus.getController().enable();
            onlineClampXplus.getController().enable();
            onlineClampYminus.getController().enable();

            onlineClampXminus.getController().writeCurrent(onlineClampXminus.getCurrentToClamp());
            onlineClampXplus.getController().writeCurrent(onlineClampXplus.getCurrentToClamp());
            onlineClampYminus.getController().writeCurrent(onlineClampYminus.getCurrentToClamp());
            
            onlineClampXminus.getController().releaseBrake();
            onlineClampXplus.getController().releaseBrake();
            onlineClampYminus.getController().releaseBrake();
            //TODO: slowDownCurrent ?

            onlineClampXminus.getController().writeCurrent(onlineClampXminus.getCurrentToUnlock());
            onlineClampXplus.getController().writeCurrent(onlineClampXplus.getCurrentToUnlock());
            onlineClampYminus.getController().writeCurrent(onlineClampYminus.getCurrentToUnlock());
        } else {
            throw new IllegalArgumentException(name + " invalid action:"+action);
        } 
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay)  {
            onlineClampXminus.abortAction(action,delay);
            onlineClampXplus.abortAction(action,delay);
            onlineClampYminus.abortAction(action,delay);
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay)  {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action)  {
        //nothing to do 
    }



    public StatusDataPublishedByAutochangerThreeClamps getStatusData() {
        return createStatusDataPublishedByThreeClamps();
    }
    
    public StatusDataPublishedByAutochangerThreeClamps createStatusDataPublishedByThreeClamps() {
        StatusDataPublishedByAutochangerThreeClamps status = new StatusDataPublishedByAutochangerThreeClamps();
        status.setName(getName());
        status.setLockStatus(lockStatus);
        return status;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData("autochangerClamps", getStatusData()));
    }

    /**
     * This methods updates the whole autochanger state in reading all the
     * sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update autochanger state in reading sensors.")
    public void updateStateWithSensors()  {
        autochanger.updateStateWithSensors();
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the
     * method.
     *
     * @param hexaValues
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors(String[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;
            this.onlineClampXminus.updateStateWithSensors(hexaValues);
            this.onlineClampXplus.updateStateWithSensors(hexaValues);
            this.onlineClampYminus.updateStateWithSensors(hexaValues);

            computeLockStatus();

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }
    }
    
    private void computeLockStatus() {
        boolean inError = onlineClampXminus.isInError()
                || onlineClampXplus.isInError()
                || onlineClampYminus.isInError();

        boolean locked = onlineClampXminus.isLocked()
                && onlineClampXplus.isLocked()
                && onlineClampYminus.isLocked();

        boolean unlocked = onlineClampXminus.isUnlocked()
                && onlineClampXplus.isUnlocked()
                && onlineClampYminus.isUnlocked();

        boolean inTravel = onlineClampXminus.isInTravel()
                && onlineClampXplus.isInTravel()
                && onlineClampYminus.isInTravel();

        if (inError) {
            this.lockStatus = LockStatus.ERROR;
        } else if (locked) {
            this.lockStatus = LockStatus.LOCKED;
        } else if (unlocked) {
            this.lockStatus = LockStatus.UNLOCKED;
        } else if (inTravel) {
            this.lockStatus = LockStatus.INTRAVEL;
        } else {
            this.lockStatus = LockStatus.UNKNOWN;
        }
    }

}
