
package org.lsst.ccs.subsystems.fcs;


import java.util.Map;
import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This is a model for online clamp which holds a filter when it is at ONLINE
 * position. The online clamps are part of the autochanger subsystem. There is 3
 * online clamps.
 *
 * @author virieux
 */
public class AutochangerOnlineClampModule extends MobileItemModule implements MovedByEPOSController {

    private final int timeoutForLocking = 10000;
    private final int timeoutForUnlocking = 10000;
    private AutoChangerModule autochanger;
    private final EPOSController controller;
    private final NumericSensor lockSensor;
    
    /*lock sensor complementary */
    private final NumericSensor lockSensorC;
    private final NumericSensor unlockSensor;
    
    /*unlock sensor complementary*/
    private final NumericSensor unlockSensorC;

    @ConfigurationParameter(isFinal = true)
    private Map<String, Integer> paramsForCurrentToLock;
    
    @ConfigurationParameter(isFinal = true)
    private Map<String, Integer> paramsForCurrentToUnlock;
    
    @ConfigurationParameter(description="current to lock ONLINE clamp in mA")
    private int currentToLock = 300; 
            
    @ConfigurationParameter(description="current to unlock ONLINE clamp, in mA")
    private int currentToUnlock = -300;
    
    @ConfigurationParameter(description="current to clamp ONLINE clamp, in mA")
    private int currentToClamp = 3000;
    
    @ConfigurationParameter(description="time to hold clamp, in millis")
    private long holdTime = 5000;
    
    @ConfigurationParameter(description="time for clamp to lock, in millis")
    private long travelTime = 5000;

    private boolean inError, lockSensorInError, unlockSensorInError;

    private FcsEnumerations.LockStatus lockStatus;
    private boolean controllerInFault;

    private volatile boolean initialized;
    private boolean controllerConfigured;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the latch state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    /**
     * Build an AutochangerOnlineClampModule with a controller, 4 sensors and 
     * 2 maps of parameters for the controller.
     * @param onlineClampController
     * @param lockSensor0
     * @param lockSensor1
     * @param unlockSensor0
     * @param unlockSensor1
     * @param paramsForCurrentToLock
     * @param paramsForCurrentToUnlock
     */
    public AutochangerOnlineClampModule(
            EPOSController onlineClampController,
            NumericSensor lockSensor0,
            NumericSensor lockSensor1,
            NumericSensor unlockSensor0,
            NumericSensor unlockSensor1,
            Map<String, Integer> paramsForCurrentToLock,
            Map<String, Integer> paramsForCurrentToUnlock) {
        super(3000);
        this.controllerConfigured = false;
        this.controller = onlineClampController;
        this.lockSensor = lockSensor0;
        this.lockSensorC = lockSensor1;
        this.unlockSensor = unlockSensor0;
        this.unlockSensorC = unlockSensor1;
        this.paramsForCurrentToLock = paramsForCurrentToLock;
        this.paramsForCurrentToUnlock = paramsForCurrentToUnlock;
    }

    /**
     * Return ONLINE clamp controller
     * @return 
     */
    public EPOSController getController() {
        return controller;
    }

    /**
     * For simulator
     * @return 
     */
    public int getCurrentToLock() {
        return currentToLock;
    }

    /**
     * For simulator
     * @return 
     */
    public int getCurrentToUnlock() {
        return currentToUnlock;
    }

    /**
     * For simulator
     * @return 
     */
    public int getCurrentToClamp() {
        return currentToClamp;
    }

    /**
     * For simulator
     * @return 
     */
    public long getTravelTime() {
        return travelTime;
    }
    
    

    /**
     * Returns true if lockSensor and lockSensorC return the same value.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if lockSensor and lockSensorC retun the same value. "
                    + "Doesn't read again sensors.")
    public boolean isLockSensorInError() {
        return lockSensorInError;
    }

    /**
     * Returns true if unlockSensor and unlockSensorC retun the same value.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if unlockSensor and unlockSensorC retun the same value. "
                    + "Doesn't read again sensors.")
    public boolean isUnlockSensorInError() {
        return unlockSensorInError;
    }

    /**
     * Returns true if LockStatus=ERROR, this means lockSensor or unlockSensor is in ERROR 
     * or unlockSensor and lockSensor return non consistant values.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=ERROR, this means that "
            + "lockSensor or unlockSensor is in ERROR or that"
            + "unlockSensor and lockSensor return non consistant values. Doesn't read again sensors.")
    public boolean isInError() {
        return lockStatus == FcsEnumerations.LockStatus.ERROR;
    }

    /**
     * Returns true if LockStatus=LOCKED.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=LOCKED. Doesn't read again sensors.")
    public boolean isLocked() {
        return lockStatus == FcsEnumerations.LockStatus.LOCKED;
    }

    /**
     * Returns true if LockStatus=UNLOCKED.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=UNLOCKED. Doesn't read again sensors.")
    public boolean isUnlocked() {
        return lockStatus == FcsEnumerations.LockStatus.UNLOCKED;
    }

    /**
     * Returns true if LockStatus=INTRAVEL.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=INTRAVEL. Doesn't read again sensors.")
    public boolean isInTravel() {
        return lockStatus == FcsEnumerations.LockStatus.INTRAVEL;
    }

    /**
     * Returns true if controller is in fault.
     * Doesn't read controller CPU.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if controller is in fault. Doesn't read controller CPU.")
    @Override
    public boolean isControllerInFault() {
        return controllerInFault;
    }

    /**
     * set controllerInFault
     * @param controllerInFault 
     */
    @Override
    public void setControllerInFault(boolean controllerInFault) {
        this.controllerInFault = controllerInFault;
    }
    
    /**
     * return controller name
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
        description = "Returns controller name.")
    @Override
    public String getControllerName() {
        return controller.getName();
    }

    /**
     * Returns true if clamp is initialized : hardware is ready (at least booted) and
     * clamp controller is initialized (parameters in the controller CPU have been checked and 
     * controller has been configured.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if clamp is initialized : controller is booted, parameters in the controller "
                    + "CPU have been checked and controller is configured.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * Initialize fields autochanger, lockStatus and listens to its controller.
     */
    @Override
    public void initModule() {
        super.initModule();
        this.autochanger = (AutoChangerModule) getComponentLookup().getComponentByName("autochanger");
        this.lockStatus = FcsEnumerations.LockStatus.UNKNOWN;
        
        /**
         * Because AutochangerOnlineClampModule implements MovedByEPOSController, it has to listen to its controller
         * to known when the controller is in Fault or when a Fault reset has been done.
         */
        if (controller == null) {
            FCSLOG.error(getName() + "==>>> onlineController == null - Please fix groovy description file.");
            throw new IllegalArgumentException(getName() + "==>>> null onlineClampController - fix groovy description file.");
        } else {
            //listens to my Controller to detect the controller's faultReset
            //or the emergency messages coming from the controller.
            if (controller instanceof Observable) {
                this.listens((Observable) controller);
            }
        }
    }
       

    @Override
    public void tick() {
        //this.publishData();
    }

    /**
     * Return true when ONLINE clamp hardware is ready. That is when controller is initialized and configured.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY,
            level = Command.ENGINEERING1,
            description = "Returns true if controller is initialized and configured.")
    @Override
    //tested on CPPM testbench in september 2015
    public boolean isCANDevicesReady() {
        return ((MainModule) getComponentLookup().getComponentByName("main")).isCANDevicesReady()
                && controller.isInitialized() && controllerConfigured;
    }

    /**
     * Check if it's safe to unlock the clamp.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY,
            level = Command.ENGINEERING1,
            description = "Check if the onlineClamp can be locked.")
    public void checkConditionsForUnlocking()  {
        checkConditionsForLocking();
        
        if (!autochanger.isHoldingFilter()) {
            throw new RejectedCommandException(name+" can't be UNLOCKED if autochanger is not HOLDING filter.");
        }
    }

    /**
     * Check if it's safe to lock the clamp.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY,
            level = Command.ENGINEERING1,
            description = "Check if the onlineClamp can be locked.")
    public void checkConditionsForLocking()  {
        if (!autochanger.isAtOnline()) {
            throw new RejectedCommandException(name+" can't be LOCKED if autochanger trucks are not at ONLINE position.");
        } else if (autochanger.isEmpty()) {
            throw new RejectedCommandException(name+" can't be LOCKED if there is no filter in autochanger trucks.");
        }
    }

    /**
     * This method is called during INITIALIZATION phase. It configures the
     * controller for the digital inputs and outputs. It changes it to CURRENT
     * mode if it's not in this mode and it writes in the CPU of the controller
     * the values of configuration for CURRENT mode parameters.
     *
     * @return
     * @throws HardwareException
     */
    @Override
    //tested on CPPM testbench in september 2015
    public TreeWalkerDiag checkHardware() throws HardwareException {
        super.checkHardware();

        FCSLOG.debug(getName() + " checking hardware.");

        try {
            controller.initializeAndCheckHardware();
            configureController();
            //check if is in mode CURRENT else changeMode(EposMode.CURRENT)
            controller.writeParameters(EposMode.CURRENT);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false, ex);

        } catch (FailedCommandException ex) {
            throw new HardwareException(true, ex);
        }
        this.initialized = true;
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * Read parameters for mode CURRENT on controller CPU.
     * If a parameter has a different value than in configuration, throws an exception.
     * 
     * @throws HardwareException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY,
            level = Command.ENGINEERING1,
            description = "Read in the CPU of the controller parameters for mode CURRENT."
            + "If a parameter has a different value than in configuration, "
            + "throws an exception.")
    //tested on CPPM testbench in september 2015
    public void checkControllerBeforeAction() throws HardwareException {

        EposMode modeInCPU = controller.readMode();
        if (!modeInCPU.equals(EposMode.CURRENT)) {
            String msg = getName() + " Controller is not in mode CURRENT";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }
        controller.checkParameters(EposMode.CURRENT);
        controller.checkFault();
        if (controller.isParametersOK()) {
            FCSLOG.debug(getName() + ":controller parameters are OK - ready for action.");

        } else {
            String msg = getName() + " Some parameter values are not"
                    + " the same in CPU and configuration system.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }
    }

    /**
     * Configure ONLINE clamp controller.
     * @throws HardwareException
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION,
            level = Command.ENGINEERING1,
            description = "Configure controller.")
    //tested on CPPM testbench in september 2015
    public void configureController() throws HardwareException {
        try {
            controllerConfigured = false;
            controller.activateBrake();
            controller.shutdown();
            this.configureDigitalInputOfOnlineClamps();
            this.configureDigitalOutputOfOnlineClamps();
            controllerConfigured = true;
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName(),ex);
        }
    }

    /**
     * This methods updates the whole autochanger state in reading all the
     * sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update online clamp state in reading sensors.")
    public void updateStateWithSensors()  {
        autochanger.updateStateWithSensors();
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the
     * method.
     *
     * @param hexaValues
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors(String[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;
            this.lockSensor.updateValue(hexaValues);
            this.lockSensorC.updateValue(hexaValues);
            this.unlockSensor.updateValue(hexaValues);
            this.unlockSensorC.updateValue(hexaValues);

            boolean locked = this.lockSensor.getDigitalValue() == 1
                    && this.lockSensorC.getDigitalValue() == 0;
            boolean unlocked = this.unlockSensor.getDigitalValue() == 1
                    && this.unlockSensorC.getDigitalValue() == 0;
            boolean inTravel = !locked && !unlocked;
            lockSensorInError = this.lockSensor.getDigitalValue()
                    == this.lockSensorC.getDigitalValue();
            unlockSensorInError = this.unlockSensor.getDigitalValue()
                    == this.unlockSensorC.getDigitalValue();
            inError = lockSensorInError || unlockSensorInError || (locked && unlocked);

            if (inError) {
                lockStatus = FcsEnumerations.LockStatus.ERROR;
            } else if (locked) {
                lockStatus = FcsEnumerations.LockStatus.LOCKED;
            } else if (unlocked) {
                lockStatus = FcsEnumerations.LockStatus.UNLOCKED;
            } else if (inTravel) {
                lockStatus = FcsEnumerations.LockStatus.INTRAVEL;
            }

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }

    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (action == MobileItemAction.LOCK_ONLINECLAMP) {
            return this.lockStatus == LockStatus.LOCKED;
            
        } else if (action == MobileItemAction.UNLOCK_ONLINECLAMP) {
            return this.lockStatus == LockStatus.UNLOCKED;
            
        } else throw new IllegalArgumentException(name + " invalid action for ONLINE clamp:"+action);
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        updateStateWithSensors(); 
    }
    
    /**
     * Lock the ONLINE clamp.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Lock the ONLINE clamp.")
    public void lock() {
        if (this.isLocked()) {
            throw new RejectedCommandException(name + " is already LOCKED.");
        }
        checkConditionsForLocking();
        this.executeAction(MobileItemAction.LOCK_ONLINECLAMP, timeoutForLocking);
    }
    
    /**
     * Unlock the ONLINE clamp.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Unlock the ONLINE clamp.")
    public void unlock() {
        if (this.isUnlocked()) {
            throw new RejectedCommandException(name + " is already UNLOCKED.");
        }        
        checkConditionsForUnlocking();
        this.executeAction(MobileItemAction.UNLOCK_ONLINECLAMP, timeoutForUnlocking);
    }
    


    /**
     * Start action of locking or unlocking.
     * @param action
     * @throws FcsHardwareException 
     */
    @Override
    //TODO modify this from what has been learned in september 2015 from testLock and testUnlock
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        
        switch (action) {
            case LOCK_ONLINECLAMP:
                controller.enable();
                controller.changeMode(EposMode.CURRENT);
                //TODO checks these parameters during initialization phase.
                //onlineClampController.writeParameters(paramsForCurrentToLock);
                controller.releaseBrake();
                controller.writeCurrent(this.currentToLock);
                break;

            case UNLOCK_ONLINECLAMP:
                controller.enable();
                controller.changeMode(EposMode.CURRENT);
                //TODO checks these parameters during initialization phase.
                //onlineClampController.writeParameters(paramsForCurrentToUnlock);
                //first apply a force on the clamp, and then releaseBrake
                //don't do it in the reverse order !!!!
                controller.writeCurrent(this.currentToClamp);
                controller.releaseBrake();
                slowDownCurrent();

                controller.writeCurrent(this.currentToUnlock);
                break;
                
            default:
                assert false: action;
        }
    }

    /**
     * Lock online clamp for CPPM test bench.
     * @return
     * @throws FcsHardwareException
     * @throws HardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Lock online clamp for CPPM test bench.")
    //tested on CPPM testbench in september 2015
    public String testLock() throws HardwareException {
        
        try {
            FCSLOG.debug(getName() + "BEGIN testLock");
            
            checkConditionsForLocking();
            FCSLOG.debug(getName() + ":condition for locking OK.");
            
            //commented in march 2015 because we want to enable the controller in all the cases.
//                if (!onlineClampController.isEnabled())
//                    throw new RejectedCommandException(getName() + "controller has to be ENABLED first.");
            controller.enable();
            checkControllerBeforeAction();
            //now the parameters are OK
            
            controller.releaseBrake();
            
            controller.writeCurrent(this.currentToLock);
            FCSLOG.debug(getName() + ": sent current to controller:" + currentToLock);
            //il y aura des capteurs mais en attendant, on attend.
            FCSLOG.debug(getName() + ": sleeping for:" + travelTime);
            FCSLOG.debug(getName() + ":time=" + System.currentTimeMillis());
            Thread.sleep(travelTime);
            FCSLOG.debug(getName() + ":time=" + System.currentTimeMillis());
            
            controller.writeCurrent(this.currentToClamp);
            FCSLOG.debug(getName() + "==> current sent to controller=" + currentToClamp);
            FCSLOG.debug(getName() + ": sleeping for:" + holdTime);
            FCSLOG.debug(getName() + ":time=" + System.currentTimeMillis());
            Thread.sleep(holdTime);
            FCSLOG.debug(getName() + ":time=" + System.currentTimeMillis());
            
            //the good order is activateBrake, then writeCurrent in this case.
            //because we want to put the pression, active brake and then set current to 0.
            controller.activateBrake();
            FCSLOG.debug(getName() + ": controller brake activated.");
            Thread.sleep(1000);
            
            controller.writeCurrent(0);
            FCSLOG.debug(getName() + "==> current sent to controller=" + 0);
            controller.disable();//added in sept2015- to be tested
            FCSLOG.debug(getName() + "END testLock");
            return getName() + " LOCKED";
        } catch (InterruptedException ex) {
            throw new FcsHardwareException("testLock was interrupted while sleeping",ex);
        }
    }

    /**
     * Unlock online clamp for CPPM test bench.
     * @return
     * @throws FcsHardwareException
     * @throws HardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Unlock online clamp for CPPM test bench.")
    //tested on CPPM testbench in september 2015
    public String testUnlock() throws HardwareException {
        try {
            FCSLOG.debug(getName() + "BEGIN testUnlock");
            
            checkConditionsForUnlocking();
            FCSLOG.debug(getName() + ":controller parameters are OK.");
            
            controller.enable();
            /*decided with Patrick in sept 2015: the mode CURRENT is set during initialization in checkHardware
            and it is checked before each action in checkControllerBeforeAction*/
            checkControllerBeforeAction();
            //now the parameters are OK
            //first apply a force on the clamp, and then releaseBrake
            //don't do it in the reverse order !!!!
            controller.writeCurrent(this.currentToClamp);
            controller.releaseBrake();
            Thread.sleep(1000);
            slowDownCurrent();
            
            controller.writeCurrent(this.currentToUnlock);
            Thread.sleep(travelTime + 3000);
            
            controller.writeCurrent(0);
            Thread.sleep(10000);
            
            //controller.activateBrake();//done in disable
            controller.disable();
            FCSLOG.debug(getName() + "END testUnlock");
            return getName() + " UNLOCKED";
        } catch (InterruptedException ex) {
            throw new FcsHardwareException("testUnlock was interrupted while sleeping",ex);
        }

    }

    /**
     * What to do to abort an action.
     * @param action
     * @param delay
     * @throws FcsHardwareException 
     */
    @Override
    //tested on CPPM testbench in september 2015
    public void abortAction(FcsEnumerations.MobileItemAction action,long delay)  {
        //TODO check if correct with Patrick
        controller.quickStop();
        controller.activateBrake();
        controller.disable();
    }

    /**
     * What to be done after the action is completed.
     * @param action
     * @throws FcsHardwareException 
     */
    @Override
    public void postAction(FcsEnumerations.MobileItemAction action)  {
        switch (action) {
            case LOCK_ONLINECLAMP:
                controller.writeCurrent(this.currentToClamp);
                try {
                    Thread.sleep(holdTime);
                } catch (InterruptedException ex) {
                    throw new FailedCommandException(getName()
                            + " was interrupted during action " + action.toString());
                }
                controller.activateBrake();
                controller.writeCurrent(0);
                controller.disable(); //added in sept2015 to be consistant with trucks
                //to be tested on CPPM test bench
                break;

            case UNLOCK_ONLINECLAMP:

                break;
                
            default:
                assert false: action;
        }
    }

    /**
     * slow down current 
     * @throws FcsHardwareException 
     */
    private void slowDownCurrent()  {
        int step = currentToClamp / 5;
        for (int curr = currentToClamp - step; curr > 0; curr = curr - step) {
            FCSLOG.debug("current=" + curr);
            controller.writeCurrent(curr);
            //this is for CPPM test bench only - for the scale 1 prototype,
            //we will do it for the 3 controllers
            try {
                Thread.sleep(500);
            } catch (InterruptedException ex) {
                throw new RejectedCommandException(getName()
                        + " was interrupted during action testLock." + ex.getMessage());
            }
            //read actualCurrent to check if it actualCurrent = curr.
            int actualCurrent = controller.readCurrent();
            if (actualCurrent != curr) {
                FCSLOG.error(getName() + ": CURRENT ERROR in"
                        + " slowDownCurrent actualCurrent=" + actualCurrent + ",curr=" + curr);
            }

        }
        FCSLOG.debug("current=" + 0);
        controller.writeCurrent(0);
    }

    /**
     * This method is used to configure the controllers of the autochanger
     * online clamps. * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Digital
     * Inputs activated for OnlineClamps controllers are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ QuickStop Input=01
     * Mask=Enabled Polarity=HighActive HE_Sensor General_A Input=02
     * Mask=Enabled Polarity=HighActive HE_Sensor\ General_B Input=03
     * Mask=Enabled Polarity=HighActive Check Brake Status
     * ******************************************** 1 - Configure the Digital
     * Input Functionnality index 0x2070 sindex input_ID Functionnality tab
     * Value = 15 General Purpose A Value = 14 General Purpose B Value = 13
     * General Purpose C Value = 12 General Purpose D Value = 11 General Purpose
     * E Value = 10 General Purpose F Value = 9 General Purpose G Value = 8
     * General Purpose H Value = 7 General Purpose I Value = 6 General Purpose J
     * Value = 5 QuickStop Value = 4 Drive enable Value = 3 Position marker
     * Value = 2 Home switch Value = 1 Positive Limit switch Value = 0 Negative
     * Limit switch msg = 'Set DigIn_1 Quick Stop,wsdo,%s,2070,01,02,0005\n' %
     * (str(cobID)) self.com_hal.executeOperation(client,msg) msg = 'Set DigIn_2
     * General_A,wsdo,%s,2070,02,02,8000\n' % (str(cobID))
     * self.com_hal.executeOperation(client,msg) msg = 'Set DigIn_3
     * Genral_B,wsdo,%s,2070,02,02,4000\n' % (str(cobID))
     * self.com_hal.executeOperation(client,msg)
     * *************************************************************** 2 -
     * Configure the Digital Input Polarity index 0x2071 subindex 0x03 Value = 0
     * High Active Value = 1 Low Active
     * **************************************************************** 3 -
     * Configure the Digital Input Mask index 0x2071 subindex 0x02 Value = 0
     * Functionnality state will not be displayed Value = 1 Functionnality state
     * will be displayed * Digital Input Functionnality tab value Bit_15 General
     * Purpose A 1 Bit_14 General Purpose B 1 Bit_13 General Purpose C 0 Bit_12
     * General Purpose D 0 Bit_11 General Purpose E 0 Bit_10 General Purpose F 0
     * Bit_9 General Purpose G 0 Bit_8 General Purpose H 0 Bit_7 General Purpose
     * I 0 Bit_6 General Purpose J 0 Bit_5 QuickStop 1 Bit_4 Drive enable 0
     * Bit_3 Position marker 0 Bit_2 Home switch 0 Bit_1 Positive Limit switch 0
     * Bit_0 Negative Limit switch 0
     * ********************************************************************* 4 -
     * Configure the Digital Input Execution Mask index 0x2071 subindex 0x04
     * Digital Input Excecution Mask Value Bit_15 to Bit_6 reserved 0 Bit_5
     * QuickStop 1 Bit_4 Drive enable 1 Bit_3 Position marker 0 Bit_2 Home
     * switch 0 Bit_1 Positive Limit switch 0 Bit_0 Negative Limit switch 0
     * **********************************************************************
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the online clamps controllers.")
    public void configureDigitalInputOfOnlineClamps()  {
        //1-Configure Digital Input Fonctionnality
        //QuickStop
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalInput1, "0005");
        //General_A
        //writeParameterInHexa(CanOpenEPOS.Parameter.ConfigurationOfDigitalInput2, "8000");
        //General_B
        //writeParameterInHexa(CanOpenEPOS.Parameter.ConfigurationOfDigitalInput3, "4000");

        //2-Configure the Digital Input Polarity
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityPolarity, "0");

        //3-Configure the Digital Input Mask
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityMask, "C20");

        //4 - Configure the Digital Input Execution Mask
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityExecutionMask, "30");

    }

    /**
     * This methods is used to configure the autochanger onlineClamps EPOS
     * controllers. * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Digital
     * Outputs activated for OnlineClamps controller are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Ready/Fault Output=01
     * Mask=Enabled Polarity=HighActive General_A Output=04 Mask=Enabled
     * Polarity=HighActive ******************************************** 1 -
     * Configure the Digital Output Functionnality index 0x2079 subindex
     * input_ID Functionnality tab Value = 15 General Purpose Out_A Value = 14
     * General Purpose Out_B Value = 13 General Purpose Out_C Value = 12 General
     * Purpose Out_D Value = 11 General Purpose Out_E Value = 10..8 not used
     * Value = 7..3 reserved Value = 2 Holding Brake Value = 1 Position compare
     * Value = 0 Ready/Fault
     * ********************************************************************************
     * 2 - Configure the Digital Output Functionnality Mask index 0x2078
     * subindex 0x02 Value = 0 functionnality not activatd Value = 1
     * functionnality activated
     * ********************************************************************************
     * Digital Output Functionnality Mask Value Bit_15 General Purpose Out_A 1
     * Bit_14 General Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12
     * General Purpose Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General
     * Purpose Out_F 0 Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H
     * 0 Bit_7 reserved 0 Bit_6 reserved 0 Bit_5 reserved 0 Bit_4 reserved 0
     * Bit_3 reserved 0 Bit_2 Holding Brake 0
     * ********************************************************************************
     * 3 - Configure the Digital Output Functionnality Polarity index 0x2078
     * subindex 0x03 Value = 0 associated output not change => HighActive Value
     * = 1 associated output inverted => LowActive
     * ********************************************************************************
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the online clamps controllers.")
    public void configureDigitalOutputOfOnlineClamps()  {
        //1 - Configure the Digital Output Functionnality
        //Ready/Fault
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput1, "0");
        //General_A
        //gives an error : CAN OPEN DEVICE ERROR CODE=6090031 index=2079 subindex=04 error name=Value too high Error
        //writeParameterInHexa(Parameter.ConfigurationOfDigitalOutput4, "8000"); //pas bon

        //2 - Configure the Digital Output Functionnality Mask
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityMask, "8001");

        //3 - Configure the Digital Output Functionnality Polarity
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityPolarity, "8001");
    }

    /**
     * What to do when the Modules we observe send there new values. This class
     * observes its controller to publish data when its controller is in fault
     * or after a faultReset. Needed to update the GUI.
     *
     * @param source
     * @param v
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        FCSLOG.debug(getName() + ":processUpdate from source=" + source.toString()
                + " ValueUpdate=" + v.getName());
        if (!(source instanceof EPOSController)) {
            return;
        }
        if (v.getValue() instanceof EmergencyMessage) {
            EmergencyMessage emcyMsg = (EmergencyMessage) v.getValue();
            FCSLOG.debug(getName() + ":EmergencyMessage received from CanOpenProxy="
                    + emcyMsg.toString());
            processEmergencyMessage(emcyMsg);


        } else if (v.getValue() instanceof String) {
            EPOSController ctrl = (EPOSController) source;
            String msgFromController = (String) v.getValue();
            if (ctrl.getName().equals(controller.getName()) &&  ("faultReset".equals(msgFromController))) {               
                    this.controllerInFault = false;
                    this.publishData();
            }
        }
    }

    /**
     * Returns the object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerOnlineClamp getStatusData() {
        return createStatusDataPublishedByOnlineClamp();
    }
    
    /**
     * Creates and returns the object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerOnlineClamp createStatusDataPublishedByOnlineClamp() {
        StatusDataPublishedByAutochangerOnlineClamp status = new StatusDataPublishedByAutochangerOnlineClamp();
        status.setName(getName());
        status.setLockSensorValue(lockSensor.getDigitalValue());
        status.setUnlockSensorValue(unlockSensor.getDigitalValue());
        status.setLockStatus(lockStatus);
        status.setLockSensorInError(lockSensorInError);
        status.setUnlockSensorInError(unlockSensorInError);
        status.setInError(inError);
        status.setControllerInFault(controllerInFault);
        return status;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), getStatusData()));
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay)  {
        controller.quickStop();
    }

}
