
package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This class is a model for the latch which lock a filter on the autochanger
 * trucks. 
 * The ACTION methods : open and close of this class is used in ENGINEERING mode only 
 * when an operator may need to open or close only one Latch. 
 * In NORMAL mode, when we have to open or close both Latches in the same
 * time, the class AutochangerTwoLatches is used. 
 * On the autochanger there are 2 latches : one on each side of the filter : at X+ and X-.
 *
 * @author virieux
 */
public class AutochangerLatchModule extends MobileItemModule implements MovedByEPOSController {

    private AutoChangerModule autochanger;
    private final EPOSController latchController;
    private final NumericSensor filterEngagedSensor;
    private final NumericSensor filterEngagedSensorB;
    private final NumericSensor lockSensor; 
    private final NumericSensor lockSensorB;
    private final NumericSensor unlockSensor;
    private final NumericSensor unlockSensorB;

    private volatile boolean filterEngaged;
    private boolean locked;
    private boolean unlocked;
    private boolean inError;
    private boolean inTravel;
    private boolean filterEngagedSensorsInError;
    private boolean lockSensorsInError;
    private boolean unlockSensorsInError;

    @ConfigurationParameter(description="in mA current to be sent to the controller to open the latch.")
    private int currentToOpen = 200;
    
    @ConfigurationParameter(description="timeout for opening a latch in milliseconds.")
    private long timeoutForOpening = 3000;
    
    @ConfigurationParameter(description="timeout for closing a latch in milliseconds.")
    private long timeoutForClosing = 3000;
    
    @ConfigurationParameter(description="min position value used by the GUI")
    private int minCurrent = -300;
    
    @ConfigurationParameter(description="max current value used by the GUI")
    private int maxCurrent = 300;
    
    private int readCurrent;

    private LockStatus lockStatus;

    private volatile boolean initialized;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the latch state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;



    /**
     * FOR THE GUI
     * true if the controller is in fault
     * false if the user does a faultReset
     */
    private boolean controllerInFault;

    /**
     * Buils a AutochangerLatchModule with a controller and 3 sensors.
     * @param latchController
     * @param lockSensor
     * @param lockSensorB
     * @param unlockSensor
     * @param filterPresenceSensor 
     * @param unlockSensorB 
     * @param filterPresenceSensorB 
     */
    public AutochangerLatchModule(
            EPOSController latchController,
            NumericSensor lockSensor,
            NumericSensor lockSensorB,
            NumericSensor unlockSensor,
            NumericSensor unlockSensorB,
            NumericSensor filterPresenceSensor,
            NumericSensor filterPresenceSensorB) {
        this.lockStatus = LockStatus.UNKNOWN;
        this.filterEngaged = false;
        this.latchController = latchController;
        this.lockSensor = lockSensor;
        this.lockSensorB = lockSensorB;
        this.unlockSensor = unlockSensor;
        this.unlockSensorB = unlockSensorB;
        this.filterEngagedSensor = filterPresenceSensor;
        this.filterEngagedSensorB = filterPresenceSensorB;
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns this latch controller name.")
    @Override
    public String getControllerName() {
        return latchController.getName();
    }

    /**
     * Return latchController
     * @return 
     */
    public EPOSController getLatchController() {
        return latchController;
    }

    /**
     * Returns true if latchController is in Fault.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if latchController is in Fault.")
    @Override
    public boolean isControllerInFault() {
        return controllerInFault;
    }

    /**
     * Set controllerInFault
     * @param controllerInFault 
     */
    @Override
    public void setControllerInFault(boolean controllerInFault) {
        this.controllerInFault = controllerInFault;
    }
    
    

    /**
     * Returns true if latch is initialized. 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if latch is initialized.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * For simulation, return currentToOpen
     * @return 
     */
    public int getCurrentToOpen() {
        return currentToOpen;
    }

    /**
     * Return timeoutForOpening
     * @return 
     */
    public long getTimeoutForOpening() {
        return timeoutForOpening;
    }

    /**
     * Return timeoutForClosing
     * @return 
     */
    public long getTimeoutForClosing() {
        return timeoutForClosing;
    }

    /**
     * Return lockStatus.
     * If another thread is updating lockStatus it waits untils update is completed and
     * then returns lockStatus.
     * Otherwise it returns lockStatus immediately.
     * 
     * @return 
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(getName() + ": interrupted in getLockStatus.",ex);
                }
            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns true if LockStatus=LOCKED
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=LOCKED. Doesn't read again sensors.",
            alias="isClosed")
    public boolean isLocked() {
        return this.getLockStatus() == LockStatus.LOCKED;
    }

    /**
     * Returns true if LockStatus=UNLOCKED
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=UNLOCKED. Doesn't read again sensors.",
            alias="isOpen")
    public boolean isUnlocked() {
        return this.getLockStatus() == LockStatus.UNLOCKED;
    }

    /**
     * Returns true if LockStatus=ERROR, this means that unlockSensor and lockSensor return 
 non consistant values.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=ERROR, this means that "
            + "unlockSensor and lockSensor return non consistant values. Doesn't read again sensors.")
    public boolean isInError() {
        return this.getLockStatus() == LockStatus.ERROR;
    }

    /**
     * Returns true if autochanger is empty.
     * 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if autochanger is empty. Doesn't read again sensors.")
    public boolean isEmpty() {
        return !filterEngaged;
    }

    /**
     * Initialize field autochanger and latchController
     */
    @Override
    public void initModule() {
        super.initModule();
        ComponentLookup lookup = getSubsystem().getComponentLookup();
        this.autochanger = (AutoChangerModule) lookup.getComponentByName("autochanger");

        if (latchController == null) {
            FCSLOG.error(getName() + "==>>> latchController == null - Please fix groovy description file.");
            throw new IllegalArgumentException(getName() + "==>>> null latchController - fix groovy description file.");
        } else {
            //listens to latchController to detect the controller's faultReset
            //or the emergency messages coming from the controller.
            if (latchController instanceof Observable) {
                this.listens((Observable) latchController);
            }
        }
    }

    @Override
    public void tick() {
        this.publishData();
    }

    /**
     * Executed during INITIALIZATION phase.
     * Initialize and check latchController.
     * @return
     * @throws HardwareException 
     */
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        try {
            latchController.initializeAndCheckHardware();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName(),ex);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false, ex);
        }
        //TODO check if there is a homing to be done.
        this.initialized = true;
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * This methods updates the whole autochanger state in reading all the
     * sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update latch state in reading sensors.")
    public void updateStateWithSensors()  {
        autochanger.updateStateWithSensors();
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the
     * method.
     *
     * @param hexaValues
     * 
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors(String[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;
            this.filterEngagedSensor.updateValue(hexaValues);
            this.filterEngagedSensorB.updateValue(hexaValues);
            this.lockSensor.updateValue(hexaValues);
            this.lockSensorB.updateValue(hexaValues);
            this.unlockSensor.updateValue(hexaValues);
            this.unlockSensorB.updateValue(hexaValues);

            filterEngaged = this.filterEngagedSensor.getDigitalValue() == 1 
                    && this.filterEngagedSensorB.getDigitalValue() == 0;
            this.filterEngagedSensorsInError = this.filterEngagedSensor.getDigitalValue() 
                    == this.filterEngagedSensorB.getDigitalValue();

            locked = this.lockSensor.getDigitalValue() == 1 
                    && this.lockSensorB.getDigitalValue() == 0;
            this.lockSensorsInError =  this.lockSensor.getDigitalValue()
                    == this.lockSensorB.getDigitalValue();
            
            unlocked = this.unlockSensor.getDigitalValue() == 1 
                    && this.unlockSensorB.getDigitalValue() == 0;
            this.unlockSensorsInError = this.unlockSensor.getDigitalValue()
                    == this.unlockSensorB.getDigitalValue();
            
            
            inError = locked && unlocked || this.filterEngagedSensorsInError
                    || this.lockSensorsInError || this.unlockSensorsInError;

            if (inError) {
                lockStatus = FcsEnumerations.LockStatus.ERROR;
                String msg = createSensorsErrorMsg();
                this.raiseAlarm("FCS_AC_Sensor", msg);
                throw new FcsHardwareException(msg);
            } else if (locked && !unlocked) {
                lockStatus = FcsEnumerations.LockStatus.LOCKED;
            } else if (unlocked && !locked) {
                lockStatus = FcsEnumerations.LockStatus.UNLOCKED;
            } else if (!locked && !unlocked) {
                inTravel = true;
                lockStatus = FcsEnumerations.LockStatus.INTRAVEL;
            }

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }
    }
    
    private String createSensorsErrorMsg() {
        String message = name + " sensors ERROR:";
        if (filterEngagedSensorsInError) {
            message = message + " the 2 filterEngaged sensors return same value "+this.filterEngagedSensor.getDigitalValue();
        }
        if (lockSensorsInError) {
            message = message + " the 2 lock sensors return same value "+this.lockSensor.getDigitalValue();
        }  
        if (unlockSensorsInError) {
            message = message + " the 2 unlock sensors return same value "+this.unlockSensor.getDigitalValue();
        }        
        if (locked && unlocked) {
            message = message + " latch is LOCKED AND UNLOCKED.";
        }
        return message;
    }

    /**
     * Updates the field readCurrent of the latch in reading the CPU of the
     * controller. This method is used in the simulator mainly.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update latch current in reading controller.")
    public void updateCurrent()  {
        try {
            this.readCurrent = this.latchController.readCurrent();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> ERROR IN READING CONTROLLER:",ex);
        }
        this.publishData();
    }

    /**
     * Returns true if autochanger CANopen hardware is connected and ready.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if autochanger CANopen hardware is connected and ready.")
    @Override
    public boolean isCANDevicesReady() {
        return autochanger.isCANDevicesReady();
    }

    /**
     * Close this latch.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Close latch.")
    public String close()  {

        if (inError) {
            throw new RejectedCommandException(getName() + " is in ERROR state. Can't close it.");
        } else if (locked) {
            throw new RejectedCommandException(getName() + " is already LOCKED.");
        }
        autochanger.checkConditionsForClosingLatches();
        return this.executeAction(FcsEnumerations.MobileItemAction.CLOSE,
                timeoutForClosing);
    }

    /**
     * Open this latch.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open latch.")
    public String open()  {

        if (inError) {
            throw new RejectedCommandException(getName() + " is in ERROR state. Can't open it.");
        } else if (unlocked) {
            throw new RejectedCommandException(getName() + " is already UNLOCKED.");
        }
        autochanger.checkConditionsForOpeningLatches();
        return this.executeAction(FcsEnumerations.MobileItemAction.OPEN,
                timeoutForOpening);
    }

    /**
     * Return true if action is completed with success.
     * @param action
     * @return 
     */
    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case OPEN:
                return lockStatus == LockStatus.UNLOCKED;

            case CLOSE:
                return lockStatus == LockStatus.LOCKED;
             
            default:
                assert false : action;
        }
        return false;
    }

    /**
     * Read sensors and update State. 
     */
    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        updateStateWithSensors();
    }

    /**
     * Start Action OPEN or CLOSE
     * @param action
     * @throws FcsHardwareException 
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        
        switch (action) {
            case OPEN:
                latchController.enable();
                latchController.writeCurrent(this.currentToOpen);
                break;

            case CLOSE:
                latchController.enable();
                latchController.writeCurrent(-this.currentToOpen);
                break;
                
            default:
                assert false : action;
        }
    }

    /**
     * Stop action OPEN or CLOSE
     * @param action
     * @param delay
     * @throws FcsHardwareException 
     */
    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay)  {
        FCSLOG.debug(getName() + " is ABORTING action " + action.toString()
                + " within delay " + delay);
        this.latchController.off();
    }

    /**
     * @throws FcsHardwareException
     * @param action
     * @param delay 
     */
    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        //TODO : is there something different to do ? TBD
        abortAction(action, delay);
    }

    /**
     * Switch off controller after a CLOSE or OPEN action.
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void postAction(FcsEnumerations.MobileItemAction action)  {
        //because we don't want to let the controller on power
        this.latchController.off();
        this.publishData();
        FCSLOG.info(getName() + ":" + action.toString() + " completed - doing postAction.");
    }


    /**
     * Create an return an object StatusDataPublishedByAutochangerLatch for publication
     * on STATUS bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerLatch getStatusData() {
        return createStatusDataPublishedByLatch();
    }
    
    /**
     * Creates an return an object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerLatch createStatusDataPublishedByLatch() {
        StatusDataPublishedByAutochangerLatch status = new StatusDataPublishedByAutochangerLatch();
        status.setName(getName());
        status.setLockSensorValue(lockSensor.getDigitalValue());
        status.setUnlockSensorValue(unlockSensor.getDigitalValue());
        status.setFilterPresenceSensorValue(filterEngagedSensor.getDigitalValue());
        status.setLockStatus(lockStatus);
        status.setControllerInFault(controllerInFault);
        status.setLockSensorsInError(lockSensorsInError);
        status.setUnlockSensorsInError(unlockSensorsInError);
        status.setFilterEngagedSensorsInError(filterEngagedSensorsInError);
        status.setInError(inError);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Publish Data on the Status Bus.")
    @Override
    public void publishData() {
        StatusDataPublishedByAutochangerLatch status = this.getStatusData();
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), status));
    }
}
