/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.*;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This is a model for the autochanger trucks.
 * The autochanger trucks move a filter from STANDBY position to ONLINE position or HANDOFF position.
 * The ONLINE position is just in front of the focal front.
 * The STANDBY position is near the carousel.
 * The HANDOFF position is a position when the loader can hold a filter.
 * The trucks are moved by 2 controllers, a master and a slave.
 * Each controller has an absolute encoder.
 * 
 * @author virieux
 */
public class AutoChangerTwoTrucksModule extends MobileItemModule {

    AutochangerTruckModule truckXminus;
    AutochangerTruckModule truckXplus;
    

    private final int encoderRibbonMinValue = 7469165;
    private final int encoderRibbonMaxValue = 8410000;
    private final int minActualPositionValue = 0;
    private final int maxActualPositionValue = 960000;

    
    @ConfigurationParameter(description="STANDBY position in micron")
    private final int standbyPosition = 0;
    
    @ConfigurationParameter(description="HANDOFF position in micron")
    private final int handoffPosition = 958000;
    
    @ConfigurationParameter(description="ONLINE position in micron")
    private final int onlinePosition = 960000;
    
    @ConfigurationParameter(description="timeout for trucks motion from ONLINE to STANDBY in milliseconds")
    private final long timeoutForTrucksMotion = 120000;

    private int current;
    private int speed;
    private int position;
    private int deltaPosition;


    private int absoluteTargetPosition;
    private int relativeTargetPosition;

    private EPOSController linearRailMasterController;
    private EPOSController linearRailSlaveController;
    private AutoChangerModule autochanger;



    /**
     * true if the master controller is in fault - false if the user does a faultReset*
     */
    private boolean masterControllerInFault;

    /**
     * true if the slave controller is in fault - false if the user does a faultReset*
     */
    private boolean slaveControllerInFault;



    /**
     * true if the homing of the trucks have been done *
     */
    private boolean homingDone;
    
    /**
     * true if Handoff position sensors for TruckXminus and TruckXplus return value 1*
     */
    private boolean atHandoff;

    /**
     * true if Online position sensors for TruckXminus and TruckXplus return value 1 *
     */
    private boolean atOnline;

    /**
     * true if Standby position sensors for TruckXminus and TruckXplus return value 1 *
     */
    private boolean atStandby;  
    
     

    /**
     * Build a new AutoChangerTrucksModule with 6 sensors.
     * Initialize fields with initial default values.
     * 
     * @param truckXminus
     * @param truckXplus
     */
    public AutoChangerTwoTrucksModule(
            AutochangerTruckModule truckXminus,
            AutochangerTruckModule truckXplus

    ) {
        this.truckXminus = truckXminus;
        this.truckXplus = truckXplus;
        this.absoluteTargetPosition = 0;
        this.relativeTargetPosition = 0;
        this.masterControllerInFault = false;
        this.slaveControllerInFault = false;
        this.homingDone = false;
        this.current = 0;
        this.speed = 0;
        this.atHandoff = false;
        this.atOnline = false;
        this.atStandby = false;
    }

    /**
     * Return actual trucks position. 
     * Doesn't read again the position on the controller.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return actual trucks position. Doesn't read again the position on the controller.")
    public int getPosition() {
        return position;
    }

    /**
     * Return HANDOFF position.
     * Doesn't read again the position on the controller.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return HANDOFF position. Doesn't read again the position on the controller.")
    public int getHandoffPosition() {
        return handoffPosition;
    }

    /**
     * Return STANDBY position.
     * Doesn't read again the position on the controller.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return STANDBY position. Doesn't read again the position on the controller.")
    public int getStandbyPosition() {
        return standbyPosition;
    }

    /**
     * Return ONLINE position.
     * Doesn't read again the position on the controller.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return ONLINE position. Doesn't read again the position on the controller.")
    public int getOnlinePosition() {
        return onlinePosition;
    }

    /**
     * return encoderRibbonMinValue
     * Used by simulation.
     * @return 
     */
    public int getEncoderRibbonMinValue() {
        return encoderRibbonMinValue;
    }

    /**
     * return encoderRibbonMaxValue
     * Used by simulation.
     * @return 
     */
    public int getEncoderRibbonMaxValue() {
        return encoderRibbonMaxValue;
    }




    /**
     * Return the timeout for trucks motion.
     * If trucks motion duration is greater than timeoutForTrucksMotion, an ALERT is raised.
     * @return timeoutForTrucksMotion
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return timeout for trucks motion.")
    public long getTimeoutForTrucksMotion() {
        return timeoutForTrucksMotion;
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atHandoff is not being updated, it
     * returns its immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at handoff position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtHandoffPosition() {
        return position == this.handoffPosition;
    }

    /**
     * Returns the boolean field atOnline. If the atOnline boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atOnline is not being updated, it
     * returns its immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at ONLINE position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtOnlinePosition() {
        return position == this.onlinePosition;
    }

    /**
     * Returns the boolean field atStandby. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atStandby is not being updated, it
     * returns its immediatly..
     *
     * @return atStandby
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at STANDBY position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtStandbyPosition() {
        return position == this.standbyPosition;
    }

    /**
     * Return true if the homing of the trucks has been done.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the homing of the trucks has been done.")
    public boolean isHomingDone() {
        return homingDone;
    }
    
    /**
     * Return true if TruckXminus position sensors or TruckXminus position sensors are in error.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if TruckXminus position sensors or TruckXminus position sensors are in error.") 
    public boolean isPositionSensorsInError() {
        return truckXminus.isPositionSensorsInError() || truckXplus.isPositionSensorsInError();
    }   

    @Override
    public void initModule() {
        super.initModule();
        ComponentLookup lookup = getSubsystem().getComponentLookup();
        autochanger = (AutoChangerModule) lookup.getComponentByName("autochanger");
        linearRailMasterController = (EPOSController) lookup.getComponentByName("linearRailMasterController");
        linearRailSlaveController = (EPOSController) lookup.getComponentByName("linearRailSlaveController");
        //listens to its controllers to detect the controller's faultReset
        //or the emergency messages coming from the controller.
        if (linearRailMasterController instanceof Observable) {
            this.listens((Observable) linearRailMasterController);
        }
        if (linearRailSlaveController instanceof Observable) {
            this.listens((Observable) linearRailSlaveController);
        }
    }

    @Override
    public void tick() {
        this.publishData();
    }

    /**
     * This method is called by method checkHardware in AutoChangerModule during
     * INITIALIZATION phase.
     *
     * @return
     * @throws HardwareException
     */
    @Override
    //tested on CPPM testbench in september 2015
    public TreeWalkerDiag checkHardware() throws HardwareException {
        super.checkHardware();

        FCSLOG.debug(getName() + " checking hardware.");
        try {
            linearRailSlaveController.initializeAndCheckHardware();
            linearRailMasterController.initializeAndCheckHardware();
            configureControllers();
            homing();
            linearRailSlaveController.enable();
            linearRailMasterController.enable();

        } catch (SDORequestException ex) {
            FCSLOG.error(getName(),ex);
        } catch (FailedCommandException ex) {
            throw new HardwareException(true, ex);
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName(),ex);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false, ex);
        }
        updatePosition();
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * Configure Autochanger trucks master and slave controllers.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Configure Autochanger trucks master and slave controllers.",
            alias = "initControllers")
    //tested on CPPM testbench in september 2015
    public void configureControllers()  {
        FCSLOG.info(getName() + " Begin configuration of the controllers");
        this.linearRailMasterController.shutdown();
        this.linearRailSlaveController.shutdown();
        configureDigitalInputOfLinearRails(linearRailMasterController);
        configureDigitalInputOfLinearRails(linearRailSlaveController);
        configureDigitalOutputOfLinearRails(linearRailMasterController);
        configureDigitalOutputOfLinearRails(linearRailSlaveController);
        FCSLOG.info(getName() + " End configuration of the controllers");
        //TODO tester si la config est bonne.
    }

    /**
     * Return true if both controllers are initialized and homing of trucks are done.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if both controllers are initialized and homing of trucks are done.")
    public boolean isInitialized() {
        return linearRailMasterController.isInitialized()
                && linearRailSlaveController.isInitialized()
                && homingDone;
    }
    
    /**
     * This method sends Exceptions if the conditions for trucks motion are not filled.
     * Autochanger trucks motion is allowed if :
     * - controllers are initialized
     * - homing is done
     * - trucks are empty OR 
     *      (loaded with a filter but latches are closed AND carousel or loader doesn't hold filter)
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the motion of trucks are allowed.")
    public void checkConditionsForTrucksMotion()  {
        
        if (!linearRailSlaveController.isInitialized() || !linearRailMasterController.isInitialized()) {
            String msg = getName() + " can't move autochanger trucks because controllers are not both initialized:\n";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
            
        }

        if (!homingDone) {
            throw new RejectedCommandException(getName()
                    + " can't be moved because the homing has not been done yet. ");
        }
        
        autochanger.checkFilterSafetyBeforeMotion();
    }

    /**
     * Move Autochanger trucks to the Handoff position.
     * If Autochanger trucks are already at HANDOFF position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Handoff position.",
            alias = "goToHandoff")
    //tested on CPPM testbench in september 2015
    public void goToHandOff()  {
        moveToAbsoluteTargetPosition(this.handoffPosition);
    }

    /**
     * Move Autochanger trucks to the Online position.
     * If Autochanger trucks are already at ONLINE position, it does nothing.
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Online position.")
    //tested on CPPM testbench in september 2015
    public void goToOnline()  {
        moveToAbsoluteTargetPosition(this.onlinePosition);
    }

    /**
     * Move Autochanger trucks to the Standby position.
     * If Autochanger trucks are already at STANDBY position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Standby position.")
    //tested on CPPM testbench in september 2015
    public void goToStandby()  {
        moveToAbsoluteTargetPosition(this.standbyPosition);
    }

    /**
     * Move the trucks to an absolute position given as argument.
     *
     * @param targetPosition
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the absolute position given as argument.",
            alias = "mobeABSPos")
    //tested on CPPM testbench in september 2015
    public void moveToAbsoluteTargetPosition(int targetPosition)  {
        
        if (targetPosition < minActualPositionValue || targetPosition > maxActualPositionValue) {
            throw new IllegalArgumentException(getName() + "argument has to be between "
                    + minActualPositionValue + " and " + maxActualPositionValue);
        }
        
        this.updatePosition();
        if (position == targetPosition) {
            FCSLOG.info(getName()
                    + " is already at target position=" + targetPosition);
        } else {
        
            checkConditionsForTrucksMotion();

            this.absoluteTargetPosition = targetPosition;

            FCSLOG.info(getName() + " going to absolute position: " + absoluteTargetPosition);

            this.executeAction(MOVE_TO_ABSOLUTE_POSITION, timeoutForTrucksMotion);
        }
    }

    /**
     * Move the trucks to a relative position given as argument.
     *
     * @param relativePosition
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the relative position given as argument.",
            alias = "moveRELPos")
    //tested on CPPM testbench in september 2015
    //ONLY for small values for test purpose to realign the trucks
    public void moveToRelativeTargetPosition(int relativePosition)  {
        
        checkConditionsForTrucksMotion();

        if (relativePosition < -2000 || relativePosition > 2000) {
            throw new IllegalArgumentException("relativePosition="+relativePosition +" Relative position value has to be between -2000 and 2000");
        }
        
        linearRailMasterController.changeMode(EposMode.PROFILE_POSITION);
        updatePosition();

        this.relativeTargetPosition = position + relativePosition;
        
        if (relativeTargetPosition < minActualPositionValue || relativeTargetPosition > maxActualPositionValue) {
            throw new IllegalArgumentException(name + " actual position + argument has to be between "
                    + minActualPositionValue + " and " + maxActualPositionValue 
                    + " relativeTargetPosition="+relativeTargetPosition);
        }

        FCSLOG.info(getName() + " going to relative position: " + relativeTargetPosition);
        this.executeAction(MOVE_TO_RELATIVE_POSITION, timeoutForTrucksMotion);
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        //TODO quand on aura les 2 encodeurs, il faudra vérifier que les 
        //paramètres de PROFILE_POSITION sont identiques sur les 2 contrôleurs.
        
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                this.enableControllersAndReleaseBrake();
                linearRailMasterController.writeTargetPosition(this.absoluteTargetPosition);
                linearRailMasterController.writeControlWord("3F");
                break;

            case MOVE_TO_RELATIVE_POSITION:
                this.enableControllersAndReleaseBrake();
                linearRailMasterController.writeTargetPosition(this.relativeTargetPosition);
                linearRailMasterController.writeControlWord("7F");
                break;
                
            default:
                assert false : action;
        }
    }
    
    /**
     * This method is used in the startAction method.
     * It is was has to be done before moving the trucks. 
     */
    private void enableControllersAndReleaseBrake()  {
        linearRailMasterController.changeMode(EposMode.PROFILE_POSITION);
        linearRailSlaveController.changeMode(EposMode.MASTER_ENCODER);
        if (!linearRailMasterController.isEnabled()) {
            linearRailMasterController.enable();
        }
        if (!linearRailSlaveController.isEnabled()) {
            linearRailSlaveController.enable();
        }
        linearRailMasterController.releaseBrake();
        linearRailSlaveController.releaseBrake();
    }

    @Override
    //TODO test on CPPM test bench
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay)  {
        FCSLOG.info("ABORTING action: "+ action.toString());

        this.linearRailMasterController.off();
        this.linearRailSlaveController.off();
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay)  {
        this.linearRailMasterController.quickStop();
        this.linearRailSlaveController.quickStop();
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void postAction(FcsEnumerations.MobileItemAction action) {
        //because we don't want to let the controller on power
        this.linearRailSlaveController.disable();
        this.linearRailMasterController.disable();
        FCSLOG.info(getName() + ":" + action.toString() + " completed - doing postAction: checking position sensors.");
        
        if (action == MOVE_TO_ABSOLUTE_POSITION) {
            checkPositionSensors();
        }
    }
    
    /**
     * This method is called in the postAction method to check that after a motion to well known position 
     * as STANDBY, HANDOFF and ONLINE, position sensors confirm the trucks position.
     * Otherwise it throws a FailedCommandException.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void checkPositionSensors()  {
        /*Read position sensors and check that values return are consistant with trucks position
        otherwise throw an Exception.*/
        this.updateStateWithSensors();
        if (this.absoluteTargetPosition == this.standbyPosition && !this.atStandby) {
            throw new FailedCommandException(name
                            + ": check with sensors: STANDBY sensors don't confirm trucks position.");
            
        } else if (this.absoluteTargetPosition == this.handoffPosition && !this.atHandoff) {
            FCSLOG.info("this.absoluteTargetPosition="+this.absoluteTargetPosition);
            FCSLOG.info("this.handoffPosition="+this.handoffPosition);
            FCSLOG.info("this.atHandoff="+this.atHandoff);
            throw new FailedCommandException(name
                            + ": check with sensors: HANDOFF sensors don't confirm trucks position.");
            
        } else if (this.absoluteTargetPosition == this.onlinePosition && !this.atOnline) {
            throw new FailedCommandException(name
                            + ": check with sensors: ONLINE sensors don't confirm trucks position.");
        }
    }

    /**
     * Do the homing of the trucks which consists in configuring the 2 controllers
     * in order to configure the encoders and set the point "0".
     * @throws RejectedCommandException
     * @throws FcsHardwareException 
     */
    //tested on CPPM testbench in september 2015
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Read actual position and do the homing of the master controller"
            + "with this actual position.")
    public void homing()  {

        if (!linearRailSlaveController.isInitialized() || !linearRailMasterController.isInitialized()) {
            FCSLOG.error(getName() + " cant' do homing because controllers are not both initialized:");
            FCSLOG.info(getName() + linearRailSlaveController.toString());
            FCSLOG.info(getName() + linearRailMasterController.toString());
        }

        //TODO check if parameters are OK in checkHardware method
        linearRailMasterController.changeMode(EposMode.PROFILE_POSITION);
        linearRailMasterController.writeParameters(EposMode.PROFILE_POSITION);
        linearRailMasterController.enable();

        linearRailSlaveController.changeMode(EposMode.MASTER_ENCODER);
        linearRailSlaveController.writeParameters(EposMode.PROFILE_POSITION);
        linearRailSlaveController.enable();

        FCSLOG.debug(getName() + "==> BEGIN homing of the trucks");
        int ssiPosition = linearRailMasterController.readSSIPosition();
        int offset = ssiPosition - encoderRibbonMinValue;
        FCSLOG.debug(getName() + " position=" + ssiPosition + ",offset=" + offset);

        //defineAbsolutePosition change EPOS mode to HOMING
        linearRailMasterController.defineAbsolutePosition(offset);
        linearRailMasterController.off();
        linearRailMasterController.changeMode(EposMode.PROFILE_POSITION);
        homingDone = true;
        FCSLOG.debug(getName() + " ==> END homing of the trucks");
    }

    /**
     * Updates the field position of the carrier in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    //tested on CPPM testbench in september 2015
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update trucks position in reading controller.")
    public void updatePosition()  {
        this.truckXminus.updatePosition();
        this.truckXplus.updatePosition(); 
        this.position = linearRailMasterController.readPosition();
        //TODO if there is a difference between this 2 positions, launch an ALERT.
        this.deltaPosition = linearRailMasterController.readPosition() - linearRailSlaveController.readPosition();
        this.publishData();
    }

    /**
     * For the GUI and end users for test purpose. Update positon in reading the
     * master controller and returns the value read.
     *
     * @return trucks position
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    //tested on CPPM testbench in september 2015
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update trucks position in reading controller and return position.")
    public int readPosition()  {
        updatePosition();
        return position;
    }

    /**
     * @return true if both controllers are initialized and homing of trucks are done.
     */
    @Override
    //tested on CPPM testbench in september 2015
    public boolean isCANDevicesReady() {
        //commented out for CPPM testbench
        //return autochanger.isCANDevicesReady();
        return this.isInitialized();
    }

    @Override
    //tested on CPPM testbench in september 2015
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                return this.position == this.absoluteTargetPosition;

            case MOVE_TO_RELATIVE_POSITION:
                return this.position == this.relativeTargetPosition;
                
            default :
                assert false: action;
        }
        return false;
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        try {
            linearRailMasterController.checkFault();
            linearRailSlaveController.checkFault();
            this.position = this.linearRailMasterController.readPosition();
            FCSLOG.debug(getName() + " position=" + this.position);
            //Commented out for CPPM testbench because autochanger doesn't exist.
            this.updateStateWithSensors();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> SDO ERROR IN READING CONTROLLER:",ex);
        } catch (SDORequestException ex) {
            FCSLOG.error(getName() + "=> SDO ERROR IN READING CONTROLLER:",ex);
        }
    }

    /**
     * This methode delegates updateStateWithSensors to AutoChangerModule.
     * Then autochanger.updateStateWithSensors() calls updateStateWithSensors(String[] readHexaValues).
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update Trucks state in reading sensors.")
    public void updateStateWithSensors()  {
        autochanger.updateStateWithSensors();
    }

    void updateStateWithSensors(String[] readHexaValues) {
        truckXminus.updateStateWithSensors(readHexaValues);
        truckXplus.updateStateWithSensors(readHexaValues);
        this.atHandoff = truckXminus.isAtHandoff() && truckXplus.isAtHandoff();
        this.atOnline = truckXminus.isAtOnline() && truckXplus.isAtOnline();
        this.atStandby = truckXminus.isAtStandby() && truckXplus.isAtStandby();
    }

    /**
     * This configures the digital inputs of the EPOS controller to work with a
     * holding brake. This configuration is required for the controllers of the
     * Linear Rails of the autochanger. Digital Inputs activated for Linear
     * Rails : Drive Enable Input=01 Mask=Enabled Polarity=HighActive QuickStop
     * Input=02 Mask=Enabled Polarity=HighActive
     *
     *          *********************************************
     * 1 - Configure the Digital Input Functionnality index 0x2070 subindex 01
     * to 04 Functionnality tab Value = 15 General Purpose A Value = 14 General
     * Purpose B Value = 13 General Purpose C Value = 12 General Purpose D Value
     * = 11 General Purpose E Value = 10 General Purpose F Value = 9 General
     * Purpose G Value = 8 General Purpose H Value = 7 General Purpose I Value =
     * 6 General Purpose J Value = 5 QuickStop 1 Value = 4 Drive enable 1 Value
     * = 3 Position marker 0 Value = 2 Home switch 0 Value = 1 Positive Limit
     * switch 0 Value = 0 Negative Limit switch 0
     * ********************************************* 2 - Configure the Digital
     * Input Polarity index 0x2071 subindex 03 Value=0 HighActive Value=1 Low
     * Active ********************************************* 3 - Configure the
     * Digital Input Mask index 0x2071 subindex 02 Value=0 HighActive Value=1
     * Low Active ******************************************** 4 - Configure the
     * Digital Input Execution Mask index 0x2071 subindex 04 Used to suppress
     * execution of the digital input functionalities. The function (Negative or
     * Positive Limit Switch Error Routine) will be executed when the associated
     * bit in functionalities state register goes high and the bit in this
     * execution mask is set. Functionnality tab Value = 15 General Purpose A 0
     * Value = 14 General Purpose B 0 Value = 13 General Purpose C 0 Value = 12
     * General Purpose D 0 Value = 11 General Purpose E 0 Value = 10 General
     * Purpose F 0 Value = 9 General Purpose G 0 Value = 8 General Purpose H 0
     * Value = 7 General Purpose I 0 Value = 6 General Purpose J 0 Value = 5
     * QuickStop 1 Value = 4 Drive enable 1 Value = 3 Position marker 0 Value =
     * 2 Home switch 0 Value = 1 Positive Limit switch 0 Value = 0 Negative
     * Limit switch 0
     *
     * To configure the autochanger linear rails controllers. This method is to
     * be executed during initialization phase.
     *
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the autochanger linear rails controllers.")
    //tested on CPPM testbench in september 2015
    public static void configureDigitalInputOfLinearRails(EPOSController ctl)  {
        //Drive Enable
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalInput1, "0004");

        //QuickStop
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalInput2, "0005");

        //DigitalInputPolarity 
        //Value=0 HighActive
        //Value=1 Low Active
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityPolarity, "0");

        //DigitalInputFonctionnalitiesMask
        //Used to display the state of the digital input functionalities. 
        //If a bit is set to “1”, the functionality state will be displayed.
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityMask, "30");

        //DigitalInputFonctionnalitiesExecutionMask
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityExecutionMask, "30");

    }

    /**
     * Configuration of an EPOS controller to work with a holding brake. This is
     * used for controllers of the Autochanger Linear Rails.
     *
     *          * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Digital Outputs
     * activated for Linear Rails are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ General_A Output=01
     * Mask=Enabled Polarity=HighActive Ready/Fault Output=02 Mask=Enabled
     * Polarity=HighActive Holding Brake Output=03 Mask=Enabled
     * Polarity=LowActive ******************************************** 1- First
     * of all, mask all outputs index 0x2079 subindex 01
     * ********************************************** 2- Configure the Digital
     * Output Fonctionnalities index 0x2079 subindex 01 to 03 Functionnality tab
     * Value = 15 General Purpose Out_A Value = 14 General Purpose Out_B Value =
     * 13 General Purpose Out_C Value = 12 General Purpose Out_D Value = 11
     * General Purpose Out_E Value = 10..8 not used Value = 7..3 reserved Value
     * = 2 Holding Brake Value = 1 Position compare Value = 0 Ready/Fault
     * ************************************************************************************
     * 3 - Configure the Digital Output Functionnality Polarity index 0x2078
     * subindex 0x03 Value = 0 associated output not change => HighActive Value
     * = 1 associated output inverted => LowActive Tab Digital Output
     * Functionnality Mask Value Bit_15 General Purpose Out_A 0 Bit_14 General
     * Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12 General Purpose
     * Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General Purpose Out_F 0
     * Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H 0 Bit_7
     * reserved 0 Bit_6 reserved 0 Bit_5 reserved 0 Bit_4 reserved 0 Bit_3
     * reserved 0 Bit_2 Holding Brake 1 (read only) Bit_1 Position Compare 0
     * (read only) Bit_0 Ready/Fault 0 (read only)
     * ********************************************************************************
     * 4 - Configure the Digital Output Functionnality Mask index 0x2078
     * subindex 0x02 Value = 0 functionnality not activatd Value = 1
     * functionnality activated
     * ********************************************************************************
     * Digital Output Functionnality Mask Value Bit_15 General Purpose Out_A 1
     * Bit_14 General Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12
     * General Purpose Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General
     * Purpose Out_F 0 Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H
     * 0 Bit_7 reserved 0 Bit_6 reserved 0 Bit_5 reserved 0 Bit_4 reserved 0
     * Bit_3 reserved 0 Bit_2 Holding Brake 1 Bit_1 Position Compare 0 Bit_0
     * Ready/Fault 1
     * ********************************************************************************
     * To configure the autochanger linear rails controllers. This method is to
     * be executed during initialization phase.
     *
     * @param ctl controller
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the autochanger linear rails controllers.")
    //tested on CPPM testbench in september 2015
    public static void configureDigitalOutputOfLinearRails(EPOSController ctl)  {
        //1-Mask all outputs
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityMask, "0");

        //2-Configuration of Digital Output Fonctionnalities
        //General_A
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput1, "000F");
        //Ready/Fault
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput2, "0");
        //Holding Brake
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput3, "0002");

        //3-Configuration of Digital Output Fonctionnality Polarity
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityPolarity, "0004");

        //4-Configuration of Digital Output Fonctionnality Mask
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityMask, "8005");
    }

    /**
     * Build the class StatusDataPublishedByAutoChangerTrucks
     *
     * @return status data
     */
    public StatusDataPublishedByAutoChangerTrucks getStatusData() {
        return createStatusDataPublishedByAutoChangerTrucks();
    }
    
    /**
     * Creates an object to be published on the status bus.
     * @return 
     */
    public StatusDataPublishedByAutoChangerTrucks createStatusDataPublishedByAutoChangerTrucks() {
        StatusDataPublishedByAutoChangerTrucks s = new StatusDataPublishedByAutoChangerTrucks();
        s.setPosition(position);
        s.setDeltaPosition(deltaPosition);
        s.setCurrent(current);
        s.setSpeed(speed);
        s.setHomingDone(homingDone);
        s.setAtHandoff(atHandoff);
        s.setAtOnline(atOnline);
        s.setAtStandby(atStandby);
        s.setInError(isPositionSensorsInError());
        return s;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), getStatusData()));
    }

}
