/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

package org.lsst.ccs.subsystems.fcs.testbench.LPC;

import java.math.BigInteger;
import java.util.Map;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenDevice;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 *
 * @author virieux
 */
public class CanOpenDigitaxVariator  extends CanOpenDevice {
    
     /**
     * Here are the parameters that we can set in an Digitax ST variator.
     * cf Digitax ST document n° 4489_en.pdf
     */
    public enum Parameter {
        target_torque("6071","00", "2",true),
        torque_actual_value("6077","00","2",true),
        motor_rated_current("6075","00", "4",false),       
        torque_slope("6087","00", "4",true),
        torque_profile_type("6088","00", "1",true),
        vl_target_velocity("6042","00", "2",true),
        vl_velocity_demand("6043","00", "2",true),
        vl_control_effort("6044","00", "2",true),
        vl_velocity_min_max_amount("6046","00","4",false),
        vl_velocity_acceleration("6048","01", "4",true),
        vl_velocity_deceleration("6049","02", "4",true),
        vl_dimension_factor("604C","00","4",true),
        vl_pole_number("604D","00", "1",false),
        ;
        
        private final String index;
        private final String subindex;
        /* Size of the parameter : numbers of bytes*/
        private final String size; 
        private final boolean signed;


        private Parameter(String index, String subindex, String size, boolean signed) {
            this.index = index;
            this.subindex = subindex;
            this.size = size;
            this.signed = signed;

        }
        
        public String display() {
            StringBuilder sb = new StringBuilder(this.toString());
            sb.append("/index=").append(this.index);
            sb.append("/subindex=").append(this.subindex);
            sb.append("/size=").append(this.size).append(" bytes");
            if (signed) sb.append("/SIGNED"); else sb.append("/UNSIGNED");
            return sb.toString();
        }
        
        
    }
    


    
    /**
     * Mode for the variator used in the scale1 prototype.
     * For each mode, some parameters have to be set : field parameters
     */
    public enum DigitaxMode {
        VELOCITY("2", CanOpenDigitaxVariator.Parameter.vl_target_velocity, 
            CanOpenDigitaxVariator.Parameter.vl_velocity_demand, 
            CanOpenDigitaxVariator.Parameter.vl_control_effort,
            CanOpenDigitaxVariator.Parameter.vl_velocity_min_max_amount, 
            CanOpenDigitaxVariator.Parameter.vl_velocity_acceleration, 
            CanOpenDigitaxVariator.Parameter.vl_velocity_deceleration,            
            CanOpenDigitaxVariator.Parameter.vl_dimension_factor,
            CanOpenDigitaxVariator.Parameter.vl_pole_number),
        TORQUE_PROFILE("4", CanOpenDigitaxVariator.Parameter.target_torque, 
            CanOpenDigitaxVariator.Parameter.torque_actual_value,
            CanOpenDigitaxVariator.Parameter.motor_rated_current, 
            CanOpenDigitaxVariator.Parameter.torque_slope,
            CanOpenDigitaxVariator.Parameter.torque_profile_type); 
     
        private final String modeInHexa;
        private final CanOpenDigitaxVariator.Parameter[] parameters;

        private DigitaxMode(String modeToHexa, CanOpenDigitaxVariator.Parameter... params) {
            this.modeInHexa = modeToHexa;
            this.parameters = params.clone();
        }
        
        String getModeInHexa() {
            return this.modeInHexa;
        } 
        
        /**
         * Retrieve and returns a mode which hexa code is given as an argument. 
         * @param hexa mode code in hexa
         * @return mode
         */
        public static DigitaxMode getMode(String hexa) {
            DigitaxMode[] modes = DigitaxMode.values();
            DigitaxMode digitaxMode = null;
            for (DigitaxMode mode : modes) {
                 if (mode.getModeInHexa().equals(hexa.toUpperCase())) {
                     digitaxMode = mode;
                 }
            }
            if (digitaxMode == null) {
                throw new IllegalArgumentException(hexa + ": this mode doesn't exist in DigitaxVariator.DigitaxMode ENUM");
            } else return digitaxMode;
            
        }
    }
    
    protected DigitaxMode mode;
    
    /* Parameters */
    private final Map<String,Integer> paramsForVelocity;   
    private final Map<String,Integer> paramsForTorque_Profile;
    
    //Used because we have to wait during the time needed to enable the actuator (very short but not null !)
    //of the clamp
    private final Lock lock = new ReentrantLock();
    private final Condition enablingCompleted = lock.newCondition();
    
     /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean enabling = false;
    

    public CanOpenDigitaxVariator(String aName, int aTickMillis, 
            String nodeID, String serialNB,
            Map<String, Integer> paramsForVelocity, 
            Map<String, Integer> paramsForTorque_Profile) {
        super(aName, aTickMillis, nodeID, serialNB);
        this.paramsForVelocity = paramsForVelocity;
        this.paramsForTorque_Profile = paramsForTorque_Profile;
    }
    
    
    
    @Override
    public void initModule() {
        //Nothing to do here.
    }
    
    /**
     * Changes the mode to the new mode given as a String argument.
     * @param modeInString
     * @return
     * @throws SDORequestException 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING3, 
            description="Change Digitax variator mode .")
    public String changeMode(String modeInString) throws SDORequestException, 
            FcsHardwareException {
        CanOpenDigitaxVariator.DigitaxMode newMode = CanOpenDigitaxVariator.DigitaxMode.valueOf(modeInString.trim().toUpperCase());
        return changeMode(newMode);
    }
    
    /**
     * This methods changes the mode to the new mode given as an argument.
     * It writes the new mode in the CPU of the EPOS controller and updates the field mode.
     * @param newMode
     * @return
     * @throws SDORequestException 
     */
        public String changeMode(CanOpenDigitaxVariator.DigitaxMode newMode) 
                throws SDORequestException, FcsHardwareException {
        this.writeSDO("6060", "0", "1", newMode.modeInHexa);
        this.mode = newMode;
        return name + " mode has been changed to: " + newMode.toString() + ". Please check parameters associated with this mode.";
    }
        
            
    /**
     * Reads the Digitax mode in the CPU of the Digitax device, updates the field mode and returns
     * the mode as a String.
     * @return
     * @throws SDORequestException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException 
     */
    public String readMode() throws SDORequestException, ShortResponseToSDORequestException, 
            FcsHardwareException {
        String modeInHexa = this.readSDO("6061", "0").trim().toUpperCase();
        DigitaxMode readMode = DigitaxMode.getMode(modeInHexa);
        this.mode = readMode;
        return readMode.toString();
    }
        
    
    /**
     * Displays the list of parameters that we have to define for a given mode.
     * @param modeInString
     * @return 
     */
    public static String displayListParameters(String modeInString) {
        StringBuilder sb = new StringBuilder("List of parameters for mode: "); sb.append(modeInString);sb.append("\n");
        CanOpenDigitaxVariator.Parameter[] params = CanOpenDigitaxVariator.DigitaxMode.valueOf(modeInString).parameters;
        for (CanOpenDigitaxVariator.Parameter param : params) {
            sb.append(param.display());sb.append("\n");
        }
        return sb.toString(); 
    }
    
     /**
     * Displays the list of parameters.
     * @return 
     */
    public static String displayListParameters() {
        StringBuilder sb = new StringBuilder("List of parameters : "); sb.append("\n");
        CanOpenDigitaxVariator.Parameter[] params = CanOpenDigitaxVariator.Parameter.values();
        for (CanOpenDigitaxVariator.Parameter param : params) {
            sb.append(param.display());sb.append("\n");
        }
        return sb.toString(); 
    }
    
    public int readTorque_actual_value() throws SDORequestException, 
            ShortResponseToSDORequestException,
            FcsHardwareException {
        return readParameter("torque_actual_value");
    }
    
    public String writeTorque(int value) throws SDORequestException, 
            FcsHardwareException {
        return writeParameter(Parameter.target_torque,value);
    }
    
    
        /**
     * This methods reads in the CPU of the Digitax Variator the values of the parameters for a given mode.
     * @param modeInString
     * @return
     * @throws SDORequestException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException 
     */
    public String readParameters(String modeInString) throws SDORequestException, 
            ShortResponseToSDORequestException,
            FcsHardwareException {
        StringBuilder sb = new StringBuilder("Value set for the parameters for the mode : ");sb.append(modeInString);sb.append("\n");
        CanOpenDigitaxVariator.Parameter[] params = CanOpenDigitaxVariator.DigitaxMode.valueOf(modeInString).parameters;
        for (CanOpenDigitaxVariator.Parameter param : params) {
            
            String valueInHexa = readSDO(param.index,param.subindex);
            int valueInt = Integer.parseInt(valueInHexa, 16);
            sb.append(param.toString());
            sb.append("=");sb.append(valueInt);sb.append("\n");
        }
        return sb.toString();
    }
    
    public String readParameters() throws SDORequestException, 
            ShortResponseToSDORequestException,
            FcsHardwareException {
        String modeInString = readMode();
        return readParameters(modeInString);
    }
    
        /**
     * Reads in the Digitax Variator CPU the value of the Parameter which parameter 
     * name is given as argument.
     * @param parameterName
     * @return value of the parameter in decimal format.
     * @throws SDORequestException 
     */
    public int readParameter(String parameterName) throws SDORequestException, 
            ShortResponseToSDORequestException, 
            FcsHardwareException {
        CanOpenDigitaxVariator.Parameter param = CanOpenDigitaxVariator.Parameter.valueOf(parameterName);
        return readParameter(param);
    }
    
    /**
     * Reads in the Digitax Variator CPU the value of the Parameter.
     * @param param
     * @return value of the parameter in decimal format.
     * @throws SDORequestException 
     */
    public int readParameter(CanOpenDigitaxVariator.Parameter param) 
            throws SDORequestException, 
            ShortResponseToSDORequestException,
            FcsHardwareException {
        String valueInHexa = readSDO(param.index,param.subindex);
        return new BigInteger(valueInHexa,16).intValue();
    }
    
     /**
     * Print the configured parameters for this Digitax for a mode name given in argument.
     * This methods doesnt't read the CPU of the Digitax.
     * So if no writeParameters(modeInString) command has been done before,
     * the values in the Digitax variator CPU can be different.
     * @param modeInString
     * @return 
     */
    public String printParameters(String modeInString){
        DigitaxMode aMode = DigitaxMode.valueOf(modeInString.trim().toUpperCase());
        Map<String,Integer> paramsMap;
        switch(aMode) {
           case VELOCITY : 
               paramsMap = this.paramsForVelocity;
               break;
           case TORQUE_PROFILE : 
               paramsMap = this.paramsForTorque_Profile;
               break;
           default: 
               throw new IllegalArgumentException(modeInString 
                       + " is a bad value for Digitax mode.");
        }
        return printParameters(paramsMap);
    }
    
    /**
     * Print the parameters map given as argument.
     * @param paramMap
     * @return 
     */
    public String printParameters(Map<String,Integer> paramMap) {
        StringBuilder sb = new StringBuilder(name);
        sb.append("Parameters in configuration file:\n");
        if (paramMap == null) {
            return "Empty Parameters List";
        } else {
            for (Map.Entry<String,Integer> entry : paramMap.entrySet()) {
                String paramName = entry.getKey();
                int value = entry.getValue();  
                sb.append("key=");sb.append(paramName);
                sb.append("/value=");sb.append(value);sb.append("\n");
            }
            return sb.toString();
        }
    }
    
    public void writeParameters(String modeInString) throws SDORequestException, 
            ErrorInCommandExecutionException,
            FcsHardwareException {
        DigitaxMode aMode = DigitaxMode.valueOf(modeInString);
        writeParameters(aMode);
    }
    
    /**
     * This methods writes in the CPU of the EPOS devis the values of the parameters 
     * set for the mode.
     * Usually the values of the parameters are given by the configuration system.
     * @param mode
     * @throws SDORequestException 
     * @throws org.lsst.ccs.messaging.ErrorInCommandExecutionException 
     */
     public void writeParameters(DigitaxMode mode) throws SDORequestException, 
             ErrorInCommandExecutionException,
             FcsHardwareException {
        String modeInString = mode.toString();
        Map<String,Integer> paramsMap = null;
        switch(mode) {
           case VELOCITY : 
               paramsMap = this.paramsForVelocity;
               break;
           case TORQUE_PROFILE : 
               paramsMap = this.paramsForTorque_Profile;
               break;
           default:
               assert false;
        }
        if (paramsMap == null) throw new ErrorInCommandExecutionException(name 
                + "parameters for mode :" + modeInString + "are NULL");
        if (paramsMap.isEmpty()) throw new ErrorInCommandExecutionException(name 
                + "parameters for mode :" + modeInString + "are not defined.");
        writeParameters(paramsMap);
        
    }
    
    /**
     * Write to the CPU of the Digitax Variator a map of parameters.
     * @param paramMap
     * @throws SDORequestException 
     */
        public void writeParameters(Map<String,Integer> paramMap) 
                throws SDORequestException, FcsHardwareException {
        for (Map.Entry<String,Integer> entry : paramMap.entrySet()) {
            String paramName = entry.getKey();
            int value = entry.getValue(); 
            writeParameter(paramName,value);
        }
    }
    
    
    /**
     * This method writes a parameter in the CPU of the Digitax Variator.
     * @param parameterName the name of the parameter
     * @param value in decimal
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException 
     */
     @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING3, 
             description="Set Digitax Variator parameter with the value given as argument.")
    public String writeParameter(String parameterName, int value) 
            throws SDORequestException, FcsHardwareException {
        CanOpenDigitaxVariator.Parameter param = CanOpenDigitaxVariator.Parameter.valueOf(parameterName);
        return this.writeParameter(param, value);
    }
    
    public String writeParameter(CanOpenDigitaxVariator.Parameter param, int value) 
            throws SDORequestException, FcsHardwareException {
         writeSDO(param.index,param.subindex,param.size,Integer.toHexString(value));
        return String.format("%s has been written to the Digitax Variator CPU, value=%d", 
                param.toString(),value);
    }
    
    @Override
    public String initializeAndCheckHardware() throws FcsHardwareException {
//        this.initialized = true;
        return getName() + ": no hardware initialization for this device to be done.";
    }

    
    public boolean isEnable() throws SDORequestException, 
            ShortResponseToSDORequestException, FcsHardwareException {
        lock.lock();
        
        try {
            while(this.enabling) {
                try {
                    this.enablingCompleted.await();
                } catch (InterruptedException ex) {
                    FCSLOG.debug(getName() + " INTERRUPTED during waiting for the actuator beeing enable");
                }

            }
        String controlWordInHexa = readControlWord();
        int controlWord = Integer.parseInt(controlWordInHexa, 16);
        return (controlWord == 15); /*F in hexa*/
        } finally {
            lock.unlock();
        }
    }
    
    public void faultReset() throws SDORequestException, 
            FcsHardwareException {
        writeControlWord("80");
    }
  
    /**
     * This methods enable the actuator : i.e. this makes the actuator able to receive commands.
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException 
     */
    public String enable() throws SDORequestException, FcsHardwareException {
        lock.lock();
              
        try {
            this.enabling = true;
            shutdown();
            switchOnEnableOperation();
            return name + " ENABLE";
            
        } finally {

            this.enabling = false;
            this.enablingCompleted.signal(); 
            lock.unlock();
        }
    }
    
    
    public String shutdown() throws SDORequestException, FcsHardwareException{      
        writeControlWord("6");
        //TODO test the return code for the command writeSDO
        return name + " DISABLE";
    }
    
    public void switchOnEnableOperation() throws SDORequestException, FcsHardwareException {
        writeControlWord("F");
    }
    
    /**
     * This method save the parameters in the actuator memory.
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException 
     */
    public Object saveParameters() throws SDORequestException, FcsHardwareException {
        return this.writeSDO("1010", "1", "4", "65766173");
    }
    
    public Object restoreParameters() throws SDORequestException, FcsHardwareException {
        //TODO if is Power Disable
        return this.writeSDO("1011", "1", "4", "64616F6C");
    }
    
    /**
     * Write a value in hexa in the control world (index=6040, subindex=0,size=2)
     * @param value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public void writeControlWord(String value) throws SDORequestException, FcsHardwareException {
        this.writeSDO("6040", "0", "2", value);
    }
    
     /**
     * Read the control world (index=6040, subindex=0,size=2)
     * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    public String readControlWord() throws SDORequestException, 
            ShortResponseToSDORequestException, FcsHardwareException {
        return this.readSDO("6040", "0");
    }
    
    /**
    * Read the status world (index=6041, subindex=0,size=2)
    * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
    */
    public String readStatusWord() throws SDORequestException, 
            ShortResponseToSDORequestException, FcsHardwareException {
        return this.readSDO("6041", "0");
    }
    

    

    
    
    public void quickStop() throws SDORequestException, FcsHardwareException {
        switch(this.mode) {
           case VELOCITY : 
               writeControlWord("B");
               break;
           case TORQUE_PROFILE :
               writeControlWord("B");
               break;
           default:
               assert false;
        }
    }
    
     /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {

        System.out.println(displayListParameters("VELOCITY")); 
        System.out.println(displayListParameters("TORQUE_PROFILE"));
        System.out.println(displayListParameters());
    }
    
}
