/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Map;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenDevice;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This class is to simulate an EPOS Controller.
 *
 * @author virieux
 */
public abstract class SimuEPOSControllerModule extends CanOpenDevice 
        implements EPOSController {

    protected boolean turnedOff;
    private boolean enabled;
    private boolean enabledToPublish;
    protected EposMode mode;
    protected int targetPosition;
    protected int position;
    protected int ssiPosition;
    protected int targetCurrent;
    protected int actualCurrent;

    /* Parameters */
    private final Map<String, Integer> paramsForHoming;
    private final Map<String, Integer> paramsForProfilePosition;
    private final Map<String, Integer> paramsForCurrent;

    private boolean inError;
    private String[] errorHistory;
    private String errorRegister;

    public SimuEPOSControllerModule(String aName, int aTickMillis,
            String nodeID, String serialNB,
            Map<String, Integer> paramsForCurrent,
            Map<String, Integer> paramsForProfilePosition,
            Map<String, Integer> paramsForHoming
    ) {
        super(aName, aTickMillis, nodeID, serialNB);
        this.paramsForHoming = paramsForHoming;
        this.paramsForProfilePosition = paramsForProfilePosition;
        this.paramsForCurrent = paramsForCurrent;
        this.position = 0;
        this.inError = false;
        this.turnedOff = false;
        this.errorRegister = "NO ERROR";
        this.errorHistory = new String[0];
        this.mode = EposMode.PROFILE_POSITION;
    }
    
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "For simulator only : Update position with a position given as argument.")
    public void setPosition(int actualPosition) {
        this.position = actualPosition;
    }

    public void setActualCurrent(int actualCurrent) {
        this.actualCurrent = actualCurrent;
    }

    @Override
    public boolean isEnabledToPublish() {
        return enabledToPublish;
    }

    @Override
    public boolean isTurnedOff() {
        return turnedOff;
    }
    
    

    //For GUI
    @Override
    public Map<String, Integer> getParamsForHoming() {
        return paramsForHoming;
    }

    //For GUI
    @Override
    public Map<String, Integer> getParamsForProfilePosition() {
        return paramsForProfilePosition;
    }
    
    //For GUI
    @Override
    public Map<String, Integer> getParamsForCurrent() {
        return paramsForCurrent;
    }

    @Override
    public boolean isInError() {
        return inError;
    }

    @Override
    public String getErrorRegister() {
        return errorRegister;
    }

    @Override
    public String[] getErrorHistory() {
        if (errorHistory == null) {
            return new String[0];
        } else {
            return errorHistory.clone();
        }
    }

    @Override
    public String getModeInString() {
        return mode.toString();
    }

    /**
     * We assum that simulated hardware's controller has always correct parameters.
     * @return 
     */
    @Override
    public boolean isParametersOK() {
        return true;
    }

    /**
     * This methods is called during INITIALIZATION phase. We want to publish
     * data in order for the controllers GUIs to be refreshed.
     *
     * @return a message
     */
    @Override
    public String initializeAndCheckHardware() throws FcsHardwareException {
        super.initializeAndCheckHardware();
        publishData();
        return name + " is INITIALIZED.";

    }

    /**
     * Changes the mode to the new mode given as a String argument.
     *
     * @param modeInString
     * @return
     * @throws SDORequestException
     */
    public String changeMode(String modeInString) throws SDORequestException {
        EposMode newMode = EposMode.valueOf(modeInString.trim().toUpperCase());
        return changeMode(newMode);
    }

    @Override
    public String changeMode(EposMode newMode) throws SDORequestException {
        this.mode = newMode;
        return getName() + " mode changed to=" + newMode.toString();
    }


    @Override
    public void defineAbsolutePosition(int position) throws SDORequestException,
            ErrorInCommandExecutionException, BadCommandException {
        this.position = position;
    }

    @Override
    public void disable() throws SDORequestException, ShortResponseToSDORequestException {
        this.enabled = true;
        this.enabledToPublish = true;
    }

    @Override
    public void writeParameters(EposMode mode) throws SDORequestException, ErrorInCommandExecutionException {
    }

    @Override
    public String writeParameterInHexa(EPOSEnumerations.Parameter parameter, String string)
            throws SDORequestException {
        return name + " " + parameter.toString() + " set to value in HEXA=" + string;
    }

    @Override
    public void writeTargetPosition(int position)
            throws BadCommandException, SDORequestException, FcsHardwareException {
        //this.targetPosition = position;
        this.position = position;
    }

    @Override
    public void writeParameters(Map<String, Integer> paramMap)
            throws SDORequestException {
    }

    @Override
    public void writeCurrent(int aValue) throws BadCommandException,
            SDORequestException, FcsHardwareException {
        this.turnedOff = false;
        this.actualCurrent = aValue;
    }


    
    @Override
    public String off() throws BadCommandException,
            ErrorInCommandExecutionException, FcsHardwareException {
        this.turnedOff = true;
        return name + " is OFF";
    }






    @Override
    public boolean isTargetPositionReached(int targetPosition) {
        return (this.position == targetPosition);
    }

    @Override
    public int readPosition() throws SDORequestException {
        return this.position;
    }

    /**
     * Read the position returned by the absolute encoder (single serial data).
     *
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Override
    public int readSSIPosition() throws SDORequestException,
            ShortResponseToSDORequestException {
        return this.ssiPosition;
    }

    @Override
    public String enable() throws SDORequestException {
        enabled = true;
        return name + "IS ENABLED";
    }

    @Override
    public int readCurrent() throws BadCommandException, SDORequestException {
        return actualCurrent;
    }

    @Override
    public EposMode readMode() throws SDORequestException {
        return mode;
    }

    @Override
    public int getMaxCurrent() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public int getMinCurrent() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public int getMaxSpeed() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public int getMaxPosition() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public int getMinPosition() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void quickStop() throws SDORequestException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public int readNumberOfErrors() throws SDORequestException, ShortResponseToSDORequestException {
        return 0;
    }

    @Override
    public String displayErrorHistory() throws SDORequestException, ShortResponseToSDORequestException {
        return "No ERROR";
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the Controller is in fault.")
    @Override
    public void checkFault() throws SDORequestException,
            ShortResponseToSDORequestException, FcsHardwareException {
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Clear faults on the controller.")
    public void faultReset() throws SDORequestException {
        writeControlWord("80");
        setChanged();
        this.notifyObservers(new ValueUpdate(name, "faultReset"));
        this.inError = false;
        this.errorRegister = "NO ERROR";
        this.errorHistory = new String[0];
        publishData();
    }

    @Override
    public int readProfileVelocity() throws SDORequestException,
            ShortResponseToSDORequestException {
        //TODO return a value of current in random between min and max
        return 0;
    }

    @Override
    public String readParameters(EposMode mode) throws SDORequestException, ShortResponseToSDORequestException {
        return name + ":no parameter to read.";
    }

    @Override
    public void checkParameters(String modeInString) throws HardwareException, SDORequestException, ShortResponseToSDORequestException {
    }

    @Override
    public void checkParameters(EposMode aMode) throws HardwareException, SDORequestException, ShortResponseToSDORequestException {

    }

    @Override
    public void writeControlWord(String f) throws SDORequestException {
        FCSLOG.debug(name + "writeControlWord :" + f);
    }

    @Override
    public String shutdown() throws SDORequestException {
        FCSLOG.debug(name + " is SHUTDOWN");
        return name + " is SHUTDOWN";
    }

    @Override
    public boolean isEnabled() throws SDORequestException {
        return enabled;
    }

    @Override
    public boolean checkTargetReached() throws SDORequestException, ShortResponseToSDORequestException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void releaseBrake() throws SDORequestException, ShortResponseToSDORequestException {
        FCSLOG.debug(name + " brake is released.");
    }

    @Override
    public void switchOnEnableOperation() throws SDORequestException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void activateBrake() throws SDORequestException, ShortResponseToSDORequestException {
        FCSLOG.debug(name + " brake is activated.");
    }

    public StatusDataPublishedByEPOSController getStatusData() {
        StatusDataPublishedByEPOSController status = FcsUtils.createStatusDataPublishedByEPOSController(this);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            alias = "refreshGUI",
            description = "Publish data for the controller on the status bus.")
    public void publishData() {
        StatusDataPublishedByEPOSController status = this.getStatusData();
        KeyValueData kvd = new KeyValueData(name, status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }

}
