/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.data.Alert;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.errors.CWrapperNotConnected;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.utilities.beanutils.WrappedException;

/**
 * This Modules : - start a tcpip server on a portNumber, - waits for the
 * connection of the client (which is supposed to be a Can OPen C wrapper), -
 * send commands received by the call method to the client and - send back the
 * client response. - it handles the asynchronous messages coming from the
 * client and for that it starts a thread which listen to the client.
 *
 * @author virieux
 */
abstract class FcsTcpProxy extends Module {

    private final int portNumber;
    private ServerSocket serverSock;
    private Thread readerThread;
    protected volatile boolean stopped = true;
    protected volatile boolean stopping = false;
    protected volatile boolean tcpServerStarted = false;
    protected ClientContext clientContext;

    private CommandDispenser commandDispenser;

    protected final Lock lock = new ReentrantLock();
    protected final Condition hardwareBooted = lock.newCondition();

    protected static class ClientContext {

        private String clientName;
        private BufferedReader reader;
        private BufferedWriter writer;
        protected Socket socket;

        ClientContext(String name, Socket socket, BufferedReader reader, OutputStream os) {
            try {
                this.clientName = name;
                this.reader = reader;
                this.socket = socket;
                writer = new BufferedWriter(new OutputStreamWriter(os, "ISO-8859-1"), 256);
            } catch (UnsupportedEncodingException e) {
                /*. SHOULD NOT HAPPEN */
                FCSLOG.error(name + ":context not started", e);
                throw new Error(e);
            }
        }
    }

    public FcsTcpProxy(String aName, int aTickMillis,
            int portNumber) {
        super(aName, aTickMillis);
        this.portNumber = portNumber;
    }

    public boolean isTcpServerStarted() {
        return tcpServerStarted;
    }

    public int getPortNumber() {
        return portNumber;
    }

    @Override
    public void initModule() {
        stopped = false;
        stopping = false;
        tcpServerStarted = false;
        commandDispenser = new CommandDispenser();
    }

    @Override
    public void start() {
        startServer();
        startThreadReader();
    }

    /**
     * Starts the server tcp on the port portNumber. And waits for a client
     * connection.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Starts the tcp server.")
    public void startServer() {
        try {
            serverSock = new ServerSocket(portNumber);
            FCSLOG.info(name + ":SERVER STARTED ON PORT:" + portNumber);
        } catch (IOException e) {
            FCSLOG.error(name + ":server not started", e);
            throw new WrappedException(e);
        }

        try {
            FCSLOG.info(name + ":WAITING FOR C-WRAPPER CLIENT...");
            FCSLOG.debug(name + ":DEBUGGING MODE");
            Socket sock = serverSock.accept();
            FCSLOG.info(name + ":socket accept on " + portNumber);
            BufferedReader reader = new BufferedReader(
                    new InputStreamReader(sock.getInputStream(), "ISO-8859-1"), 256);

            String nameAndProtocol = reader.readLine();
            if (nameAndProtocol == null) {
                FCSLOG.error(name + ": nameAndProtocol is null : that should not happen.");
            } else {
                String[] words = nameAndProtocol.split(" ");
                String cname = words[0];

                clientContext = new ClientContext(cname, sock, reader, sock.getOutputStream());
                FCSLOG.info(name + ":REGISTERED : " + cname);
                tcpServerStarted = true;
                stopped = false;
            }

        } catch (IOException e) {
            // LOG
            FCSLOG.error(name + " unexpected ", e);
        }
    }

    /**
     * Starts a thread which read on the tcp socket, waiting for messages coming
     * from tcp proxy. If the message has a known token (already registred in
     * commandDispenser) it is a response to a can open command that we had sent
     * to the proxy. If the token of the message is null, that means that is's
     * an unsynchronous message coming from the can open stack and transmitted
     * directly from the tcp proxy. That can be : - a boot message :
     * "boot,nodeID" - an emergency message : "emcy,80+NodeID"
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Starts to listen to the tcp client.")
    public void startThreadReader() {
        FCSLOG.info(name + "startThreadReader");
        if (!isTcpServerStarted()) {
            String errorMsg = name + " tcp server has to be started before startThreadReader.";
            FCSLOG.error(errorMsg);
            Alert alert = new Alert("FCS001"+name, errorMsg);
            this.getSubsystem().raiseAlert(alert, ALARM);
        }
        /*We start thread to read from tcpProxy*/
        Runnable readingFromTcpProxy;
        readingFromTcpProxy = new Runnable() {
            @Override
            public void run() {
                while (!stopped) {
                    try {
                        String readline = clientContext.reader.readLine();
                        if (readline == null) {
                            if (stopping) {
                                FCSLOG.info(name + " : CWrapper is stopped.");
                            } else {
                                FCSLOG.error(name + " :nameAndProtocol is null : that should not happen.");
                            }
                        } else {

                            FCSLOG.finest(name + ":message read from socket=" + readline);

                            if (readline.startsWith("emcy")) {
                                processEmcyMessage(readline);

                            } else if (readline.startsWith("Unknown")) {
                                processUnknownCommand(readline);

                            } else {
                                String[] words = readline.split(",");
                                String commandWord = words[0];
                                String nodeID = words[1];
                                if (commandWord.equals("boot")) {
                                    FCSLOG.info(name + ":boot command received for node: " + nodeID);
                                    processBootMessage(nodeID);
                                } else {
                                    String token;
                                    if (readline.startsWith("sync")) {
                                        token = "sync";
                                    } else {
                                        token = commandWord + nodeID;
                                    }
                                    FCSLOG.finest("Corresponding token=" + token);
                                    if (commandDispenser.isTokenUsed(token)) {
                                        FCSLOG.finest("Response to a registred command:" + readline);
                                        try {
                                            commandDispenser.registerResponse(token, readline);
                                        } catch (Exception ex) {
                                            FCSLOG.error(ex);
                                            FCSLOG.error(name + ":Error in registering response to command :" + readline, ex.getCause());
                                        }
                                    } else {
                                        FCSLOG.warning(name + ":Unsynchronous message read from tcpProxy=" + readline);

                                    }
                                }
                            }
                        }

                    } catch (IOException ex) {
                        FCSLOG.error(ex);
                        FCSLOG.error(name + ": Error in Thread reading from the tcp client.");
                    }
                }
            }
        };
        FCSLOG.info(name + ":STARTING Thread reader");
        readerThread = new Thread(readingFromTcpProxy);
        readerThread.start();
    }

    @Override
    public void shutdownNow() {
        super.shutdownNow();
        this.stopServer();
    }

    /**
     * Stops the tcp server.
     */
    public void stopServer() {
        if (!tcpServerStarted) {
            FCSLOG.warning(name + " is stopped; nothing to stop.");
            return;
        }
        FCSLOG.info(name + ": ABOUT TO STOP TCP server.....");
        FCSLOG.finest("clientName=" + clientContext.clientName);
        stopping = true;
        try {
            call(clientContext.clientName, "quit");

        } catch (CanOpenCallTimeoutException ex) {
            String msg = name + ": could not stop Cwrapper.";
            FCSLOG.error(msg + ex);
            Alert alert = new Alert("FCS001:"+name, msg + ex.getMessage());
            this.getSubsystem().raiseAlert(alert, ALARM);
            
        } catch (CWrapperNotConnected ex) {
            String msg = name + ": could not stop because CWrapper is not connected.";
            FCSLOG.error(msg + ex);
            Alert alert = new Alert("FCS001:"+name, msg + ex);
            this.getSubsystem().raiseAlert(alert, ALARM);
        }

        stopped = true;
        readerThread.interrupt();

        try {
            serverSock.close();
            this.tcpServerStarted = false;
            FCSLOG.info(name + ": Subsystem " + this.getSubsystem().getName() + " TCPIP SERVER STOPPED:");
        } catch (IOException ex) {
            FCSLOG.error(ex);
            FCSLOG.error(name + " : stop method does not work properly when closing socket");
        }

        try {
            clientContext.reader.close();
            clientContext.writer.close();
            clientContext.socket.close();

        } catch (IOException ex) {
            FCSLOG.error(ex);
            FCSLOG.error(name + " : stop method does not work properly");
        }

    }

    public synchronized boolean isReady(String clientName) {
        if (clientName == null) {
            throw new IllegalArgumentException("Client name can't be null.");
        }
        if (clientContext == null || clientContext.clientName == null) {
            return false;
        }
        if (stopped) {
            return false;
        }
        if (!tcpServerStarted) {
            return false;
        }
        FCSLOG.debug(name + "client name:= " + clientContext.clientName + "#");
        return clientContext.clientName.equals(clientName);
    }

    /**
     * This methods send a command to the tcp client. The command is a String
     * and can be understood by the client. Example for a Can Open command :
     * rsdo,1,1018,0 The command is stored in the command dispenser with a token
     * and command corelation. When the response comes back from the proxy we
     * can retrieve it from the dispenser with the call token.
     *
     * @param clientName
     * @param command
     * @return result of the call or null if the call failed
     */
    public Object call(String clientName, String command)
            throws CWrapperNotConnected, CanOpenCallTimeoutException {

        FCSLOG.finest(" CALL :" + clientName + " " + command);
        if (clientContext == null) {
            throw new CWrapperNotConnected(this.portNumber, clientName);
        }
        if (stopped) {
            throw new IllegalStateException(name + ":textTCP module stopped");
        }

        try {

            String sentCommand = command + "\r\0\n";
            FCSLOG.finest(name + ":Sent command=" + command);
            clientContext.writer.write(sentCommand, 0, sentCommand.length());
            clientContext.writer.flush();

            //if we want to wait for the response we have to store this command.
            if (command.startsWith("rsdo")
                    || command.startsWith("wsdo")
                    || command.startsWith("info")
                    || command.startsWith("sync")
                    || command.startsWith("srtr")) {
                String commandToken = commandDispenser.register(command);
                try {
                    //TODO put the following timeout in configuration file
                    long timeout = 2 * this.tickMillis;
                    return commandDispenser.getCommandResponse(commandToken, timeout);
                } finally {
                    commandDispenser.remove(commandToken);
                }
            } else {
                return "asynchronous command sent:" + command;
            }

        } catch (IOException ex) {
            FCSLOG.error(name + ":ERROR in tcpProxy call method");
            FCSLOG.error(ex);
            //TODO change this : we shouldn't use a WrappedException
            throw new WrappedException(name + ":ERROR in tcpProxy call method " + clientName, ex);
        }

    }

    /**
     * Process a boot message coming from the can open stack.
     *
     * @param nodeID
     */
    abstract void processBootMessage(String nodeID);

    /**
     * Process an emergency message coming from the can open stack.
     *
     * @param nodeID
     */
    abstract void processEmcyMessage(String message);

    /**
     * Process a response to unknown command.
     *
     * @param nodeID
     */
    abstract void processUnknownCommand(String message);

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("/port=");
        sb.append(this.portNumber);
        return sb.toString();
    }

}
