
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.subsystems.fcs.common.TcpProxyInterface;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import static org.lsst.ccs.bus.states.AlertState.WARNING;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.CWrapperNotConnected;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.PDOBadResponseException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.utils.FcsUtils.MAX_NODE_ID;
import static org.lsst.ccs.subsystems.fcs.utils.FcsUtils.MAX_VALUE_2BYTES;

/**
 * This Module starts a tcpip server, waits for the connection of a client whose
 * name is the value of the field myClientName. 
 * This client is supposed to be a CANopen C-wrapper. When the client starts it scans the CANopen 
 * network and send to the server some information on the CANopen bootedNodes living on the CAN bus.
 * The main job of the class CanOpenProxy is to check that the CANopen devices booted on the CAN bus
 * are the CANopen devices which are listed in the configuration system.
 * This class launch ALERTS :
 * - FCS001 in sendCanOpenCommand, readDeviceInfo
 * - FCS002 in processEmcyMessage, processErrorRegister
 * - FCS003 in processBootMessage
 * 
 * 
 *
 * @author virieux
 */
public class CanOpenProxy extends FcsTcpProxy implements HardwareController, 
        ClearAlertHandler, TcpProxyInterface {


    /*This is the key word that has to be sent by the client when it connects to the tcp proxy. */
    private final String myClientName;


    /**
     * An array of booted hardware.
     * This represents the list of CANopen nodes which are booted on the CANbus.
     */
    private Map<String,CanOpenNode> bootedNodes;

    private boolean hardwareBootProcessEnded = false;
    private boolean hardwareIdentified = false;

    /**
     * A timeout for the booting process : we don't want to wait longer that
     * this elapse of time. This has to be tuned during the test bench. UNIT =
     * milliseconds
     */
    private long hardwareBootTimeout;


    
    /**
     * An array of CANopen hardware controled by this subsystem.
     * This represents the hardware listed in the configuration file which are 
     * children of BridgeToHardware and implement PieceOfHardware.
     * This list of hardware is initialized in the method initModule of the 
     * Module BridgeToCanOpenHardware.
     * protected because used by SimuCanOpenProxy
     */
    protected PieceOfHardware[] hardwareList;

    private boolean canOpenNodeNumbersOK = false;
    private StringBuilder errorMessageSB;
    private boolean hardwareIDError;

    private PDOStorage pdoStorage;
    
    private ScheduledFuture<?> checkDevicesHandle;
    private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(1);
    //Used because we have to wait until all the pieces of hardware are booted.
    protected final Condition bootingCompleted = lock.newCondition();

    /**
     * Creates a CanOpenProxy with a tcpip port number to start the tcp server on,
     * a client name, and a hardware booting process timeout.
     * @param aName
     * @param aTickMillis
     * @param portNumber
     * @param aClientName
     * @param hardwareBootTimeout 
     */
    public CanOpenProxy(String aName, int aTickMillis,
            int portNumber,
            String aClientName,
            long hardwareBootTimeout) {
        super(aName, aTickMillis, portNumber);
        this.myClientName = aClientName;
        this.hardwareBootTimeout = hardwareBootTimeout;
        bootedNodes = new HashMap<>();

    }

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** SETTERS AND GETTERS  *****************************************************
     */
    /**
     * ***********************************************************************************************
     */
    /**
     * @return the expectedNodesNB
     */
    public int getExpectedNodesNB() {
        return hardwareList.length;
    }


    public long getHardwareBootTimeout() {
        return hardwareBootTimeout;
    }

    @ConfigChanger
    public void setHardwareBootTimeout(long hardwareBootTimeout) {
        this.hardwareBootTimeout = hardwareBootTimeout;
    }

    @ConfigChanger
    public void setHardwareBootTimeout(int hardwareBootTimeout) {
        this.hardwareBootTimeout = hardwareBootTimeout;
    }

    public PieceOfHardware[] getHardwareList() {
        return hardwareList.clone();
    }

    protected void setHardwareList(PieceOfHardware[] hardwareList) {
        this.hardwareList = hardwareList.clone();
    }

    /**
     * @return the myClientName
     */
    public String getMyClientName() {
        return myClientName;
    }

    public PDOStorage getPdoStorage() {
        return pdoStorage;
    }

    public boolean isHardwareIdentified() {
        return hardwareIdentified;
    }

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** END OF SETTERS AND GETTERS  **********************************************
     */
    /**
     * ***********************************************************************************************
     */
    /**
     * This methods stops the CWrapper client, and the TCP server.
     */
    @Override
    public void disconnectHardware() {
        //TODO we have to test if all the hardware is stopped before doing stopServer
        this.stopServer();
        bootedNodes = new HashMap<>();
        hardwareBootProcessEnded = false;
        hardwareIdentified = false;
        canOpenNodeNumbersOK = false;
        this.hardwareIDError = false;
        this.pdoStorage = new PDOStorage();
    }

    @Override
    public void connectHardware() throws HardwareException {
        this.startServer();
        this.startThreadReader();
    }

    @Override
    public void initModule() {
        super.initModule();
        hardwareBootProcessEnded = false;
        hardwareIdentified = false;
        canOpenNodeNumbersOK = false;
        this.hardwareIDError = false;
        this.pdoStorage = new PDOStorage();
        /* To be able to clear alerts. */
        getSubsystem().addClearAlertHandler(this);
    }


    
    
    /**
     * This method has to be tested on test benches. It should replace checkHardware.
     * 
     * @return
     * @throws HardwareException 
     */
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        hardwareBootProcessEnded = false;

        FCSLOG.debug(name + ": BEGIN CHECKHARDWARE");
        lock.lock();
        try {
            readDevicesInfo(System.currentTimeMillis(), this.hardwareBootTimeout);
            waitForEndOfBooting();

        } finally {
            lock.unlock();
        }
        
        FCSLOG.debug(name + ": END OF BOOT PROCESS");
        FCSLOG.info(name + "=>" + listNodes());

        /* Here the task of reading devices info is completed because the timeout is reached or
        /*because the number of booted nodes is the number of expected hardware.
        */
        checkCanOpenNodes();
        FCSLOG.debug(name + ": END CHECKHARDWARE");
        //for the GUI
        this.publishData();
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }
    
    /**
     * This method is called in the checkHardware method during the initialization process. 
     * It check devices at fixed rate with a scheduler until all devices are booted 
     * or booting timeout is exceeded.
     * @param beginTime
     * @param timeout 
     */
    public void readDevicesInfo(
            final long beginTime, final long timeout)  {

        final Runnable checkDevices = () -> {
            try {                
                // Retrieve some information from the booted CanOpen bootedNodes
                //(info,node_id)
                identifyHardware();
                this.publishData();
                long duration = System.currentTimeMillis() - beginTime;
                if (duration > timeout || bootedNodes.size() == hardwareList.length) {
                    hardwareBootProcessEnded = true;
                    cancelReadDevicesInfo();
                } 

                
            } catch (HardwareException ex) {
                FCSLOG.error(ex);
                Alert alert = new Alert("FCS001"," couln't read devices info because:" + ex.getMessage());
                this.getSubsystem().raiseAlert(alert, ALARM);
            }
        };
        checkDevicesHandle = scheduler.scheduleAtFixedRate(checkDevices, 
                500, 500, TimeUnit.MILLISECONDS);
    }
    
    /**
     * This stops the reading of the sensors.
     *
     */
    private void cancelReadDevicesInfo() {
        lock.lock();
        try {
            FCSLOG.debug(getName() + " => stop waiting for devices info.");
            bootingCompleted.signalAll();
        } finally {
            lock.unlock();
        }
        this.checkDevicesHandle.cancel(true);
        FCSLOG.debug(getName() + " => readDevicesInfo canceled");
    }
    
    /**
     * This method waits until the booting process is completed. 
     * This methods is called by checkHardwareToBeTested and has already acquired the lock.
     */
    private void waitForEndOfBooting() {
        while (!hardwareBootProcessEnded) {
            try {
                FCSLOG.info(name + " waiting until all pieces of hardware are booted.");
                bootingCompleted.await();
            } catch (InterruptedException ex) {
                FCSLOG.info(name + ": InterruptedException received=" + ex);
                break;
            }
        }
        FCSLOG.info(name + " STOP WAITING FOR END OF BOOTING PROCESS");
    }

    /**
     * checkNewHardware is called by command checkStarted() in higher level
     * Modules and is executed when a command completeInitialization is sent to
     * the subsystem. The command completeInitialization is sent by the end-user
     * to complete the initialization phase when the subsystem was stopped
     * during this phase because of an error (HardwareException).
     *
     * @throws org.lsst.ccs.HardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Retrieve information for booted devices and "
            + "checks if CANopen node ID and serial number match those"
            + "in description file .")
    @Override
    public void checkNewHardware() throws HardwareException {
        FCSLOG.info(name + " BEGIN checkStartedToCompleteInitialization");
        
        if (!(isReady(this.myClientName))) {
            throw new HardwareException(true, name + ": not yet connected with CWrapper.");
        }
        if (!this.isTcpServerStarted()) {
            throw new HardwareException(true, name
                    + ": could not start tcp server.");
        }

        identifyHardware();

        checkCanOpenNodes();

    }

    @Override
    public void checkStarted() throws HardwareException {
        FCSLOG.info(name + " BEGIN checkStarted");
    }

    @Override
    public void checkStopped() throws HardwareException {
        //TODO find something cleaver to be done here or delete this method.
    }
    
    /**
     * This shutdowns the scheduler. This is executed during CLOSING phase when
     * the CanOpenProxy is stopped. cf checkStopped
     */
    @Override
    public void shutdownNow() {
        super.shutdownNow();
        scheduler.shutdown();
    }

    /**
     *
     * @param aNodeID
     * @return
     */
    public String getNodeName(String aNodeID) {
        for (PieceOfHardware hardware : this.hardwareList) {
            if (hardware.getNodeID().equals(aNodeID)) {
                return hardware.getName();
            }
        }
        return "Unbooted nodeID:" + aNodeID;
    }

    /**
     * Check if the String nodeID already exists in the bootedNodes table.
     * If it is booted it is on the bootedNodes table.    
     * It means that this node ID is on the can open bus.
     * @param aNodeID
     * @return 
     */
    private boolean isBooted(String aNodeID) {

        if (bootedNodes.isEmpty()) {
            return false;
        } else {
            return bootedNodes.containsKey(aNodeID);
        }        
    }

    /**
     * List the can open bootedNodes which are in the bootedNodes table.
     *
     * @return the list of can open bootedNodes and the information stored in
     * this.bootedNodes.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, 
            description = "Print the list of CANopen nodes which are booted on "
                    + "the CAN bus.")
    public String listNodes() {

        StringBuilder sb = new StringBuilder("List of booted Nodes = (values are in HEXA)\n");
        if (bootedNodes.isEmpty()) {
            sb.append(name);
            sb.append(": No booted CANopen devices.");
        }
        for (Map.Entry<String,CanOpenNode> entry: bootedNodes.entrySet()) {
            CanOpenNode bootedNode = entry.getValue();
            sb.append(bootedNode);
        }
        return sb.toString();

    }

    /**
     * For end users, in engineering mode, this method can be used to send Can Open commands
     * to the Wrapper.
     * Test if the command is valid before sending it to the CAN bus.
     *
     * @param command A Can Open command that the Wrapper should understand.
     * @return the response from the Wrapper
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, 
            description = "Send a CanOpen command to the Can Bus.")
    public String sendCanOpenCommand(String command) throws FcsHardwareException {
        if (!this.tcpServerStarted) {
            throw new FcsHardwareException(name + ": is not started, can't send CanOpen commands.");
        }       
        checkCanOpenCommand(command, name);
        return sendCanOpen(command);
    }
    
    /**
     * Send a command on the CANbus.
     * @param command
     * @return
     * @throws FcsHardwareException 
     */
    private String sendCanOpen(String command) throws FcsHardwareException {
        try {
            return (String) call(getMyClientName(), command);

        } catch (CWrapperNotConnected ex) {
            FCSLOG.error(name + ":CWrapper not connected-" + ex);
            Alert alert = new Alert("FCS001","Can't communicate with hardware because "
                    + "CWrapper is not connected - check ethernet connections.");
            this.getSubsystem().raiseAlert(alert, ALARM);
            throw new FcsHardwareException("CWrapper not connected.",ex);
            
            
        } catch (CanOpenCallTimeoutException ex) {
            String msg = name + ": timeout expired while waiting to a response from CANbus "
                    + "to command: " + command + " POWER FAILURE ? ";
            FCSLOG.error(msg);
            Alert alert = new Alert("FCS001",msg);
            this.getSubsystem().raiseAlert(alert, ALARM);
            throw new FcsHardwareException(msg,ex);
        }
    }

    /**
     * build a Can Open wsdo Command that can understand the C wrapper. exemple
     * : wsdo,2,6411,01,2,3000
     */
    private static String buildWsdoCommand(String nodeID, String index, String subindex, String size, String data) {
        char sep = ',';
        StringBuilder sb = new StringBuilder("wsdo").append(sep);
        sb.append(nodeID).append(sep);
        sb.append(index).append(sep);
        sb.append(subindex).append(sep);
        sb.append(size).append(sep);
        sb.append(data);
        return sb.toString();
    }

    /**
     * build a Can Open rsdo Command that can understand the C wrapper. exemple
     * : rsdo,1,1018,0
     */
    private static String buildRsdoCommand(String nodeID, String index, String subindex) {
        char sep = ',';
        StringBuilder sb = new StringBuilder("rsdo").append(sep);
        sb.append(nodeID).append(sep);
        sb.append(index).append(sep);
        sb.append(subindex);
        return sb.toString();
    }
    
    /**
     * Tests that a CANOpen command is valid.
     * If the command is not valid it throws a BadCommandException.
     * This is used in the sendCanOpen command method which can be entered by an end user in a console.
     * 
     * @param command CANOpen command to be tested
     * @param tcpProxyName just to be added in the BadCommandException message for end user information.
     * @throws BadCommandException 
     */
    private static void checkCanOpenCommand(String command, String tcpProxyName) {
        String[] words = command.split(",");
        String keyWord = words[0];
        switch (keyWord) {
            case "rsdo":
                if (words.length != 4) {
                    throw new IllegalArgumentException("rsdo command must have 3 parameters: "
                            + "nodeID, index, subindex");
                }
                break;
            case "wsdo":
                if (words.length != 6) {
                    throw new IllegalArgumentException("wsdo command must have 5 parameters: "
                            + "nodeID, index, subindex, size, data");
                }
                break;
            case "info":
                if (words.length != 2) {
                    throw new IllegalArgumentException("info command must have only one "
                            + "parameter: nodeID");
                }
                break;
            case "sync":
                if (words.length != 1) {
                    throw new IllegalArgumentException("sync command takes no "
                            + "parameter");
                }
                break;
            case "srtr":
                if (words.length < 2) {
                    throw new IllegalArgumentException("srtr command takes at "
                            + "least one parameter (cobib)");
                }
                break;
            case "quit":
                if (words.length != 1) {
                    throw new IllegalArgumentException("quit command takes no "
                            + "parameter");
                }
                break;
            default:
                throw new IllegalArgumentException(tcpProxyName + " Invalid CANopen command:" + command);
        }
    }

    /**
     * Write a SDO request and send it to the can open stack, then analyses the
     * response or throws an exception if the request failed. The parameters are
     * given in HEXA.
     *
     * @param nodeID FORMAT=HEXA
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=HEXA
     * @param value FORMAT=HEXA
     * @return "OK" if OK
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public String writeSDO(String nodeID, String index, String subindex, String size, String value) 
            throws FcsHardwareException {
        String request = buildWsdoCommand(nodeID, index, subindex, size, value);
        String sdoResponseLine = sendCanOpen(request);
        /*response of a writeSDO command is : wsdo,nodeID,errorCode*/
        String[] words = sdoResponseLine.split(",");
        
        String errorCode = words[2];
        String msg;
        String deviceName;
        switch (errorCode) {
            case "0":
                return "OK";
            case "-1":
                msg = name + ":wrong writeSDO command : " + sdoResponseLine;
                deviceName = getNodeName(nodeID);
                FCSLOG.error(msg);
                this.getSubsystem().raiseAlert(new Alert(deviceName,msg), ALARM);
                throw new SDORequestException(name + ":wrong writeSDO command : " + sdoResponseLine);
            default:
                String message = processSDORequestError(request,sdoResponseLine,nodeID,errorCode, ALARM);
                throw new SDORequestException(message);
        }
        

    }

    /**
     * Command to be used by the end user at the console. Write a SDO message
     * and send it to the can open stack. The parameters are given in the
     * following format:
     *
     * @param nodeID FORMAT=decimal
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=decimal
     * @param value FORMAT=decimal
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, 
            description = "Send a CanOpen writeSDO command to the Can Bus.")
    public String writeSDO(int nodeID, String index, String subindex, int size, int value) 
            throws FcsHardwareException {
        if (nodeID < 0 || nodeID > MAX_NODE_ID) {
            throw new IllegalArgumentException("nodeID must be > 0 and <=" + MAX_NODE_ID);
        }
        if (size < 0 || size > 4) {
            throw new IllegalArgumentException("size must be > 0 and < 4");
        }
        return writeSDO(Integer.toHexString(nodeID), index, subindex, Integer.toHexString(size), 
                Integer.toHexString(value));
    }

    /**
     * Read a SDO with the given index and subindex RETURNS the value read in
     * hexa (String) if no error occured or returns the error code if a error
     * was detected.
     *
     * @param nodeID
     * @param index
     * @param subindex
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    public String readSDO(String nodeID, String index, String subindex) 
            throws ShortResponseToSDORequestException, 
            FcsHardwareException {
            String request = buildRsdoCommand(nodeID, index, subindex);
            
            String sdoLine = sendCanOpen(buildRsdoCommand(nodeID, index, subindex));
            String[] words = sdoLine.split(",");
            int responseLength = words.length;
            String msg;
            String deviceName;
            /*response of a readSDO command is : rsdo,nodeID,errorCode,data*/
            String errorCode = words[2];

            if ("0".equals(errorCode) && (responseLength > 3)) {
                return words[3];

            } else if ("0".equals(errorCode) && responseLength == 3) {
                msg = name + ":readSDO request received a too short response=" + sdoLine;
                FCSLOG.warning(msg);
                deviceName = getNodeName(nodeID);
                this.getSubsystem().raiseAlert(new Alert(deviceName,msg), WARNING);
                throw new ShortResponseToSDORequestException(msg);

            } else if ("-1".equals(errorCode)) {
                msg = name + ":wrong readSDO command : " + sdoLine;
                FCSLOG.error(msg);
                deviceName = getNodeName(nodeID);
                this.getSubsystem().raiseAlert(new Alert(deviceName,msg), ALARM);
                throw new SDORequestException(msg);

            } else {
                String message = processSDORequestError(request,sdoLine,nodeID,errorCode,ALARM);
                throw new SDORequestException(message);
            }

    }
    
    /**
     * When we receive a response with an error to a SDO request, this methods retreives the errorName
     * in the CANOpen tables, create an error message and raise an Alert.
     * 
     * @param request
     * @param response
     * @param errorCode
     * @param nodeID 
     * @param alertState 
     * @return msg a message to be added to an exception if needed. 
     */
    public String processSDORequestError(String request, String response, 
            String nodeID, String errorCode, AlertState alertState) {
                        String error = String.format("%08d", Integer.parseInt(errorCode));
        String errorName = CanOpenErrorsTable.commErrorCodes.getProperty(error);
        String deviceName = getNodeName(nodeID);
        String msg = name + ": SDO request was going wrong for device: " + deviceName ;
        msg = msg + ",request="+request+",received response=" + response;
        msg = msg + ",errorCode="+errorCode + ",errorName=" + errorName;
        if (alertState.equals(WARNING)) {
            FCSLOG.warning(msg);
        }
        else {
            FCSLOG.error(msg);
        }
        this.getSubsystem().raiseAlert(new Alert(deviceName,msg), alertState);
        return msg;
    }
    

    /**
     * Command to be used by the end user at the console in engineering mode.
     *
     * @param nodeID FORMAT=decimal
     * @param index FORMAT=hexa
     * @param subindex FORMAT=hexa
     * @return the value sent by the can open stack if the read SDO request
     * succeded.
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.messaging.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, 
            description = "Send a CanOpen readSDO command to the Can Bus.")
    public String readSDO(int nodeID, String index, String subindex) 
            throws SDORequestException, BadCommandException, ShortResponseToSDORequestException, 
            FcsHardwareException {
        if (nodeID < 0 || nodeID > MAX_NODE_ID) {
            throw new IllegalArgumentException("nodeID must be > 0 and < 128");
        }
        return readSDO(Integer.toHexString(nodeID), index, subindex);
    }

    /**
     * This method sends a sync command to the can open stack and returns the
     * reply. example of sync reply we can receive :
     * sync,23_1=25a7,23_2=113,23_3=10,23_4=109,23_5=108,23_6=104,23_7=101,23_8=112
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public PDOStorage readPDOs() throws FcsHardwareException {
        try {
            String replyToSync = sendCanOpen("sync");
            FCSLOG.finest(name + ": replyToSync=" + replyToSync);
            this.pdoStorage.updatePDOs(replyToSync);
            return this.pdoStorage;

        } catch (PDOBadResponseException ex) {
            String msg = name + ": error in response to a sync command.";
            this.getSubsystem().raiseAlert(new Alert("FCS001",msg), ALARM);
            FCSLOG.error(ex);
            throw  new FcsHardwareException(msg,ex);

        } 
    }



    /**
     * Identification of the hardware : we want to retrieve the information
     *  stored in the hardware of the booted bootedNodes and update the array of 
     * bootedNodes with this information. 
     * So for each booted node, we send a command :
     * - info,node_ID to the CWrapper and then we update the bootedNodes list 
     * with the information returned in the command result. 
     * The serial number is a main information to retrieve.
     * @throws org.lsst.ccs.HardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Identification of the hardware : we want to retrieve the information \n"
            + "     * stored in the hardware of the booted nodes and update the array of nodes\n"
            + "     * with this information.")
    public void identifyHardware() throws HardwareException {
        FCSLOG.info(name + ":Identification of the hardware");
        for (Map.Entry<String,CanOpenNode> entry: bootedNodes.entrySet()) {
            CanOpenNode bootedNode = entry.getValue();
            String bootedNodeID = entry.getKey();
            if ((bootedNode != null) && (!bootedNode.isIdentified())) {
                updateDeviceInfo(bootedNodeID,bootedNode);
            }
        }
        this.hardwareIdentified = true;
        publishData();
    }
    
    /**
     * 
     * @param bootedNodeID
     * @param bootedNode
     * @throws HardwareException 
     */
    public void updateDeviceInfo(String bootedNodeID, CanOpenNode bootedNode) 
            throws HardwareException {
        try {
                FCSLOG.debug(name + ":Sending to can open command : info,"
                        + bootedNodeID);
                Object result = sendCanOpen("info," + bootedNodeID);
                FCSLOG.debug(name + ":Received on socket command = " + result);
                String infoLine = (String) result;
                String[] words = infoLine.split(",");
                String command = words[0];
                String nodeID = words[1];

                if ("info".equals(command) && bootedNodeID.equals(nodeID)) {
                    FCSLOG.debug(name + ":updating Node Info for node " + nodeID);
                    String type = words[2];
                    String vendor = words[3];
                    String productCode = words[4];
                    String revision = words[5];
                    String serialNB = words[6];
                    bootedNode.setNodeInfo(type, vendor, productCode, revision, serialNB);
                } else {
                    //this should not happen.
                    FCSLOG.error(name + ":ERROR for command = " + command
                            + "NodeID = " + nodeID);
                    //TODO throw an exception
                }

            } catch (FcsHardwareException ex) {
                FCSLOG.error(ex);
                throw new HardwareException(true, ex);
            }
    }
    


    /**
     * This method returns true if : all the hardware items are booted and
     * identified and the hardware have the node ID expected within the
     * configuration and the hardware is initialized.
     *
     * @return a boolean
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all hardware is booted and identified.")
    public boolean isHardwareReady() {
        FCSLOG.info(name + ":hardwareBootProcessEnded=" + hardwareBootProcessEnded);
        FCSLOG.info(name + ":hardwareIdentified=" + hardwareIdentified);
        FCSLOG.info(name + ":canOpenNodeNumbersOK=" + this.canOpenNodeNumbersOK);
        FCSLOG.info(name + ":hardwareIDError=" + this.hardwareIDError);
        return this.hardwareBootProcessEnded && this.hardwareIdentified
                && this.canOpenNodeNumbersOK && !this.hardwareIDError;
    }


    @Override
    void processBootMessage(String nodeID) {

        /*The node is already in the table of bootedNodes*/
        if (isBooted(nodeID)) {
            if (hardwareBootProcessEnded) {
                /*the boot process is ended: something weird has just happened*/
                String msg = " An unsynchronous boot message is arrived for node ID = " + nodeID
                        +" Possible power failure?";
                FCSLOG.error(msg);
                this.getSubsystem().raiseAlert(new Alert("FCS003:"+name+":"+nodeID,msg), WARNING);
            }

            /*the node is not already in the table of bootedNodes : we have to add it*/
        } else {

            //TODO : if the number of bootedNodes we found is greater than the expectedNodesNB, 
            //throw an exception. 
            bootedNodes.put(nodeID, new CanOpenNode(nodeID));
            FCSLOG.info("Node " + nodeID + " added");
            FCSLOG.info("Number of booted devices=" + bootedNodes.size());
            FCSLOG.info("Number of expected devices=" + hardwareList.length);
            if (hardwareBootProcessEnded) {
                /*the boot process is ended: the device has been powered on ?*/
                String msg = name + ": boot message is arrived for node ID:"+ nodeID +" when boot process is completed. +"
                        + "Has the device been powered on ?";
                FCSLOG.warning(msg);
                this.getSubsystem().raiseAlert(new Alert("FCS003:"+name+":"+nodeID,msg), WARNING);
            }
        }
    }

    /**
     * Process an emergency message received from a CANopen device on the status bus :
     * - ALARM or WARNING Alert is raised,
     * - a message is logged,
     * - Observers are notified (CANopen devices are CanOpenProxy Observers).
     * @param message 
     */
    @Override
    void processEmcyMessage(String message) {
        String[] words = message.split(",");
        String nodeID = words[1];
        String deviceErrorCode = words[2];
        String errReg = words[3];
        
        String deviceErrorName = CanOpenErrorsTable.getDeviceErrorNameByCode(deviceErrorCode);
        String errorRegisterName = CanOpenErrorsTable.getErrorRegisterNameByCode(errReg);        
        String deviceName = this.getNodeName(nodeID);
        setChanged();
        EmergencyMessage emcyMsg = new EmergencyMessage(name, nodeID, deviceName, deviceErrorCode,
                deviceErrorName, errReg, errorRegisterName);
        this.notifyObservers(new ValueUpdate(name, emcyMsg));
        FCSLOG.error(name + " received EMERGENCY message="+message+ " for nodeID="+nodeID);

        if ("00".equals(errReg)) {
            FCSLOG.warning(name + " received EMERGENCY message with no error from nodeID=" 
                            + nodeID + ". Is it after a faultReset ?");
        } else {
            FCSLOG.error(emcyMsg);
            Alert alert = new Alert("FCS002"+getName()+nodeID,emcyMsg.toString());
            this.getSubsystem().raiseAlert(alert, ALARM);
        }
    }
    

    @Override
    void processUnknownCommand(String message) {
        String[] words = message.split(",");
        String command = words[1];
        String errorMessage = "Unknown command:" + command;
        FCSLOG.error(errorMessage);

    }

    /**
     * Check if the list of the booted CANopen devices on the CANbus is
     * identical to the hardware list we have in the description.
     *
     * @throws HardwareException
     */
    public void checkCanOpenNodes() throws HardwareException {
        
        FCSLOG.info(name + ":CHECKING HARDWARE CONFIGURATION ");

        if (bootedNodes.isEmpty()) {
            String msg = String.format(" (hardwareBootTimeout=%02d,"
                    + "expectedNodesNB=%02d)", this.hardwareBootTimeout,
                    hardwareList.length);
            FCSLOG.error(name + ":NO HARDWARE DETECTED - POWER FAILURE ?" + msg);
            throw new HardwareException(true, name + ":NO HARDWARE DETECTED - POWER FAILURE ?" + msg);

        } else if (bootedNodes.size() == hardwareList.length) {
            this.canOpenNodeNumbersOK = true;
        }

        errorMessageSB = new StringBuilder(name);
        hardwareIDError = false;

        for (PieceOfHardware hardware : this.hardwareList) {
            FCSLOG.info("ABOUT TO CHECK: " + hardware.getName() + " NODE_ID=" + hardware.getNodeID());
            checkHardwareID(hardware);           
        }

        if (hardwareIDError) {
            throw new HardwareException(true, errorMessageSB.toString());
        }
        publishData();
    }

    /**
     * When the hardware is booted and we have retrieve the information (serial
     * number), we want to check for each piece of hardware that we have the
     * good node ID and the good serial number: so we compare the serial number
     * found on the CANopen bus (in Map bootedNodes) with the serial number in the configuration
     * file.
     *
     * @param pieceOfHardware
     * @throws org.lsst.ccs.HardwareException
     */
    public void checkHardwareID(PieceOfHardware pieceOfHardware)
            throws HardwareException {
        String msg;

        if (bootedNodes.containsKey(pieceOfHardware.getNodeID())) {
            //the node id of this piece of hardware has been found in the booted bootedNodes list.
            CanOpenNode bootedNode = bootedNodes.get(pieceOfHardware.getNodeID());
            
            if (bootedNode.getSerialNB().equals(pieceOfHardware.getSerialNB())) {
                //and it's the same serial ID : every thing is OK  
                pieceOfHardware.setBooted(true);
                this.publishHardwareData(pieceOfHardware);
                FCSLOG.info(pieceOfHardware.getName() + " BOOTED AND SERIAL NUMBER OK:" 
                        + pieceOfHardware.toString());
            } else {
                //bad serial number for this pieceOfHardware
                pieceOfHardware.setBooted(false);
                msg = pieceOfHardware.getName() + " has a wrong serial number. Serial number found="
                        + bootedNode.getSerialNB() + ", should be=" + pieceOfHardware.getSerialNB();
                                    FCSLOG.error(msg);
                errorMessageSB.append(msg);
                this.hardwareIDError = true;
                //TODO here a HardwareException should be thrown - to be tested.
                //throw new HardwareException(false, name + msg);
            }

        } else {
            if (getBootedNodeBySerialNumber(pieceOfHardware.getSerialNB()) == null) {
                msg = String.format(":HARDWARE NOT DETECTED - "
                            + "Possible power failure for node ID %s ? - "
                            + "(hardware:%s,serial number:%s)",
                            pieceOfHardware.getNodeID(), pieceOfHardware.getName(), 
                            pieceOfHardware.getSerialNB());
                FCSLOG.error(msg);
                errorMessageSB.append(msg);
                this.hardwareIDError = true;
                //TODO here a HardwareException should be thrown - to be tested.
                //throw new HardwareException(false, name + msg);
            } else {
                //the serial number of this piece of hardware has been found in the bootedNodes list
                //with another node ID.
                pieceOfHardware.setBooted(false);
                msg = pieceOfHardware.getName() + " has a wrong serial number. Serial number found="
                        + getBootedNodeBySerialNumber(pieceOfHardware.getSerialNB()).getSerialNB() 
                        + ", should be=" + pieceOfHardware.getSerialNB();
                FCSLOG.error(msg);
                errorMessageSB.append(msg);
                this.hardwareIDError = true;
                //TODO here a HardwareException should be thrown - to be tested.
                //throw new HardwareException(false, name + msg);
            }
        }
    }
    
    /**
     * Find a node by its serial number in the list of booted bootedNodes.
     * Returns null if this serial number is not on the CAN bus.
     * @param sn
     * @return CANopen node which serial number is sn or null.
     */
    public CanOpenNode getBootedNodeBySerialNumber(String sn) {
        for (Map.Entry<String,CanOpenNode> entry: bootedNodes.entrySet()) {
            CanOpenNode bootedNode = entry.getValue();
            if (bootedNode.getSerialNB().equalsIgnoreCase(sn)) {
                return bootedNode;
            }
        }
        return null;
    }
    
    /**
     * Returns a String representation of a CANopen node booted on the CANbus 
     * for the device which serial number is given as a parameter.
     * This is for the end users.
     * @param sn
     * @return a String
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Returns a String representation of a CANopen node booted"
            + " on the CANbus for the device"
            + " which serial number is given as a parameter.")
    public String printBootedNodeBySerialNumber(String sn) {   
        if (sn == null ) {
            throw new IllegalArgumentException(name + " Serial Number must be not null.");
        } else {
            return String.valueOf(getBootedNodeBySerialNumber(sn).toString());
        }
    } 


    /**
     * Return a String with the list of hardware expected in this subsystem.
     * For debug purpose.
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return a String with the list of hardware.")
    @Override
    public String listHardware() {
        StringBuilder sb = new StringBuilder("List of hardware:\n");
        for (PieceOfHardware pieceOfHardware : this.hardwareList) {
            sb.append(pieceOfHardware).append('\n');
        }
        return sb.toString();
    }
    
    /**
     * Return an Array containing hardware names handled by this component.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return an Array containing hardware names handled by this component.")
    @Override
    public List<String> listHardwareNames() {
        List<String> res = new ArrayList<>();
        for (PieceOfHardware pieceOfHardware : this.hardwareList) {
            res.add(pieceOfHardware.getName());
        }
        return res;
    }
    
    /**
     * Overridden method from ClearAlertHandler interface to define conditions when Alerts can be cleared.
     * 
     * @param alert
     * @return 
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert) {
        switch (alert.getAlertId()) {
            case "FCS001":
                if (this.clientContext.socket.isConnected() 
                        && !this.clientContext.socket.isClosed()) {
                    return ClearAlertCode.CLEAR_ALERT;
                } else {
                    return ClearAlertCode.DONT_CLEAR_ALERT;
                }
            
            case "FCS002":
                return ClearAlertCode.CLEAR_ALERT;
                
            default:
                return ClearAlertCode.UNKWNOWN_ALERT;
        }
    }
    
    

    /**
     * Configure a node as a hearbeat producer. Every heartbeatTime
 milliseconds this node will produce a message like: 0x700+nodeID: 05 -
 1340268404.980000
     *
     * @param nodeID node ID in hexa
     * @param heartbeatTime FORMAT=hexa UNIT=milliseconds
     * @return error code returned by writeSDO
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws java.util.concurrent.TimeoutException
     * @throws org.lsst.ccs.messaging.BadCommandException
     */
    public String configAsHeartbeatProducer(String nodeID, String heartbeatTime) 
            throws TimeoutException, BadCommandException, 
            FcsHardwareException {
        return writeSDO(nodeID, "1017", "0", "2", heartbeatTime);
    }

    /**
     * Command to be used by the end users. Configure a node as a heartbeat
     * producer. Parameters given in decimal format.
     *
     * @param nodeID FORMAT=decimal
     * @param heartbeatTime FORMAT=decimal UNIT=milliseconds
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws java.util.concurrent.TimeoutException
     * @throws org.lsst.ccs.messaging.BadCommandException
     */
    public String configAsHeartbeatProducer(int nodeID, int heartbeatTime) 
            throws TimeoutException, BadCommandException, 
            FcsHardwareException {

        if (heartbeatTime < 0 || heartbeatTime > MAX_VALUE_2BYTES) {
            throw new IllegalArgumentException("heartbeat time is coded on 2 bytes"
                    + " can't be > " + MAX_VALUE_2BYTES);
        }
        return configAsHeartbeatProducer(Integer.toHexString(nodeID), Integer.toHexString(heartbeatTime));
    }

    /**
     * Configure a node as a hearbeat consumer. Every heartbeatTime
 milliseconds this node will produce a message like: 0x700+nodeID: 05 -
 1340268404.980000
     *
     * @param nodeID node ID in decimal
     * @param producerNodeID
     * @param heartbeatTime FORMAT=decimal UNIT=milliseconds
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws java.util.concurrent.TimeoutException
     * @throws org.lsst.ccs.messaging.BadCommandException
     */
    public String configAsHeartbeatConsumer(int nodeID, int producerNodeID, int heartbeatTime) 
            throws TimeoutException, BadCommandException, FcsHardwareException {
        if (nodeID < 0 || nodeID > MAX_NODE_ID) {
            throw new IllegalArgumentException("nodeID must be > 0 and <=" + MAX_NODE_ID);
        }
        if (producerNodeID < 0 || producerNodeID > MAX_NODE_ID) {
            throw new IllegalArgumentException("producerNodeID must be > 0 and <="+ MAX_NODE_ID);
        }
        if (heartbeatTime < 0 || heartbeatTime > MAX_VALUE_2BYTES) {
            throw new IllegalArgumentException("heartbeat time is coded on 2 bytes"
                    + " can't be > " + MAX_VALUE_2BYTES);
        }
        String value = String.format("%04x", producerNodeID) + String.format("%04x", heartbeatTime);
        return writeSDO(Integer.toHexString(nodeID), "1016", "1", "4", value);
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        for (PieceOfHardware device : hardwareList) {
            publishHardwareData(device);
        }
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     * @param device
     */
    public void publishHardwareData(PieceOfHardware device) {
        StatusDataPublishedByHardware status
                = FcsUtils.createStatusDataPublishedByHardware(device);
        FCSLOG.debug(name + ":publishHardwareData is publishing:" + status);
        KeyValueData kvd = new KeyValueData("tcpProxy", status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }


    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append("/myClientName=");
        sb.append(this.myClientName);
        return sb.toString();
    }

}
